package com.hyprmx.android.sdk.core

import android.content.Context
import com.hyprmx.android.sdk.core.HyprMXController.Companion.HYPRMX_PREFS_INTERNAL
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.HyprMXProperties
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.File
import java.io.FileInputStream
import java.io.FileNotFoundException
import java.io.IOException

interface StorageHelperIf {
  suspend fun writeToCoreJSFile(content: String): Boolean
  suspend fun writeToFile(path: String, content: String): Boolean
  suspend fun isCoreJSFileExist(): Boolean
  suspend fun isFileExist(path: String): Boolean

  /**
   * Write content of sdk_core.js in android assets folder to file in internal storage
   * if the file does not exist.
   */
  @Throws(FileNotFoundException::class)
  suspend fun getCoreJSFromFile(): String

  suspend fun readCoreJSFile(): String?
  suspend fun readFile(path: String): String?
  suspend fun deleteCoreJSFile(): Boolean
  suspend fun getCoreJSFilePath(): String

  /**
   * Deletes shared js stored to disk if SDK Version is updated.
   * This function blocks current thread until completion because it needs to delete the previous
   * SDK upgraded shared js before loading the packaged in shared js to the WebView.
   */
  suspend fun deleteSharedJSIfSdkVersionUpdated()
}

/**
 * Helper class to write, read and delete shared javascript file from disk.
 */
internal class StorageHelper(
  private val applicationContext: Context,
  private val ioDispatcher: CoroutineDispatcher = Dispatchers.IO,
) : StorageHelperIf {

  companion object {
    const val PREF_SDK_BUILD_VERSION = "sdk_build_version"
    const val HYPRMX_CORE_JS_FILENAME = "hyprMX_sdk_core.js"
  }

  private lateinit var coreJSFilePath: String
  override suspend fun getCoreJSFilePath(): String = withContext(ioDispatcher) {
    return@withContext if (!::coreJSFilePath.isInitialized) {
      val hyprmxDirectoryPath = "${applicationContext.filesDir.absolutePath}/hyprMX_js"
      val directory = File(hyprmxDirectoryPath)
      if (!directory.exists()) {
        directory.mkdirs()
      }
      coreJSFilePath = "$hyprmxDirectoryPath/$HYPRMX_CORE_JS_FILENAME"
      coreJSFilePath
    } else {
      coreJSFilePath
    }
  }

  override suspend fun writeToCoreJSFile(content: String): Boolean = withContext(ioDispatcher) {
    HyprMXLog.d("writetoCoreJSFile")
    writeToFile(getCoreJSFilePath(), content)
  }

  override suspend fun writeToFile(path: String, content: String): Boolean =
    withContext(ioDispatcher) {
      try {
        File(path).bufferedWriter().use { out -> out.write(content) }
      } catch (e: IOException) {
        HyprMXLog.e("Failed writing to file.")
        return@withContext false
      }
      return@withContext true
    }

  override suspend fun isCoreJSFileExist(): Boolean {
    return isFileExist(getCoreJSFilePath())
  }

  override suspend fun isFileExist(path: String): Boolean {
    return File(path).exists()
  }

  /**
   * Write content of sdk_core.js in android assets folder to file in internal storage
   * if the file does not exist.
   */
  @Throws(FileNotFoundException::class)
  override suspend fun getCoreJSFromFile(): String = withContext(ioDispatcher) {
    val jsFile = "sdk_core.min.js"

    return@withContext if (!isCoreJSFileExist()) {
      applicationContext.assets.open(jsFile).bufferedReader().use { it.readText() }
    } else {
      readCoreJSFile()
        ?: applicationContext.assets.open(jsFile).bufferedReader().use { it.readText() }
    }
  }

  override suspend fun readCoreJSFile(): String? = withContext(ioDispatcher) {
    readFile(getCoreJSFilePath())
  }

  override suspend fun readFile(path: String): String? {
    val file = File(path)
    return try {
      FileInputStream(file).bufferedReader().use { it.readText() }
    } catch (e: FileNotFoundException) {
      HyprMXLog.e("Error reading file for path $path", e)
      null
    }
  }

  override suspend fun deleteCoreJSFile(): Boolean = withContext(ioDispatcher) {
    val file = File(getCoreJSFilePath())
    try {
      file.delete()
    } catch (e: SecurityException) {
      HyprMXLog.e("Failed to delete core JS file", e)
      false
    }
  }

  /**
   * Deletes shared js stored to disk if SDK Version is updated.
   * This function blocks current thread until completion because it needs to delete the previous
   * SDK upgraded shared js before loading the packaged in shared js to the WebView.
   */
  override suspend fun deleteSharedJSIfSdkVersionUpdated() = withContext(ioDispatcher) {
    val sharedPreferences =
      applicationContext.getSharedPreferences(HYPRMX_PREFS_INTERNAL, Context.MODE_PRIVATE)
    val sdkBuildVersion = sharedPreferences.getInt(PREF_SDK_BUILD_VERSION, 0)

    if (sdkBuildVersion != HyprMXProperties.buildNumber) {
      deleteCoreJSFile()
      sharedPreferences.edit()
        .putInt(PREF_SDK_BUILD_VERSION, HyprMXProperties.buildNumber)
        .apply()
    }
  }
}
