package com.hyprmx.android.sdk.core.js

import com.hyprmx.android.sdk.analytics.ClientErrorControllerIf
import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.DefaultContinuationController
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.koushikdutta.quack.QuackContext

internal class QuickJSEngine : JSEngine {
  private var quack: QuackContext? = try {
    QuackContext.create()
  } catch (error: Error) {
    HyprMXLog.e("Error creating context: $error")
    null
  }

  override var errorController: ClientErrorControllerIf? = null
  override val uncaughtErrorListeners: MutableList<UncaughtErrorListener> = ArrayList()
  override val hasFailures: Boolean = false

  private val continuationController = DefaultContinuationController(this)

  override fun insertGlobal(constant: String, constructor: String) {
    try {
      debugLog("insertGlobal $constant")
      val tryCatchScript =
        """
          var constructorResponse = undefined;
          var errorResponse = undefined;
          try {
            constructorResponse = $constructor;
          } catch (error) {
            errorResponse = "$ERROR_TAG " + error;
          }
          const $constant = constructorResponse;
          errorResponse;
        """.trimIndent()
      val response: Any? = quack?.evaluate(tryCatchScript)
      if (response?.toString()?.contains(ERROR_TAG) == true) {
        throw Exception(response.toString())
      }
    } catch (exception: Exception) {
      HyprMXLog.e("insertGlobal $constant failed with exception $exception", exception)
      uncaughtErrorListeners.forEach {
        it.onUncaughtError(
          exception.localizedMessage ?: exception.stackTrace.toString(),
        )
      }
    }
  }

  override fun evaluate(script: String) {
    try {
      debugLog("evaluate $script")
      val tryCatchScript =
        """
          var response = undefined;
          try {
            $script
          } catch (error) {
            response = "$ERROR_TAG " + error;
          }
          response;
        """.trimIndent()
      val response: Any? = quack?.evaluate(tryCatchScript)
      if (response?.toString()?.contains(ERROR_TAG) == true) {
        throw Exception(response.toString())
      }
    } catch (exception: Exception) {
      HyprMXLog.e("Evaluate $script failed with exception $exception", exception)
      uncaughtErrorListeners.forEach {
        it.onUncaughtError(
          exception.localizedMessage ?: exception.stackTrace.toString(),
        )
      }
    }
  }

  override fun evaluateScriptForResponse(script: String): Any? {
    return try {
      debugLog("evaluateScriptForResponse $script")
      val tryCatchScript =
        """
          var response = undefined;
          try {
            response = $script
          } catch (error) {
            response = "$ERROR_TAG " + error;
          }
          response;
        """.trimIndent()
      val response: Any? = quack?.evaluate(tryCatchScript)
      if (response?.toString()?.contains(ERROR_TAG) == true) {
        throw Exception(response.toString())
      }
      response
    } catch (exception: Exception) {
      HyprMXLog.e("evaluateScriptForResponse $script failed with exception $exception", exception)
      uncaughtErrorListeners.forEach {
        it.onUncaughtError(
          exception.localizedMessage ?: exception.stackTrace.toString(),
        )
      }
      null
    }
  }

  override fun addJavascriptInterface(obj: Any, name: String) {
    quack?.globalObject?.set(name, obj)
  }

  override fun removeJavascriptInterface(name: String) {
    quack?.globalObject?.set(name, null)
  }

  override fun loadSharedJS(script: String): Boolean {
    if (quack == null) {
      HyprMXLog.e("There was an error creating the JS Interpreter")
      return false
    }

    return try {
      quack?.evaluate(script)
      true
    } catch (exception: Exception) {
      HyprMXLog.e("Error loading shared code", exception)
      uncaughtErrorListeners.forEach {
        it.onUncaughtError(
          exception.localizedMessage ?: exception.stackTrace.toString(),
        )
      }
      false
    }
  }

  override fun addUncaughtErrorListener(listener: UncaughtErrorListener) {
    uncaughtErrorListeners.add(listener)
  }

  override fun removeUncaughtErrorListener(listener: UncaughtErrorListener) {
    uncaughtErrorListeners.remove(listener)
  }

  override fun close() {
    quack?.close()
  }

  override suspend fun asyncEvaluateScriptForResponse(
    method: String,
    caller: String,
    args: String,
  ): Any? {
    return continuationController.asyncEvaluateScriptForResponse(method, caller, args)
  }

  companion object {
    const val ERROR_TAG = "HyprEvalError"
    private var scriptDebugging = false

    @RetainMethodSignature
    fun enableScriptDebugging(debuggable: Boolean) {
      scriptDebugging = debuggable
    }

    fun debugLog(message: String) {
      if (scriptDebugging) {
        HyprMXLog.d(message)
      }
    }
  }
}
