package com.hyprmx.android.sdk.footer

import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.ImageCacheManagerIf
import com.hyprmx.android.sdk.utility.Result
import com.hyprmx.android.sdk.utility.parseColor
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlin.coroutines.CoroutineContext

/**
 * Presenter for the footer.  Responsible for for handling the view setup and interaction
 * between the view and URL/Nav presenters.
 */
internal class FooterPresenter(
  private val urlPresenter: FooterContract.URLPresenter?,
  private val navigationPresenter: FooterContract.NavigationPresenter?,
  footerUIModel: Footer,
  private val footerView: FooterContract.View,
  enableNav: Boolean,
  private val imageCacheManager: ImageCacheManagerIf,
) : FooterContract.Presenter, CoroutineScope {

  private val job = Job()

  override val coroutineContext: CoroutineContext
    get() = job + Dispatchers.Main

  init {
    footerView.presenter = this

    // Hide the footer if its only for navigation
    // Hide the footer if the text is empty and there are no icons
    footerView.setVisible(
      (
        !footerUIModel.navOnly &&
          !footerUIModel.text?.trim().isNullOrEmpty()
        ) ||
        footerUIModel.icon1 != null,
    )

    footerView.setMinimumHeight(footerUIModel.minHeight)
    footerView.enableNavigation(enableNav)

    footerUIModel.text?.let {
      if (it.trim().isNotEmpty()) {
        footerView.setText(it)
      }
    }
    footerUIModel.bgColor?.let { footerView.setBackgroundColor(it.parseColor()) }

    footerUIModel.icon1?.let { image ->
      if (image.width > 0 && image.height > 0) {
        launch {
          val bitmapResult = imageCacheManager.retrieveBitmap(image.portraitUrl)
          if (bitmapResult is Result.Success) {
            footerView.setIcon1(bitmapResult.value, image.width, image.height)
          } else {
            HyprMXLog.e("Unable to load footer icon1.")
          }
        }
      }
    }

    footerUIModel.icon2?.let { image ->
      if (image.width > 0 && image.height > 0) {
        launch {
          val bitmapResult = imageCacheManager.retrieveBitmap(image.portraitUrl)

          if (footerView.isContextInvalid()) {
            return@launch
          }

          if (bitmapResult is Result.Success) {
            footerView.setIcon2(bitmapResult.value, image.width, image.height)
          } else {
            HyprMXLog.e("Unable to load footer icon2.")
          }
        }
      }
    }
  }

  override fun didTapBack() {
    navigationPresenter?.didTapBack()
  }

  override fun didTapForward() {
    navigationPresenter?.didTapForward()
  }

  override fun didTapURL(url: String) {
    urlPresenter?.didTapURL(url)
  }

  override fun didTapIcon(index: Int) {
    navigationPresenter?.didTapIcon(index)
  }

  override fun enableForwardNavigation(enable: Boolean) {
    if (enable) {
      footerView.setVisible(true)
    }
    footerView.enableForwardNavigation(enable)
  }

  override fun enableBackwardNavigation(enable: Boolean) {
    if (enable) {
      footerView.setVisible(true)
    }
    footerView.enableBackNavigation(enable)
  }

  override fun setVisible(visible: Boolean) {
    footerView.setVisible(visible)
  }
}
