package com.hyprmx.android.sdk.header

import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.graphics.drawable.Drawable
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.ProgressBar
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.annotation.Keep
import androidx.core.content.res.ResourcesCompat
import androidx.fragment.app.Fragment
import com.hyprmx.android.R
import com.hyprmx.android.sdk.utility.Utils.getSpannedText
import com.hyprmx.android.sdk.utility.convertDpToPixel

/**
 * The implementation of the header view.
 *
 * This view is designed to show the following:
 * // Page count
 * // Close button -- title -- Next/Finish/Spinner
 *
 * All components are optional and if none are available defaults to a 0 height.
 *
 * Adding components converts their visibility to VISIBLE instead of GONE.
 *
 */
@Keep
internal class WebTrafficHeaderFragment : Fragment(), WebTrafficHeaderContract.View {

  override lateinit var presenter: WebTrafficHeaderContract.Presenter

  lateinit var header: View
  private lateinit var titleTextView: TextView
  private lateinit var closeButton: ImageView
  private lateinit var headerContainer: LinearLayout
  private val pageCountLines = ArrayList<View>()
  private lateinit var nextButtonText: TextView
  private lateinit var nextButtonChevron: ImageView
  private lateinit var nextButton: RelativeLayout
  private lateinit var finishButtonText: TextView
  private lateinit var finishButtonChevron: ImageView
  private lateinit var finishButton: RelativeLayout
  private lateinit var countDownText: TextView
  private lateinit var spinner: ProgressBar
  private lateinit var pageCountIndicator: LinearLayout

  override fun onCreateView(
    inflater: LayoutInflater,
    container: ViewGroup?,
    savedInstanceState: Bundle?,
  ): View {
    header = inflater.inflate(R.layout.hyprmx_header, container, false)
    with(header) {
      headerContainer = findViewById(R.id.hyprmx_header)
      titleTextView = findViewById(R.id.hyprmx_title)
      closeButton = findViewById(R.id.hyprmx_close_button)
      nextButton = findViewById(R.id.hyprmx_next_container)
      nextButtonText = findViewById(R.id.hyprmx_next)
      nextButtonChevron = findViewById(R.id.hyprmx_next_chevron)
      finishButtonText = findViewById(R.id.hyprmx_finish)
      finishButtonChevron = findViewById(R.id.hyprmx_finish_chevron)
      finishButton = findViewById(R.id.hyprmx_finish_container)
      countDownText = findViewById(R.id.hyprmx_countdown)

      finishButton.setOnClickListener { presenter.didTapFinish() }
      closeButton.setOnClickListener { presenter.didTapClose() }
      nextButton.setOnClickListener { presenter.didTapNext() }

      spinner = findViewById(R.id.hyprmx_progress_spinner)
      pageCountIndicator = header.findViewById(R.id.hyprmx_page_count)
    }
    return header
  }

  override fun setTitleText(text: String) {
    titleTextView.text = getSpannedText(text)
  }

  override fun setMinHeight(minHeight: Int) {
    headerContainer.minimumHeight = minHeight.convertDpToPixel(context)
  }

  override fun setBackgroundColor(color: Int) {
    headerContainer.setBackgroundColor(color)
  }

  override fun setPageCount(count: Int, color: Int) {
    pageCountIndicator.removeAllViews()
    pageCountLines.clear()
    pageCountIndicator.visibility = View.VISIBLE

    repeat(count) {
      activity?.let { activity ->
        val view =
          activity.layoutInflater.inflate(R.layout.hyprmx_page_count_indicator, pageCountIndicator)
        val pageCountLine: View = view.findViewById(R.id.page_count_line)
        pageCountLine.setBackgroundColor(color)
        pageCountLine.id = it
        pageCountLines.add(pageCountLine)
      }
    }
  }

  override fun setPageCountState(index: Int, color: Int) {
    // [PLAYER-22267] Prevent out of bounds access crash seen in console
    if (index < pageCountLines.size) {
      pageCountLines[index].setBackgroundColor(color)
    }
  }

  override fun hidePageCount() {
    pageCountIndicator.visibility = View.INVISIBLE
  }

  override fun showNextButton(
    text: String,
    color: Int,
    chevronColor: Int,
    minHeight: Int,
    minWidth: Int,
  ) {
    nextButton.visibility = View.VISIBLE
    nextButtonText.text = getSpannedText(text)
    nextButton.background.mutate().colorFilter =
      PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN)
    nextButton.minimumHeight = minHeight.convertDpToPixel(context)
    nextButton.minimumWidth = minWidth.convertDpToPixel(context)

    nextButtonChevron.setColorFilter(chevronColor, PorterDuff.Mode.SRC_IN)
  }

  override fun hideNextButton() {
    nextButton.visibility = View.GONE
  }

  override fun setCountDown(countDown: String) {
    countDownText.visibility = View.VISIBLE
    countDownText.text = getSpannedText(countDown)
  }

  override fun hideCountDown() {
    countDownText.visibility = View.GONE
  }

  override fun showFinishButton(
    text: String,
    color: Int,
    chevronColor: Int,
    minHeight: Int,
    minWidth: Int,
  ) {
    finishButton.visibility = View.VISIBLE
    finishButtonText.text = getSpannedText(text)
    finishButton.background.mutate().colorFilter = PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN)
    finishButton.minimumWidth = minWidth.convertDpToPixel(context)
    finishButton.minimumHeight = minHeight.convertDpToPixel(context)
    finishButtonChevron.setColorFilter(chevronColor, PorterDuff.Mode.SRC_IN)
  }

  override fun hideFinishButton() {
    finishButton.visibility = View.GONE
  }

  override fun showCloseButton(color: Int) {
    activity?.let { activity ->
      val drawable: Drawable? = ResourcesCompat.getDrawable(activity.resources, R.drawable.hyprmx_close_button, null)

      drawable?.mutate()?.colorFilter = PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN)

      closeButton.setImageDrawable(drawable)
      closeButton.visibility = View.VISIBLE
    }
  }

  override fun hideCloseButton() {
    closeButton.visibility = View.GONE
  }

  override fun showProgressSpinner(color: Int) {
    spinner.indeterminateDrawable.colorFilter = PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN)
    spinner.visibility = View.VISIBLE
  }

  override fun showProgressSpinner() {
    spinner.visibility = View.VISIBLE
  }

  override fun hideProgressSpinner() {
    spinner.visibility = View.GONE
  }

  override fun enableCloseButton() {
    closeButton.isEnabled = true
  }

  override fun disableCloseButton() {
    closeButton.isEnabled = false
  }
}
