package com.hyprmx.android.sdk.header

import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.parseColor
import java.util.UnknownFormatConversionException

/**
 * Presenter for the header.  Responsible for for handling the view setup and interaction
 * between the view and model
 */
internal class WebTrafficHeaderPresenter(
  private val headerUIModel: WebTrafficHeader,
  private val webTrafficHeaderView: WebTrafficHeaderContract.View,
  private val closeable: Boolean,
  private val navigationPresenter: WebTrafficHeaderContract.NavigationPresenter,
) : WebTrafficHeaderContract.Presenter {

  init {

    webTrafficHeaderView.presenter = this

    with(webTrafficHeaderView) {
      if (closeable) {
        showCloseButton(headerUIModel.closeButtonColor.parseColor())
      }
      setBackgroundColor(headerUIModel.bgColor.parseColor())
      setMinHeight(headerUIModel.minimumHeaderHeight)
    }
  }

  override fun didTapNext() {
    navigationPresenter.didTapNext()
  }

  override fun didTapClose() {
    navigationPresenter.didTapClose()
  }

  override fun didTapFinish() {
    navigationPresenter.didTapFinish()
  }

  override fun showNextButton() {
    webTrafficHeaderView.hideCountDown()
    webTrafficHeaderView.hideFinishButton()
    webTrafficHeaderView.hideProgressSpinner()
    webTrafficHeaderView.showNextButton(
      headerUIModel.nextButtonText,
      headerUIModel.nextButtonColor.parseColor(),
      headerUIModel.chevronColor.parseColor(),
      headerUIModel.nextButtonMinHeight,
      headerUIModel.nextButtonMinWidth,
    )
  }

  override fun showFinishButton() {
    webTrafficHeaderView.hideCloseButton()
    webTrafficHeaderView.hideCountDown()
    webTrafficHeaderView.hideNextButton()
    webTrafficHeaderView.hideProgressSpinner()
    webTrafficHeaderView.showFinishButton(
      headerUIModel.finishButtonText,
      headerUIModel.finishButtonColor.parseColor(),
      headerUIModel.chevronColor.parseColor(),
      headerUIModel.finishButtonMinHeight,
      headerUIModel.finishButtonMinWidth,
    )
  }

  override fun hideFinishButton() {
    webTrafficHeaderView.hideCountDown()
    webTrafficHeaderView.hideNextButton()
    webTrafficHeaderView.hideProgressSpinner()
    webTrafficHeaderView.hideFinishButton()
  }

  override fun showWithTitleAndPageCount(pageCount: Int) {
    webTrafficHeaderView.setPageCount(pageCount, headerUIModel.pageIndicatorColor.parseColor())
    webTrafficHeaderView.setTitleText(headerUIModel.titleText)
  }

  override fun setPageVisited(pageNumber: Int) {
    webTrafficHeaderView.setPageCountState(
      pageNumber,
      headerUIModel.pageIndicatorSelectedColor.parseColor(),
    )
  }

  override fun showCountDown(time: String) {
    webTrafficHeaderView.hideFinishButton()
    webTrafficHeaderView.hideNextButton()
    webTrafficHeaderView.hideProgressSpinner()
    val countdown = try {
      headerUIModel.countDownText.format(time)
    } catch (exception: UnknownFormatConversionException) {
      HyprMXLog.e("UnknownFormatConversionException formatting time.  Using default time format.")
      time
    }

    webTrafficHeaderView.setCountDown(countdown)
  }

  override fun showProgressSpinner() {
    webTrafficHeaderView.hideCountDown()
    webTrafficHeaderView.hideFinishButton()
    webTrafficHeaderView.hideNextButton()

    if (headerUIModel.spinnerColor == null) {
      webTrafficHeaderView.showProgressSpinner()
    } else {
      webTrafficHeaderView.showProgressSpinner(headerUIModel.spinnerColor.parseColor())
    }
  }

  override fun showCloseButtonOnly() {
    webTrafficHeaderView.hideCountDown()
    webTrafficHeaderView.hideFinishButton()
    webTrafficHeaderView.hideNextButton()
    webTrafficHeaderView.setTitleText("")
    webTrafficHeaderView.hidePageCount()
    webTrafficHeaderView.hideProgressSpinner()
    webTrafficHeaderView.showCloseButton(headerUIModel.closeButtonColor.parseColor())
  }

  override fun backgrounded() {
    webTrafficHeaderView.disableCloseButton()
  }

  override fun foregrounded() {
    webTrafficHeaderView.enableCloseButton()
  }
}
