package com.hyprmx.android.sdk.initialization

import android.annotation.SuppressLint
import android.content.Context
import com.hyprmx.android.sdk.analytics.ClientErrorControllerIf
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.core.js.UncaughtErrorListener
import com.hyprmx.android.sdk.utility.HyprMXErrorType
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.HyprMXProperties
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import java.net.URL
import kotlin.coroutines.Continuation
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

/**
 * Performs initialization requests
 */
@SuppressLint("AddJavascriptInterface")
internal class InitializationController(
  private val jsEngine: JSEngine,
  private val errorCaptureController: ClientErrorControllerIf,
  private val context: Context,
  private val scope: CoroutineScope,
) :
  InitializationControllerIf,
  InitializationControllerIf.InitializationNativeInterface,
  UncaughtErrorListener,
  CoroutineScope by scope + CoroutineName("InitializationController") {

  lateinit var initializationDelegator: InitializationDelegator

  private var continuation: Continuation<InitializationResult>? = null

  init {
    jsEngine.addJavascriptInterface(this, JSNAME)
  }

  override suspend fun initialize(
    initializationDelegator: InitializationDelegator,
  ): InitializationResult = suspendCoroutine { cont1 ->
    this.initializationDelegator = initializationDelegator
    continuation = cont1
    val baseUrl = URL(HyprMXProperties.baseUrl)
    val url = if (baseUrl.port != -1) {
      "${baseUrl.host}:${baseUrl.port}"
    } else {
      baseUrl.host
    }
    with(this@InitializationController) {
      jsEngine.addUncaughtErrorListener(this)
      jsEngine.evaluate("$JSCONTROLLER.initFromNative('$url', ${hyprmxDelegate.failureCount});")
    }
  }

  override suspend fun javascriptUpgradeFailed(errorMsg: String): InitializationResult =
    suspendCoroutine { cont1 ->
      HyprMXLog.e(errorMsg)
      continuation = cont1
      errorCaptureController.sendClientError(
        HyprMXErrorType.HYPRErrorTypeSDKInternalError,
        errorMsg,
        ClientErrorControllerIf.SEVERITY_4,
      )
      with(this@InitializationController) {
        jsEngine.evaluate("$JSCONTROLLER.javascriptUpgradeFailed('$errorMsg');")
      }
    }

  companion object {
    const val TAG = "InitController"
    const val JSNAME = "HYPRInitListener"
    const val JSCONTROLLER = "globalThis.initializationController"
    const val ROLLBACK_HTTP_CODE = "406"
  }

  override fun updateJavascript(url: String, version: Int, timeout: Int) {
    HyprMXLog.d("updateJavascript to version $version")
    onInitializationComplete(InitializationResult.UpdateJavascript(url, version, timeout))
  }

  override fun initializationSuccessWithPlacements(placementsJsonString: String, jsVersion: Int) {
    hyprmxDelegate.hyprMXController?.mobileJSVersion = jsVersion
    onInitializationComplete(InitializationResult.Success(placementsJsonString))
  }

  override fun initializationFailed(error: String) {
    if (error.contains(ROLLBACK_HTTP_CODE)) {
      onInitializationComplete(InitializationResult.RollbackJavascript)
    } else {
      onInitializationComplete(InitializationResult.Failed(error))
    }
  }

  override fun setSharingEndpoint(sharingEndpoint: String) {
    launch {
      initializationDelegator.onSharingEndpointReceived(sharingEndpoint)
    }
  }

  override fun initializeOMSDK(omSdkUrl: String, omPartnerName: String, omApiVersion: String) {
    launch {
      initializationDelegator.onInitializeOMSDK(omSdkUrl, omPartnerName, omApiVersion)
    }
  }

  override fun setEnableAllLogs(enabled: Boolean) {
    launch {
      HyprMXLog.setAllLoggingEnabled(context, enabled)
    }
  }

  private fun onInitializationComplete(result: InitializationResult) {
    if (continuation == null) {
      errorCaptureController.sendClientError(
        HyprMXErrorType.HYPRErrorTypeSDKInternalError,
        "Initialization received complete already. Ignoring " +
          result.javaClass.simpleName,
        ClientErrorControllerIf.SEVERITY_4,
      )
      return
    }
    continuation?.let {
      continuation = null
      it.resume(result)
      jsEngine.removeUncaughtErrorListener(this)
    }
  }

  override fun onUncaughtError(error: String) {
    onInitializationComplete(InitializationResult.Failed(error))
  }
}
