package com.hyprmx.android.sdk.initialization

import com.hyprmx.android.sdk.analytics.ClientErrorControllerIf
import com.hyprmx.android.sdk.consent.ConsentStatus
import com.hyprmx.android.sdk.core.HyprMXController
import com.hyprmx.android.sdk.core.StorageHelperIf
import com.hyprmx.android.sdk.network.ConnectionConfiguration
import com.hyprmx.android.sdk.network.NetworkController
import com.hyprmx.android.sdk.network.NetworkResponse
import com.hyprmx.android.sdk.utility.HyprMXErrorType
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.withTimeout

/**
 * Interface to update or rollback the core JS code
 */

internal interface UpdateControllerIf {
  /**
   * Request to update javascript with a given timeout.
   *
   * @param timeout The timeout in seconds on how long this request should take.  If it takes longer, a failed result will be returned.
   * @return Returns the result of the update attempt.  If successful, it will return a new hypr controller to be used.
   */
  suspend fun updateJavascript(
    url: String,
    currentHyprMXController: HyprMXController,
    consentStatus: ConsentStatus,
    timeout: Long,
  ): UpdateResult

  suspend fun sharedJSRollback(
    hyprMXController: HyprMXController,
    consentStatus: ConsentStatus,
  ): UpdateResult
}

/**
 * This class is used to return the result of the update or rollback calls
 */
internal sealed class UpdateResult {
  object Failure : UpdateResult()
  object FailureWithTimeout : UpdateResult()

  class Success(val hyprMXController: HyprMXController) : UpdateResult()
}

/**
 * Represents the result of requesting new javascript
 */
sealed class JavascriptResponse {
  class Success(val newCoreJS: String) : JavascriptResponse()
  class Failure(val error: String) : JavascriptResponse()
}

/**
 * Responsible updating/rolling back the JS
 */

internal class UpdateController(
  private val networkController: NetworkController,
  private val storageHelper: StorageHelperIf,
  private val errorCaptureController: ClientErrorControllerIf,
) : UpdateControllerIf {
  override suspend fun updateJavascript(
    url: String,
    currentHyprMXController: HyprMXController,
    consentStatus: ConsentStatus,
    timeout: Long,
  ): UpdateResult {
    return try {
      withTimeout(timeout * 1000) {
        val jsDownloadResult = requestNewJavascript(url, timeout * 1000L)
        if (jsDownloadResult is JavascriptResponse.Success) {
          val updateResult =
            onUpdateJavascript(jsDownloadResult.newCoreJS, currentHyprMXController, consentStatus)
          if (updateResult is UpdateResult.Success) {
            return@withTimeout updateResult
          }
        }
        return@withTimeout UpdateResult.Failure
      }
    } catch (exception: Exception) {
      return UpdateResult.FailureWithTimeout
    }
  }

  /**
   * Attempts to download the new javascript
   */
  suspend fun requestNewJavascript(url: String, timeout: Long = 7000): JavascriptResponse {
    val response = networkController.getRequest(
      url,
      connectionConfiguration = ConnectionConfiguration(connectionTimeout = timeout.toInt()),
    )
    return if (response is NetworkResponse.Success && response.isResponseCodeSuccessful()) {
      val newCoreJS = response.value
      JavascriptResponse.Success(newCoreJS)
    } else {
      JavascriptResponse.Failure("Error upgrading javascript with unsuccessful network response.")
    }
  }

  /**
   * Performs a javascript core update.
   * If successful, returns a new HyprController to be used
   */
  suspend fun onUpdateJavascript(
    newCoreJS: String,
    currentHyprMXController: HyprMXController,
    consentStatus: ConsentStatus,
  ): UpdateResult {
    val tempHyprMXController = HyprMXController(
      currentHyprMXController.applicationContext,
      currentHyprMXController.distributorId,
      consentStatus = consentStatus,
    )
    val result = tempHyprMXController.initialize(
      newCoreJS = newCoreJS,
    )

    return if (result is HyprMXController.InitResult.Success) {
      HyprMXController.copyPlacements(tempHyprMXController, currentHyprMXController)
      UpdateResult.Success(tempHyprMXController)
    } else {
      tempHyprMXController.cancelJavascriptExecution()
      UpdateResult.Failure
    }
  }

  /**
   * Rolls back the javascript to the one that originally shipped with the SDK
   * On Success returns an initialized HyprController to be used instead
   */
  override suspend fun sharedJSRollback(
    hyprMXController: HyprMXController,
    consentStatus: ConsentStatus,
  ): UpdateResult {
    HyprMXLog.v("Received SDK Rollback Message.")

    val deleteSharedJSFromDisk = storageHelper.deleteCoreJSFile()
    if (!deleteSharedJSFromDisk) {
      errorCaptureController.sendClientError(
        HyprMXErrorType.HYPRErrorTypeSDKInternalError,
        "Rollback requested against base SDK",
        ClientErrorControllerIf.SEVERITY_3,
      )
      return UpdateResult.Failure
    }

    val tempHyprMXController = HyprMXController(
      hyprMXController.applicationContext,
      hyprMXController.distributorId,
      consentStatus = consentStatus,
    )

    val result = tempHyprMXController.initialize()

    return if (result is HyprMXController.InitResult.Success) {
      HyprMXController.copyPlacements(tempHyprMXController, hyprMXController)
      UpdateResult.Success(tempHyprMXController)
    } else {
      tempHyprMXController.cancelJavascriptExecution()
      UpdateResult.Failure
    }
  }
}
