
package com.hyprmx.android.sdk.network

import android.content.Context
import android.os.Build
import android.webkit.WebSettings
import com.hyprmx.android.sdk.extensions.readTextAndClose
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.InputStream
import java.net.HttpURLConnection
import java.net.URL

internal class HttpNetworkController(private val context: Context) : NetworkController {

  /**
   * Creates a request.
   *
   * This warning BlockingMethodInNonBlockingContext can be safely ignored
   * because this is run on a non-blocking context (Dispatchers.IO)
   */
  @Suppress("BlockingMethodInNonBlockingContext")
  override suspend fun <T> request(
    url: String,
    body: String?,
    method: String,
    connectionConfiguration: ConnectionConfiguration,
    response: suspend (InputStream) -> T,
  ): NetworkResponse<T> = withContext(Dispatchers.IO) {
    var errorResponseCode = 0
    try {
      val connection = URL(url).openConnection() as HttpURLConnection
      with(connection) {
        try {
          instanceFollowRedirects = connectionConfiguration.followRedirect
          readTimeout = connectionConfiguration.readTimeout
          connectTimeout = connectionConfiguration.connectionTimeout
          requestMethod = method

          if (method == "PATCH") {
            // URL Connection does not support PATCH by default.
            // These overrides allow it to work.
            requestMethod = "POST"
            setRequestProperty("X-HTTP-Method-Override", "PATCH")
          }

          if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            // Replace the User-Agent of the device with the one of the webview so that
            // the server can validate the version of the webview
            setRequestProperty("User-Agent", WebSettings.getDefaultUserAgent(context))
          }

          // Accept all MIME types.
          // when the Accept value was set to only application/json
          setRequestProperty("Accept", "*/*")
          setRequestProperty("Accept-Language", "en-us")

          connectionConfiguration.additionalHeaders.forEach {
            setRequestProperty(it.key, it.value)
          }
          if ((method == "POST" || method == "PUT" || method == "PATCH") && body != null) {
            // Measure based on UTF-8 size
            setFixedLengthStreamingMode(body.toByteArray().size)

            doOutput = true
            outputStream.writer().use { it.write(body) }
          }

          val contentLength = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            contentLengthLong
          } else {
            contentLength.toLong()
          }
          NetworkResponse.Success(
            responseCode,
            value = response(inputStream),
            length = contentLength,
            headers = headerFields,
          )
        } finally {
          errorResponseCode = responseCode
          disconnect()
        }
      }
    } catch (e: Exception) {
      // Something went wrong making the request, return a response code of 0
      // to the WebView, along with the reason why
      NetworkResponse.Failure(errorResponseCode, e.toString())
    }
  }

  override suspend fun getRequest(
    url: String,
    connectionConfiguration: ConnectionConfiguration,
  ): NetworkResponse<String> =
    request(
      url = url,
      body = null,
      method = "GET",
      connectionConfiguration = connectionConfiguration,
      response = { it.readTextAndClose() },
    )

  override suspend fun putRequest(
    url: String,
    body: String,
    connectionConfiguration: ConnectionConfiguration,
  ): NetworkResponse<String> =
    request(
      url = url,
      body = body,
      method = "PUT",
      connectionConfiguration = connectionConfiguration,
      response = { it.readTextAndClose() },
    )

  override suspend fun postRequest(
    url: String,
    body: String,
    connectionConfiguration: ConnectionConfiguration,
  ): NetworkResponse<String> =
    request(
      url = url,
      body = body,
      method = "POST",
      connectionConfiguration = connectionConfiguration,
      response = { it.readTextAndClose() },
    )
}
