@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.network

import java.io.InputStream

/**
 * Generic suspend network interface that turns a network request into a defined response.
 */
internal interface NetworkController {

  /**
   * Make a request, returning the body as the value in the Response.
   *
   * @param url The url to connect to
   * @param body Optional data to send with the request
   * @param connectionConfiguration Configuration for the connection.  Uses defaults that can be over-ridden
   * @param method HTTP method (GET, POST, PUT, etc.)
   * @param response lambda defining how to read the InputStream. Be sure to close the InputStream when done with it.
   */
  suspend fun <T> request(
    url: String,
    body: String? = null,
    method: String = "GET",
    connectionConfiguration: ConnectionConfiguration = DefaultGetConfiguration(),
    response: suspend (InputStream) -> T,
  ): NetworkResponse<T>

  /**
   * Makes a GET request returning the body as the value in the Response
   *
   * @param url The url to connect to
   * @param connectionConfiguration Configuration for the connection.  Uses defaults that can be over-ridden
   */
  suspend fun getRequest(
    url: String,
    connectionConfiguration: ConnectionConfiguration = DefaultGetConfiguration(),
  ): NetworkResponse<String>

  /**
   * Makes a PUT request returning the body as the value in the Response
   *
   * @param url The url to connect to
   * @param body String formatted JSON message
   * @param connectionConfiguration Configuration for the connection.  Uses defaults that can be over-ridden
   */
  suspend fun putRequest(
    url: String,
    body: String,
    connectionConfiguration: ConnectionConfiguration = DefaultPutConfiguration(),
  ): NetworkResponse<String>

  /**
   * Makes a POST request returning the body as the value in the Response
   *
   * @param url The url to connect to
   * @param body String formatted JSON message
   * @param connectionConfiguration Configuration for the connection.  Uses defaults that can be over-ridden
   */
  suspend fun postRequest(
    url: String,
    body: String,
    connectionConfiguration: ConnectionConfiguration = DefaultPostConfiguration(),
  ): NetworkResponse<String>
}

sealed class NetworkResponse<out T> {
  class Success<T>(
    override val code: Int,
    val value: T,
    val headers: Map<String, List<String>>,
    val length: Long,
  ) : NetworkResponse<T>()

  class Failure<T>(override val code: Int, val errorMessage: String) : NetworkResponse<T>()

  abstract val code: Int

  fun isResponseCodeSuccessful(): Boolean {
    return code in 200..299
  }

  fun isResponseCodeARedirect(): Boolean {
    return when (this.code) {
      in 300..308 -> true
      in 304..306 -> false
      else -> false
    }
  }
}

class ConnectionConfiguration(
  val followRedirect: Boolean = true,
  val readTimeout: Int = 30000,
  val connectionTimeout: Int = 30000,
  val additionalHeaders: Map<String, String> = HashMap(),
)

fun DefaultGetConfiguration(): ConnectionConfiguration = ConnectionConfiguration()
fun DefaultPutConfiguration(): ConnectionConfiguration = ConnectionConfiguration(additionalHeaders = mapOf("Content-Type" to "application/json"))
fun DefaultPostConfiguration(): ConnectionConfiguration = ConnectionConfiguration(additionalHeaders = mapOf("Content-Type" to "application/json"))
