@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.om

import android.content.Context
import android.view.View
import android.webkit.WebView
import com.hyprmx.android.sdk.network.NetworkController
import com.hyprmx.android.sdk.network.NetworkResponse
import com.hyprmx.android.sdk.tracking.VideoTracking
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.iab.omid.library.jungroup.Omid
import com.iab.omid.library.jungroup.adsession.FriendlyObstructionPurpose
import com.iab.omid.library.jungroup.adsession.Partner
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONException
import java.io.File

internal interface OpenMeasurementController {

  var adSession: OpenMeasurementAdSession?

  fun startJavascriptOMSession(
    sessionData: String,
    webView: WebView,
  )

  fun endSession()
  fun sessionActive(): Boolean

  /**
   * Add friendly obstruction which should then be excluded from all ad session viewability calculations.
   * @param friendlyObstruction View to be excluded from all ad session viewability calculations.
   * @param purpose Why this obstruction was necessary
   * @param detailedReason An explanation for why this obstruction is part of the ad experience if not already obvious from the FriendlyObstructionPurpose selected. Must be 50 characters or less and only contain characters `A-z`,`0-9` or space.
   */
  fun addFriendlyObstruction(
    friendlyObstruction: View,
    purpose: FriendlyObstructionPurpose,
    detailedReason: String?,
  )

  fun removeFriendlyObstruction(friendlyObstruction: View)
  fun removeAllFriendlyObstruction()
  fun getVideoTracker(duration: Float): VideoTracking

  fun generateBannerSession(): OpenMeasurementBannerSession
}

const val OMSDK_FOLDER_NAME = "hyprmx_omsdk"

/**
 * This class handles the interaction with OpenMeasurement.
 *
 * It is responsible for downloading the OM SDK javascript, initializing the partner,
 * and creating the appropriate AdSession
 */
internal class DefaultOpenMeasurementController(
  private val omPartner: Partner,
  private val networkController: NetworkController,
  private val coroutineScope: CoroutineScope,
  private val ioDispatcher: CoroutineDispatcher,

) : CoroutineScope by coroutineScope, OpenMeasurementController {

  override var adSession: OpenMeasurementAdSession? = null

  suspend fun fetchOmSdkJSLibrary(context: Context, omSdkUrl: String): String? = withContext(ioDispatcher) {
    val filename = omSdkUrl.substringAfterLast("/")
    val omSdkFile = openFile(context, filename)

    return@withContext if (omSdkFile.exists()) {
      val content = omSdkFile.readText()
      content
    } else {
      val response = networkController.getRequest(omSdkUrl)
      if (response is NetworkResponse.Success) {
        val js = response.value
        try {
          putFileToDisk(context, filename, js)
        } catch (exception: Exception) {
          HyprMXLog.e("Unable to store Open Measurement JS")
        }
        js
      } else {
        HyprMXLog.e("Error with network call to fetch OM SDK js library. $omSdkUrl")
        null
      }
    }
  }

  override fun startJavascriptOMSession(
    sessionData: String,
    webView: WebView,
  ) {
    if (adSession != null) {
      HyprMXLog.d("Existing Ad Session already exists.")
      return
    }

    try {
      adSession = OpenMeasurementJSAdSession(omPartner, sessionData)
      adSession?.initializeAdSession(webView)
    } catch (e: JSONException) {
      HyprMXLog.d("Error starting js om ad session - ${e.localizedMessage}")
    }
  }

  override fun endSession() {
    adSession?.endSession()
    adSession = null
  }

  override fun sessionActive(): Boolean {
    return adSession != null
  }

  override fun addFriendlyObstruction(
    friendlyObstruction: View,
    purpose: FriendlyObstructionPurpose,
    detailedReason: String?,
  ) {
    try {
      adSession?.addFriendlyObstruction(friendlyObstruction, purpose, detailedReason)
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("Error registering obstruction with error msg - ${e.localizedMessage}")
    }
  }

  override fun removeFriendlyObstruction(friendlyObstruction: View) {
    try {
      adSession?.removeFriendlyObstruction(friendlyObstruction)
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("Error removing registered obstruction with error msg - ${e.localizedMessage}")
    }
  }

  override fun removeAllFriendlyObstruction() {
    try {
      adSession?.removeAllFriendlyObstruction()
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("Error removing all friendly obstruction with error msg - ${e.localizedMessage}")
    }
  }

  override fun getVideoTracker(duration: Float): VideoTracking {
    return adSession?.getOMTrackingHandler(duration)
      ?: object : VideoTracking {}
  }

  override fun generateBannerSession(): OpenMeasurementBannerSession {
    return OpenMeasurementBannerSession(partner = omPartner)
  }

  private suspend fun openFile(context: Context, fileName: String): File = withContext(ioDispatcher) {
    return@withContext File("${context.cacheDir.absolutePath}/$OMSDK_FOLDER_NAME/$fileName")
  }

  /**
   * Puts the file to the cache directory.  Only 1 file is allowed in the directory at a time.
   *
   * @param fileName The file name to store
   */
  private suspend fun putFileToDisk(context: Context, fileName: String, content: String) = withContext(ioDispatcher) {
    val omsdkDir = File("${context.cacheDir.absolutePath}/$OMSDK_FOLDER_NAME/")
    if (omsdkDir.exists()) {
      HyprMXLog.d("Cleaning cache directory successful = ${omsdkDir.deleteRecursively()}")
    }
    omsdkDir.mkdir()
    File(omsdkDir, fileName).writeText(content)
  }
}

internal fun OpenMeasurementController(
  appContext: Context,
  omPartnerName: String,
  omApiVersion: String,
  networkController: NetworkController,
  coroutineScope: CoroutineScope,
  ioDispatcher: CoroutineDispatcher = Dispatchers.IO,
): OpenMeasurementController? {
  val activated = try {
    Omid.activate(appContext)
    true
  } catch (e: IllegalArgumentException) {
    HyprMXLog.e("Open Measurement SDK failed to activate with exception: ${e.localizedMessage}")
    false
  }

  if (!activated) {
    HyprMXLog.e("Open Measurement SDK failed to activate")
    return null
  }

  val omPartner = try {
    Partner.createPartner(omPartnerName, omApiVersion)
  } catch (e: IllegalArgumentException) {
    HyprMXLog.e("Error creating Open Measurement Partner with error: ${e.localizedMessage}")
    return null
  }

  return DefaultOpenMeasurementController(omPartner, networkController, coroutineScope, ioDispatcher)
}
