package com.hyprmx.android.sdk.overlay

import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.ActivityEventInterface
import com.hyprmx.android.sdk.core.CaptureImageNativeInterface
import com.hyprmx.android.sdk.core.PermissionRequestNativeInterface
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import java.lang.ref.WeakReference

internal interface BrowserNativeInterface : ActivityEventInterface, CaptureImageNativeInterface, PermissionRequestNativeInterface {
  @RetainMethodSignature
  fun webViewLoadStarted(url: String)

  @RetainMethodSignature
  fun closeBrowser()

  @RetainMethodSignature
  fun displayError(message: String)

  @RetainMethodSignature
  fun setBackButtonEnabled(enabled: Boolean)

  @RetainMethodSignature
  fun setForwardButtonEnabled(enabled: Boolean)

  @RetainMethodSignature
  fun setHeader(header: String)
}

internal interface BrowserPresenterInterface {
  suspend fun webViewLoadStarted(url: String)
  suspend fun closeBrowser()
  suspend fun displayError(message: String)
  suspend fun setBackButtonEnabled(enabled: Boolean)
  suspend fun setForwardButtonEnabled(enabled: Boolean)
  suspend fun setTitle(title: String)
  suspend fun captureImage()
  suspend fun permissionRequest(permissions: String, permissionId: Int)
  suspend fun openShareSheet(data: String)
  suspend fun createCalendarEvent(data: String)
  suspend fun storePicture(url: String)
  suspend fun showToast(resourceId: Int)
  suspend fun openOutsideApplication(url: String)
}

internal class BrowserPresenterAdapter(
  presenter: BrowserPresenterInterface,
  scope: CoroutineScope,
) :
  BrowserNativeInterface, CoroutineScope by scope {
  private val browserPresenter: WeakReference<BrowserPresenterInterface> = WeakReference(presenter)
  override fun webViewLoadStarted(url: String) {
    launch { browserPresenter.get()?.webViewLoadStarted(url) }
  }

  override fun closeBrowser() {
    launch { browserPresenter.get()?.closeBrowser() }
  }

  override fun permissionRequest(permissions: String, permissionId: Int) {
    launch { browserPresenter.get()?.permissionRequest(permissions, permissionId) }
  }

  override fun captureImage() {
    launch { browserPresenter.get()?.captureImage() }
  }

  override fun displayError(message: String) {
    launch { browserPresenter.get()?.displayError(message) }
  }

  override fun setBackButtonEnabled(enabled: Boolean) {
    launch { browserPresenter.get()?.setBackButtonEnabled(enabled) }
  }

  override fun setForwardButtonEnabled(enabled: Boolean) {
    launch { browserPresenter.get()?.setForwardButtonEnabled(enabled) }
  }

  override fun setHeader(header: String) {
    launch { browserPresenter.get()?.setTitle(header) }
  }

  override fun openOutsideApplication(url: String) {
    launch { browserPresenter.get()?.openOutsideApplication(url) }
  }

  override fun openShareSheet(data: String) {
    launch { browserPresenter.get()?.openShareSheet(data) }
  }

  override fun createCalendarEvent(data: String) {
    launch { browserPresenter.get()?.createCalendarEvent(data) }
  }

  override fun storePicture(url: String) {
    launch { browserPresenter.get()?.storePicture(url) }
  }

  override fun showToast(resourceId: Int) {
    launch { browserPresenter.get()?.showToast(resourceId) }
  }
}
