package com.hyprmx.android.sdk.overlay

import com.hyprmx.android.sdk.core.ImageCaptureInterface
import com.hyprmx.android.sdk.core.SharedInterface
import com.hyprmx.android.sdk.presentation.ArgumentKey
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PublishingEvent
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.PermissionResult

internal interface BrowserSharedInterface : ImageCaptureInterface, SharedInterface {
  fun onClose()
  fun onBrowserReady()
  fun onPermissionResponse(permissionResults: List<PermissionResult>, requestCode: Int)
  fun onNavigateBackPressed()
  fun onNavigateForwardPressed()
  fun onBackButtonPressed()
  fun onSharePressed()
}

internal class BrowserSharedConnector(private val eventPublisher: PresentationEventPublisher) :
  BrowserSharedInterface {
  override var viewModelIdentifier: String
    get() = eventPublisher.viewModelIdentifier
    set(value) {
      eventPublisher.viewModelIdentifier = value
    }

  override fun onClose() {
    eventPublisher.publishEvent(PublishingEvent.ON_CLOSE)
  }

  override fun onBrowserReady() {
    eventPublisher.publishEvent(PublishingEvent.ON_BROWSER_READY)
  }

  override fun onPermissionResponse(permissionResults: List<PermissionResult>, requestCode: Int) {
    HyprMXLog.d("permissionResponse - $requestCode")
    eventPublisher.publishEvent(
      PublishingEvent.PERMISSION_RESULT,
      mapOf(
        ArgumentKey.PERMISSION_LIST to permissionResults.map { it.toJSON() },
        ArgumentKey.PERMISSION_ID to requestCode,
      ),
    )
  }

  override fun onNavigateBackPressed() {
    eventPublisher.publishEvent(PublishingEvent.NAVIGATE_BACK_PRESSED)
  }

  override fun onNavigateForwardPressed() {
    eventPublisher.publishEvent(PublishingEvent.NAVIGATE_FORWARD_PRESSED)
  }

  override fun onBackButtonPressed() {
    eventPublisher.publishEvent(this::onBackButtonPressed.name)
  }

  override fun onSharePressed() {
    eventPublisher.publishEvent(PublishingEvent.SHARE_SHEET_PRESSED)
  }

  override fun attach(nativeObject: Any) {
    eventPublisher.setPresenter(nativeObject)
  }

  override fun destroy() {
    eventPublisher.destroy()
  }

  override fun imageCaptured(url: String?) {
    eventPublisher.publishEvent(PublishingEvent.IMAGE_CAPTURED, mapOf(ArgumentKey.URL to (url ?: "")))
  }
}
