package com.hyprmx.android.sdk.overlay

import android.content.Context
import android.widget.Toast
import androidx.annotation.VisibleForTesting
import com.hyprmx.android.sdk.core.HyprMXReInit
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.mvp.LifecycleEventAdapter
import com.hyprmx.android.sdk.mvp.LifecycleEventHandler
import com.hyprmx.android.sdk.presentation.BrowserEventPublisher
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PresentationFactory
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.startResolvedActivity
import com.hyprmx.android.sdk.utility.toArrayList
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONArray

internal class HyprMXBrowserPresenter(
  override var view: HyprMXBrowserContract.View?,
  viewModelIdentifier: String,
  jsEngine: JSEngine,
  coroutineScope: CoroutineScope,
  eventPublisher: PresentationEventPublisher = BrowserEventPublisher(jsEngine, viewModelIdentifier),
  lifeCycleHandler: LifecycleEventHandler = LifecycleEventAdapter(eventPublisher, coroutineScope),
  private val sharedInterface: BrowserSharedInterface = BrowserSharedConnector(eventPublisher),
) : HyprMXBrowserContract.Presenter,
  CoroutineScope by coroutineScope,
  LifecycleEventHandler by lifeCycleHandler,
  BrowserPresenterInterface,
  BrowserSharedInterface by sharedInterface,
  HyprMXReInit {

  /**
   * Debounce navigation event.  If true, blocks navigation events until UI event received.
   */
  @VisibleForTesting
  var navBlockedUntilUIEvent = false
  override var launchContext: Context? = null
  override var stateListener: PresentationStateListener? = null

  @VisibleForTesting
  var presenterFactory: PresentationFactory? = null
    get() = field ?: hyprmxDelegate.hyprMXController?.presenterFactory

  init {
    attach(BrowserPresenterAdapter(this, this))
    hyprmxDelegate.hyprMXController?.registerSDKReInitListener(this)
    initializeViewState()
  }

  override fun initializeViewState() {
    view?.navigateBackEnabled(false)
    view?.navigateForwardEnabled(false)
    view?.setTitleText("")
    view?.closeButtonEnabled(true)
  }

  override fun cleanup() {
    destroy()
    view = null
    stateListener = null
    launchContext = null
  }

  override fun onNavigateBackPressed() {
    if (navBlockedUntilUIEvent) {
      return
    }
    navBlockedUntilUIEvent = true
    sharedInterface.onNavigateBackPressed()
  }

  override fun onNavigateForwardPressed() {
    if (navBlockedUntilUIEvent) {
      return
    }
    navBlockedUntilUIEvent = true
    sharedInterface.onNavigateForwardPressed()
  }

  override fun onCloseClicked() {
    sharedInterface.onClose()
    stateListener?.onBrowserClosed()
    view?.hyprMXBrowserClosed()
    launch {
      view?.finishActivity()
    }
  }

  override fun onBackPressed() {
    if (navBlockedUntilUIEvent) {
      return
    }
    navBlockedUntilUIEvent = true
    sharedInterface.onBackButtonPressed()
  }

  override fun onSharePressed() {
    if (navBlockedUntilUIEvent) {
      return
    }
    navBlockedUntilUIEvent = true
    sharedInterface.onSharePressed()
  }

  override suspend fun webViewLoadStarted(url: String): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("webViewLoadStarted $url")
  }

  override suspend fun closeBrowser(): Unit = withContext(Dispatchers.Main) {
    onCloseClicked()
  }

  override suspend fun permissionRequest(permissions: String, permissionId: Int): Unit =
    withContext(Dispatchers.Main) {
      view?.requestPermission(JSONArray(permissions).toArrayList().toTypedArray(), permissionId)
    }

  override suspend fun captureImage(): Unit = withContext(Dispatchers.Main) {
    view?.captureImage()
  }

  override suspend fun displayError(message: String): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("Dislay error occured while displaying the browser: $message")

    view?.finishActivity()
    cleanup()
  }

  override suspend fun setBackButtonEnabled(enabled: Boolean): Unit =
    withContext(Dispatchers.Main) {
      HyprMXLog.d("Updating back navigation to ($enabled)")
      navBlockedUntilUIEvent = false
      view?.navigateBackEnabled(enabled)
    }

  override suspend fun setForwardButtonEnabled(enabled: Boolean): Unit =
    withContext(Dispatchers.Main) {
      HyprMXLog.d("Updating forward navigation to ($enabled)")
      navBlockedUntilUIEvent = false
      view?.navigateForwardEnabled(enabled)
    }

  override suspend fun setTitle(title: String): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("Updating title to ($title)")
    navBlockedUntilUIEvent = false
    view?.setTitleText(title)
  }

  override suspend fun openOutsideApplication(url: String): Unit = withContext(Dispatchers.Main) {
    launchContext?.let {
      if (startResolvedActivity(it, url)) {
        stateListener?.onOutsideAppPresented()
      }
    }
  }

  override suspend fun openShareSheet(data: String): Unit = withContext(Dispatchers.Main) {
    view?.openShareSheet(data)
    navBlockedUntilUIEvent = false
  }

  override suspend fun createCalendarEvent(data: String): Unit = withContext(Dispatchers.Main) {
    view?.createCalendarEvent(data)
  }

  override suspend fun storePicture(url: String): Unit = withContext(Dispatchers.Main) {
    view?.asyncSavePhoto(url)
  }

  override suspend fun showToast(resourceId: Int): Unit = withContext(Dispatchers.Main) {
    launchContext?.let { context ->
      Toast.makeText(
        context,
        context.getString(resourceId),
        Toast.LENGTH_SHORT,
      ).show()
    }
  }

  override fun onSDKReInit() {
    view?.cleanupPresenters()
    view?.finishActivity()
    cleanup()
  }
}

internal interface PresentationStateListener {
  fun onBrowserPresented()
  fun onBrowserClosed()
  fun onOutsideAppPresented()
}
