package com.hyprmx.android.sdk.placement

import android.annotation.SuppressLint
import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import org.json.JSONArray
import org.json.JSONException

internal interface PlacementController {

  var placements: Set<PlacementImpl>

  /**
   * Initializes placements by injecting placements to javascript's PlacementController.
   */
  @Throws(JSONException::class)
  suspend fun initializePlacements(
    placementsJsonString: String,
    placementDelegator: PlacementImpl.PlacementDelegator,
  )

  /**
   * Gets the placement object associated with the placement name.
   *
   * @param placementName The name of the placement to retrieve
   * @return The placement with the corresponding name.  If not found, returns an INVALID PlacementType.
   */
  fun getPlacement(placementName: String): Placement

  /**
   * Loads the ad for the placement
   * @param placementName The placement to load ads for
   */
  suspend fun loadAd(placementName: String): Boolean

  /**
   * Inbound API from JS
   *
   * Indicates the ad associated with the placement is no longer available
   *
   * @param placementName The name of the placement
   *
   */
  @RetainMethodSignature
  fun onAdCleared(placementName: String)

  /**
   * Inbound API from JS
   *
   * Indicates the ad associated with the placement has expired and is no longer available
   *
   * @param placementName The name of the placement
   *
   */
  @RetainMethodSignature
  fun onAdExpired(placementName: String)

  /**
   * Inbound API from JS
   *
   * An impression occurred on this placement
   *
   * @param placementName The name of the placement
   *
   */
  @RetainMethodSignature
  fun adImpression(placementName: String)

  /**
   * Gets the ad availability from the shared code
   * @return True if an ad is available
   */
  fun isAdAvailable(placementName: String): Boolean
}

/**
 * Controller class that handles managing placements, requesting ads for placements
 * and determining if a placement has a valid offer to show.
 */
@SuppressLint("AddJavascriptInterface")
internal class PlacementControllerImpl(
  private val jsEngine: JSEngine,
) :
  PlacementController, CoroutineScope by MainScope() {

  companion object {
    const val TAG = "PlacementController"
    const val JS_INTERFACE_NAME = "HYPRPlacementListener"
    const val JS_CONTROLLER_NAME = "HYPRPlacementController"

    fun fromJson(
      placementDelegator: PlacementImpl.PlacementDelegator,
      placementsJsonString: String,
    ): Set<PlacementImpl> {
      val jsonArray = JSONArray(placementsJsonString)
      return (0 until jsonArray.length())
        .map { PlacementImpl.fromJson(placementDelegator, jsonArray.get(it).toString()) }
        .toSet()
    }
  }

  override var placements: Set<PlacementImpl> = mutableSetOf()

  init {
    jsEngine.addJavascriptInterface(this, JS_INTERFACE_NAME)
  }

  /**
   * Initializes placements by injecting placements to javascript's PlacementController.
   */
  @Throws(JSONException::class)
  override suspend fun initializePlacements(
    placementsJsonString: String,
    placementDelegator: PlacementImpl.PlacementDelegator,
  ) {
    fromJson(placementDelegator, placementsJsonString)
      .forEach { newPlacement ->
        val placement = placements.find { it.name == newPlacement.name }
        if (placement != null) {
          placement.type = newPlacement.type
          placement.placementDelegate = placementDelegator
        } else {
          (placements as MutableSet).add(newPlacement)
        }
      }
  }

  /**
   * Gets the placement object associated with the placement name.
   *
   * @param placementName The name of the placement to retrieve
   * @return The placement with the corresponding name.  If not found, returns an INVALID PlacementType.
   */
  override fun getPlacement(placementName: String): Placement {
    return placements.firstOrNull { placementName == it.name } ?: run {
      val placement = invalidPlacement(placementName)
      (placements as MutableSet).add(placement as PlacementImpl)
      placement
    }
  }

  override suspend fun loadAd(placementName: String) =
    jsEngine.asyncEvaluateScriptForResponse(
      method = "loadAd",
      caller = JS_CONTROLLER_NAME,
      args = "['$placementName']",
    ) as? Boolean == true

  @RetainMethodSignature
  override fun onAdCleared(placementName: String) {
    launch {
      HyprMXLog.d("onAdCleared - $placementName")
      getPlacement(placementName) as PlacementImpl
    }
  }

  @RetainMethodSignature
  override fun onAdExpired(placementName: String) {
    launch {
      HyprMXLog.d("onAdExpired - $placementName")
      val placement: PlacementImpl = getPlacement(placementName) as PlacementImpl
      val placementListener = placement.placementExpiryListener
      placementListener?.onAdExpired(placement)
    }
  }

  @RetainMethodSignature
  override fun adImpression(placementName: String) {
    launch {
      HyprMXLog.d("onAdImpression - $placementName")
      val placement: PlacementImpl = getPlacement(placementName) as PlacementImpl
      val placementListener = placement.showListener
      placementListener?.onAdImpression(placement)
    }
  }
  override fun isAdAvailable(placementName: String): Boolean =
    jsEngine.evaluateScriptForResponse("$JS_CONTROLLER_NAME.isAdAvailable('$placementName')") as Boolean
}
