package com.hyprmx.android.sdk.preferences

import android.content.Context
import android.content.SharedPreferences
import androidx.preference.PreferenceManager
import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.js.JSEngine
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus
import org.json.JSONObject

/**
 * This class provides a bridge from the JS code into the shared preferences.
 * Note:  The sync methods are called on the JS Bridge thread
 */
internal class PreferencesController(
  appContext: Context,
  private val jsEngine: JSEngine,
  val scope: CoroutineScope,
) :
  PreferencesControllerIf,
  SharedPreferences.OnSharedPreferenceChangeListener,
  CoroutineScope by scope + CoroutineName("PreferencesController") {

  // This has to be accessed lazily from the IO thread to avoid
  // disk reads on the main thread.
  internal val preferences: SharedPreferences by lazy {
    PreferenceManager.getDefaultSharedPreferences(appContext)
  }

  internal val keysToMonitor: MutableMap<String, String> = HashMap()

  companion object {
    const val JSCONTROLLER = "HYPRSharedDataController"
  }

  init {
    jsEngine.addJavascriptInterface(this, JSCONTROLLER)
    launch(Dispatchers.IO) {
      preferences.registerOnSharedPreferenceChangeListener(this@PreferencesController)
    }
  }

  /**
   *  Gets the value for the key.  Returns a JSON formatted string in the following form:
   *
   *  { "value":"" }
   *
   *  @return The boolean if it exists, null otherwise
   */
  @RetainMethodSignature
  override fun getSharedValue(key: String): String {
    val jsonObject = JSONObject()
    val value = preferences.all[key]
    when (value) {
      is Boolean -> preferences.getBoolean(key, false)
      is String -> preferences.getString(key, "")
      is Int -> preferences.getInt(key, 0)
      is Float -> preferences.getFloat(key, 0F)
      is Long -> preferences.getLong(key, 0)
      else -> null
    }.let { jsonObject.put("value", it) }

    return jsonObject.toString()
  }

  /**
   *  Request to monitor the key in the preferences
   *  for changes
   *
   *  Note:  Only 1 listener per key
   *
   *  @listener The object name of the JS Component to call back to
   *  @key The key to monitor
   */
  @RetainMethodSignature
  override fun monitorSharedValue(listener: String, key: String) {
    keysToMonitor[key] = listener
  }

  /**
   * Called when a shared preference is changed, added, or removed. This
   * may be called even if a preference is set to its existing value.
   *
   * This callback will be run on your main thread.
   *
   * @param sharedPreferences The [SharedPreferences] that received
   * the change.
   * @param key The key of the preference that was changed, added, or
   * removed.
   */
  override fun onSharedPreferenceChanged(sharedPreferences: SharedPreferences?, key: String?) {
    if (keysToMonitor.contains(key) && sharedPreferences != null) {
      val value = sharedPreferences.all[key]
      val jsonObject = JSONObject()
      jsonObject.put("key", key)
      jsonObject.put("value", value)
      val jsonString = jsonObject.toString()
      jsEngine.evaluate("${keysToMonitor[key]}.onValueChanged($jsonString);")
    }
  }

  /**
   * Stops monitoring for shared preference changes
   */
  override fun stopMonitoring() {
    keysToMonitor.clear()
    preferences.unregisterOnSharedPreferenceChangeListener(this)
  }
}
