@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.presentation

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import com.hyprmx.android.sdk.activity.ActivityDependencyHolder
import com.hyprmx.android.sdk.activity.HyprMXNoOffersActivity
import com.hyprmx.android.sdk.activity.HyprMXOfferViewerActivity
import com.hyprmx.android.sdk.activity.HyprMXRequiredInformationActivity
import com.hyprmx.android.sdk.analytics.ClientErrorControllerIf
import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.api.data.Ad
import com.hyprmx.android.sdk.api.data.RequiredInformation
import com.hyprmx.android.sdk.api.data.UiComponents
import com.hyprmx.android.sdk.core.ApplicationModule
import com.hyprmx.android.sdk.core.HyprMXErrors
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.fullscreen.FullScreenSharedConnector
import com.hyprmx.android.sdk.placement.HyprMXRewardedShowListener
import com.hyprmx.android.sdk.placement.Placement
import com.hyprmx.android.sdk.placement.PlacementImpl
import com.hyprmx.android.sdk.presentation.PresentationController.Companion.JS_CONTROLLER_NAME
import com.hyprmx.android.sdk.presentation.PresentationController.Companion.JS_INTERFACE_NAME
import com.hyprmx.android.sdk.utility.HyprMXErrorType
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.Result
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus

/**
 *
 */
internal interface PresentationController {

  companion object {
    const val JS_INTERFACE_NAME = "HYPRPresentationListener"
    const val JS_CONTROLLER_NAME = "HYPRPresentationController"
    const val JS_PUBLISH_EVENT = "ViewModelController.publishEvent"
    const val JS_DESTROY_VIEW_MODEL = "ViewModelController.removeViewModel"
    const val JS_BIND_BANNER_VIEW_MODEL = "HYPRPresentationController.bindBannerViewModel"
    const val JS_GET_VIEW_MODEL = "ViewModelController.getViewModel"
    const val JS_SET_PRESENTER = "setPresenter"
    const val JS_SET_WEBVIEW_PRESENTER = "setWebViewPresenter"
  }

  suspend fun showAd(placement: PlacementImpl)

  /**
   * Incoming interface from the shared code
   */
  interface PresentationNativeInterface {
    @RetainMethodSignature
    fun adStarted(placementName: String)

    @RetainMethodSignature
    fun adFinished(placementName: String)

    @RetainMethodSignature
    fun adCanceled(placementName: String)

    @RetainMethodSignature
    fun adDisplayError(placementName: String, errorMsg: String)

    @RetainMethodSignature
    fun showRequiredInfo(requiredInfoString: String, uiComponentsString: String)

    @RetainMethodSignature
    fun onDisplayAd(viewModelId: String, viewType: String, offerJson: String)

    @RetainMethodSignature
    fun showNoAd(uiComponentsString: String)

    @RetainMethodSignature
    fun adRewarded(placementName: String, rewardText: String, rewardQuantity: Int)
  }
}

internal fun PresentationController(
  applicationModule: ApplicationModule,
  presentationDelegator: PresentationDelegator,
): PresentationController = DefaultPresentationController(
  applicationModule = applicationModule,
  clientErrorController = applicationModule.errorCaptureController,
  context = applicationModule.applicationContext,
  jsEngine = applicationModule.jsEngine,
  presentationDelegator = presentationDelegator,
  scope = applicationModule.scope,
)

interface PresentationDelegator {
  fun getPlacement(placementName: String): Placement
}

@SuppressLint("AddJavascriptInterface")
internal class DefaultPresentationController(
  private val applicationModule: ApplicationModule,
  val clientErrorController: ClientErrorControllerIf,
  val context: Context,
  private val jsEngine: JSEngine,
  val presentationDelegator: PresentationDelegator,
  val scope: CoroutineScope,
) :
  PresentationController,
  PresentationController.PresentationNativeInterface,
  ActivityResultListener,
  CoroutineScope by scope + CoroutineName("DefaultPresentationController") {

  companion object {
    const val TAG = "DefaultPresentationController"
  }

  init {
    jsEngine.addJavascriptInterface(this, JS_INTERFACE_NAME)
  }

  override suspend fun showAd(placement: PlacementImpl) {
    val placementName = placement.name
    jsEngine.evaluate("$JS_CONTROLLER_NAME.showFullscreenAd('$placementName');")
  }

  override suspend fun onAdDismissed(wasCompleted: Boolean) {
    ActivityDependencyHolder.resetInstanceFields()
    jsEngine.evaluate("$JS_CONTROLLER_NAME.adDismissed($wasCompleted);")
  }

  override suspend fun onAdRewarded() {
    jsEngine.evaluate("$JS_CONTROLLER_NAME.adRewarded();")
  }

  override suspend fun requiredInfoPresentationCancelled() {
    jsEngine.evaluate("$JS_CONTROLLER_NAME.requiredInfoPresentationCancelled();")
  }

  override suspend fun requiredInfoPresentationCompletedWithParams(requiredInfoParams: String) {
    jsEngine.evaluate(
      JS_CONTROLLER_NAME + ".requiredInfoPresentationCompletedWithParams(" +
        requiredInfoParams + ");",
    )
  }

  @RetainMethodSignature
  override fun adStarted(placementName: String) {
    launch {
      val placement =
        presentationDelegator.getPlacement(placementName) as PlacementImpl
      val placementListener = placement.showListener
      placementListener?.onAdStarted(placement as Placement)
    }
  }

  @RetainMethodSignature
  override fun adFinished(placementName: String) {
    launch {
      val placement =
        presentationDelegator.getPlacement(placementName) as PlacementImpl
      val placementListener = placement.showListener
      placementListener?.onAdClosed(placement as Placement, true)
      placement.showListener = null
    }
  }

  @RetainMethodSignature
  override fun adCanceled(placementName: String) {
    launch {
      val placement =
        presentationDelegator.getPlacement(placementName) as PlacementImpl
      val placementListener = placement.showListener
      placementListener?.onAdClosed(placement as Placement, false)
      ActivityDependencyHolder.resetInstanceFields()
    }
  }

  @RetainMethodSignature
  override fun adDisplayError(placementName: String, errorMsg: String) {
    launch {
      val errorLogMsg = "adDisplayError with error: $errorMsg"
      HyprMXLog.d(errorLogMsg)
      val placement =
        presentationDelegator.getPlacement(placementName) as PlacementImpl
      val placementListener = placement.showListener
      placementListener?.onAdDisplayError(placement as Placement, HyprMXErrors.DISPLAY_ERROR)
      clientErrorController.sendClientError(
        HyprMXErrorType.HYPRErrorAdDisplay,
        errorLogMsg,
        ClientErrorControllerIf.SEVERITY_2,
      )
    }
  }

  @RetainMethodSignature
  override fun adRewarded(placementName: String, rewardText: String, rewardQuantity: Int) {
    launch {
      val placement =
        presentationDelegator.getPlacement(placementName) as PlacementImpl
      val placementListener = placement.showListener
      if (placementListener is HyprMXRewardedShowListener) {
        placementListener.onAdRewarded(placement as Placement, rewardText, rewardQuantity)
      }
    }
  }

  @RetainMethodSignature
  override fun showRequiredInfo(requiredInfoString: String, uiComponentsString: String) {
    launch {
      val intent = Intent(context, HyprMXRequiredInformationActivity::class.java).apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
      }

      when (
        val requiredInformationResult =
          RequiredInformation.fromRequiredInfoJSON(requiredInfoString)
      ) {
        is Result.Success -> {
          ActivityDependencyHolder.requiredInfoControllerFactory =
            applicationModule.createRequiredInformationControllerFactory(
              activityResultListener = this@DefaultPresentationController,
              imageCacheManager = applicationModule.imageCacheManager,
              uiComponents = UiComponents.fromJson(uiComponentsString),
              requiredInformation = requiredInformationResult.value,
            )
          context.startActivity(intent)
        }
        is Result.Failure -> {
          HyprMXLog.e(
            "Cancelling ad because Required Information is Invalid. " +
              requiredInformationResult.message,
          )
          requiredInfoPresentationCancelled()
        }
      }
    }
  }

  override fun onDisplayAd(viewModelId: String, viewType: String, offerJson: String) {
    startAdActivity(offerJson, viewModelId)
  }

  @RetainMethodSignature
  override fun showNoAd(uiComponentsString: String) {
    launch {
      val intent = Intent(context, HyprMXNoOffersActivity::class.java).apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
      }
      ActivityDependencyHolder.noAdViewControllerFactory =
        applicationModule.createNoAdControllerFactory(
          activityResultListener = this@DefaultPresentationController,
          uiComponents = UiComponents.fromJson(uiComponentsString),
        )
      context.startActivity(intent)
    }
  }

  private fun startAdActivity(
    adJSONString: String,
    viewModelId: String,
  ) {
    when (val result = Ad.fromJson(adJSONString, clientErrorController)) {
      is Result.Success -> {
        val intent = Intent(context, HyprMXOfferViewerActivity::class.java).apply {
          flags = Intent.FLAG_ACTIVITY_NEW_TASK
        }

        val eventPublisher = FullScreenPresentationEventPublisher(jsEngine, viewModelId)
        ActivityDependencyHolder.adControllerFactory =
          applicationModule.createViewControllerFactory(
            applicationModule = applicationModule,
            ad = result.value,
            activityResultListener = this@DefaultPresentationController,
            eventPublisher = eventPublisher,
            fullScreenSharedConnector = FullScreenSharedConnector(eventPublisher),
          )
        context.startActivity(intent)
      }
      is Result.Failure -> {
        launch {
          onAdDismissed(false)
        }
      }
    }
  }
}
