package com.hyprmx.android.sdk.presentation

import android.annotation.SuppressLint
import android.content.Context
import com.hyprmx.android.sdk.banner.HyprMXBannerContract
import com.hyprmx.android.sdk.banner.HyprMXBannerPresenter
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.overlay.HyprMXBrowserContract
import com.hyprmx.android.sdk.overlay.HyprMXBrowserPresenter
import com.hyprmx.android.sdk.webview.HyprMXWebView
import com.hyprmx.android.sdk.webview.HyprMXWebViewContract
import com.hyprmx.android.sdk.webview.HyprMXWebViewPresenter
import kotlinx.coroutines.CoroutineScope

internal class DefaultPresentationFactory(
  private val jsEngine: JSEngine,
  private val coroutineScope: CoroutineScope,
) : PresentationFactory {

  private val preloadedWebViews = mutableMapOf<String, HyprMXWebView>()
  private var preloadedBrowserPresenter = mutableMapOf<String, HyprMXBrowserContract.Presenter>()

  override fun clearPreloadedWebView(viewModelIdentifier: String, destroy: Boolean) {
    if (destroy) {
      preloadedWebViews[viewModelIdentifier]?.cleanup()
    }
    preloadedWebViews.remove(viewModelIdentifier)
  }

  override fun makeBrowserWebView(
    context: Context,
    viewModelIdentifier: String,
  ): HyprMXWebView {
    return HyprMXWebView(context).apply {
      initialize(viewModelIdentifier)
    }
  }

  override fun makePreloadedWebView(
    context: Context,
    viewModelIdentifier: String,
  ): HyprMXWebView {
    return preloadedWebViews[viewModelIdentifier] ?: HyprMXWebView(context).also {
      it.initialize(viewModelIdentifier = viewModelIdentifier)
      preloadedWebViews[viewModelIdentifier] = it
    }
  }

  override fun makeBannerPresenter(
    view: HyprMXBannerContract.View?,
    placementName: String,
  ): HyprMXBannerContract.Presenter {
    return HyprMXBannerPresenter(view, placementName, jsEngine, coroutineScope)
  }

  // Suppressed.  Used in API 21 and above which is what we show on.
  @SuppressLint("NewApi")
  override fun makeWebViewPresenter(view: HyprMXWebViewContract.View, viewModelIdentifier: String): HyprMXWebViewContract.Presenter {
    return HyprMXWebViewPresenter(
      view,
      viewModelIdentifier = viewModelIdentifier,
      eventPublisher = WebViewPresentationEventPublisher(jsEngine, viewModelIdentifier),
      jsEngine = jsEngine,
      scope = coroutineScope,
    )
  }

  override fun makeHyprMXBrowserPresenter(
    view: HyprMXBrowserContract.View?,
    viewModelIdentifier: String,
  ): HyprMXBrowserContract.Presenter {
    var presenter = preloadedBrowserPresenter[viewModelIdentifier]
    return if (presenter != null) {
      preloadedBrowserPresenter.remove(viewModelIdentifier)
      presenter
    } else {
      presenter = HyprMXBrowserPresenter(
        view,
        viewModelIdentifier,
        jsEngine,
        coroutineScope,
      )
      preloadedBrowserPresenter[viewModelIdentifier] = presenter
      presenter
    }
  }
}

internal interface PresentationFactory {
  fun clearPreloadedWebView(viewModelIdentifier: String, destroy: Boolean)
  fun makeBrowserWebView(context: Context, viewModelIdentifier: String): HyprMXWebView
  fun makePreloadedWebView(context: Context, viewModelIdentifier: String): HyprMXWebView
  fun makeBannerPresenter(view: HyprMXBannerContract.View?, placementName: String): HyprMXBannerContract.Presenter
  fun makeWebViewPresenter(view: HyprMXWebViewContract.View, viewModelIdentifier: String): HyprMXWebViewContract.Presenter
  fun makeHyprMXBrowserPresenter(view: HyprMXBrowserContract.View?, viewModelIdentifier: String): HyprMXBrowserContract.Presenter
}
