package com.hyprmx.android.sdk.tracking

import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import java.util.Collections.synchronizedList
import java.util.Collections.synchronizedSet

/**
 * Delegates calls to tracking listeners.
 * Ensures duplicate events are not triggered and turns progress events into quartile events
 */
internal class VideoTrackingDelegate : VideoTrackingDelegateIf {

  private val mutex = Mutex()
  private val trackers: MutableList<VideoTracking> = synchronizedList(mutableListOf())
  private val timeSet: MutableSet<Long> = synchronizedSet(mutableSetOf())
  private var didVideoStart = false
  private var firstQuartile = false
  private var secondQuartile = false
  private var thirdQuartile = false
  private var didImpression = false
  private var didComplete = false
  private var didViewable = false
  private var didView = false
  private var didClose = false
  private var didSkip = false

  companion object {
    private const val FIRST_QUARTER_MARKER = 0.25f
    private const val MID_POINT_MARKER = 0.50f
    private const val THIRD_QUARTER_MARKER = 0.75f
  }

  override suspend fun addTracker(tracker: VideoTracking) {
    mutex.withLock {
      trackers.add(tracker)
    }
  }

  override suspend fun onProgress(duration: Long, currentPosition: Long) {
    mutex.withLock {
      if (duration > 0) {
        val currentPositionInSeconds = currentPosition / 1000
        val progressPercentage = currentPosition.toFloat() / duration.toFloat()

        if (!didVideoStart) {
          didVideoStart = true
          trackers.forEach { it.onStart() }
        }
        if (!firstQuartile && progressPercentage > FIRST_QUARTER_MARKER) {
          firstQuartile = true
          trackers.forEach { it.onFirstQuartile() }
        }
        if (!secondQuartile && progressPercentage > MID_POINT_MARKER) {
          secondQuartile = true
          trackers.forEach { it.onMidPoint() }
        }
        if (!thirdQuartile && progressPercentage > THIRD_QUARTER_MARKER) {
          thirdQuartile = true
          trackers.forEach { it.onThirdQuartile() }
        }
        if (!timeSet.contains(currentPositionInSeconds)) {
          timeSet.add(currentPositionInSeconds)
          trackers.forEach { it.onProgress(currentPositionInSeconds) }
        }
      }
    }
  }

  override suspend fun onComplete() {
    mutex.withLock {
      if (!didComplete) {
        didComplete = true
        trackers.forEach { it.onComplete() }
      }
    }
  }

  override suspend fun onImpression() {
    mutex.withLock {
      if (!didImpression) {
        didImpression = true
        trackers.forEach { it.onImpression() }
      }
    }
  }

  override suspend fun onClick() {
    mutex.withLock {
      trackers.forEach { it.onClick() }
    }
  }

  override suspend fun onView() {
    mutex.withLock {
      if (!didView) {
        didView = true
        trackers.forEach { it.onView() }
      }
    }
  }

  override suspend fun onClose() {
    mutex.withLock {
      if (!didClose) {
        didClose = true
        trackers.forEach { it.onClose() }
      }
    }
  }

  override suspend fun onPause() {
    mutex.withLock {
      trackers.forEach { it.onPause() }
    }
  }

  override suspend fun onResume() {
    mutex.withLock {
      trackers.forEach { it.onResume() }
    }
  }

  override suspend fun onSkip() {
    mutex.withLock {
      if (!didSkip) {
        didSkip = true
        trackers.forEach { it.onSkip() }
      }
    }
  }

  override suspend fun onNonViewable() {
    mutex.withLock {
      if (!didViewable) {
        didViewable = true
        trackers.forEach { it.onNonViewable() }
      }
    }
  }

  override suspend fun onViewable() {
    mutex.withLock {
      if (!didViewable) {
        didViewable = true
        trackers.forEach { it.onViewable() }
      }
    }
  }
}
