package com.hyprmx.android.sdk.utility

import androidx.annotation.VisibleForTesting
import kotlinx.coroutines.delay
import kotlin.math.pow

@VisibleForTesting
const val MAX_RETRY_COUNT = 3
private const val INITIAL_BACKOFF_MILLIS = 500

internal suspend fun <T> withBackoff(
  method: String,
  maxRetriesAttempts: Int = MAX_RETRY_COUNT,
  initialBackoffTimeInMilli: Int = INITIAL_BACKOFF_MILLIS,
  block: suspend () -> T,
): T {
  var retryCount = 0

  while (true) {
    try {
      return block()
    } catch (ex: Exception) {
      if (retryCount >= maxRetriesAttempts) {
        HyprMXLog.e("Aborted retrying $method method after failing $retryCount times.")
        throw ex
      }

      val backoffDelay = calculateExponentialBackoffDelay(retryCount, initialBackoffTimeInMilli)
      HyprMXLog.w("$method failed and has been delayed $backoffDelay millis before retrying ${retryCount + 1} times.")
      delay(backoffDelay)
      retryCount++
    }
  }
}

private fun calculateExponentialBackoffDelay(retryCount: Int, initialBackoffTimeInMilli: Int) =
  initialBackoffTimeInMilli * (2.0.pow(retryCount.toDouble())).toLong()
