package com.hyprmx.android.sdk.utility;

import android.app.Dialog;
import android.content.DialogInterface;
import android.os.Build;
import android.view.ViewTreeObserver;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

/**
 * Wrapper class for DialogInterface.OnClickListener to clear the reference to the listener
 * when the dialog window is detached. This fixes the issue of potential memory leaks on OS prior to
 * Android Lollipop when using dialogs.
 */
public final class DetachableClickListener implements DialogInterface.OnClickListener {

  private DialogInterface.OnClickListener delegateOrNull;

  private DetachableClickListener(DialogInterface.OnClickListener delegate) {
    this.delegateOrNull = delegate;
  }

  @NonNull
  public static DetachableClickListener wrap(DialogInterface.OnClickListener delegate) {
    return new DetachableClickListener(delegate);
  }

  @Override
  public void onClick(DialogInterface dialog, int which) {
    if (delegateOrNull != null) {
      delegateOrNull.onClick(dialog, which);
    }
  }

  /**
   * Nulls out listener when dialog is detached from window.
   * Must be set after Dialog.show() is called.
   *
   * @param dialog
   */
  @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
  public void clearOnDetach(Dialog dialog) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
      dialog.getWindow()
        .getDecorView()
        .getViewTreeObserver()
        .addOnWindowAttachListener(new ViewTreeObserver.OnWindowAttachListener() {
          @Override
          public void onWindowAttached() {
          }

          @Override
          public void onWindowDetached() {
            delegateOrNull = null;
          }
        });
    }
  }
}
