package com.hyprmx.android.sdk.utility

import android.content.Context
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import com.google.android.gms.common.GooglePlayServicesNotAvailableException
import com.hyprmx.android.R
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

object FetchGAIDImpl : FetchGaid {
  override suspend fun retrieveGaid(context: Context): GaidResult = withContext(Dispatchers.IO) {
    // Do not call this function from the main thread. Otherwise,
    // an IllegalStateException will be thrown.
    val adInfo = try {
      AdvertisingIdClient.getAdvertisingIdInfo(context)
    } catch (ex: GooglePlayServicesNotAvailableException) {
      HyprMXLog.d("Google Play Services not available on device.")
      null
    } catch (error: NoClassDefFoundError) {
      HyprMXLog.d("Google Play Services not linked.")
      null
    } catch (throwable: Throwable) {
      HyprMXLog.d("There was a problem retrieving GAID: $throwable ")
      null
    }
    return@withContext getResult(context, adInfo)
  }

  internal fun getResult(context: Context, adInfo: AdvertisingIdClient.Info?): GaidResult {
    return if (adInfo != null) {
      GaidResult.Success(adInfo.id ?: context.getString(R.string.hyprmx_gaid_fallback), adInfo.isLimitAdTrackingEnabled)
    } else {
      GaidResult.Failure
    }
  }
}

interface FetchGaid {
  suspend fun retrieveGaid(context: Context): GaidResult
}

sealed class GaidResult {
  class Success(val googleAdId: String, val limitAdTracking: Boolean) : GaidResult()
  object Failure : GaidResult()
}
