package com.hyprmx.android.sdk.utility

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import com.hyprmx.android.sdk.network.NetworkController
import com.hyprmx.android.sdk.network.NetworkResponse
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.BufferedOutputStream
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.lang.ref.SoftReference
import java.util.concurrent.ConcurrentHashMap

internal interface ImageCacheManagerIf {
  suspend fun retrieveBitmap(url: String): Result<Bitmap>
  suspend fun downloadAndCacheImage(url: String)
  suspend fun isMapped(url: String): Boolean
}

/**
 * This class provides cache for loaded HyprMX images, both in memory and on disk.
 */
internal class ImageCacheManager(
  private val context: Context,
  private val networkController: NetworkController,
  private val memoryCache: MutableMap<String, SoftReference<Bitmap>> = ConcurrentHashMap(),
  private val maxRetries: Int = 3,
) : ImageCacheManagerIf {

  private val inProgressDownloads: MutableSet<String> = HashSet()

  override suspend fun isMapped(url: String): Boolean = withContext(Dispatchers.IO) {
    memoryCache.contains(url) || openImageFileByUrl(url).exists()
  }

  private suspend fun openImageFileByUrl(url: String): File = withContext(Dispatchers.IO) {
    File(context.cacheDir, url.hashCode().toString())
  }

  /**
   * Retrieves the bitmap from cache.  If none found, returns null
   * @param url The url of the bitmap to retrieve
   */
  private suspend fun getBitmapFromCache(url: String): Bitmap? = withContext(Dispatchers.IO) {
    memoryCache[url]?.get() ?: getBitmapFromFileCache(url)
  }

  // Gets called on both the main thread and the background thread.
  // Thread safe to use since it doesn't touch shared data.
  private suspend fun getBitmapFromFileCache(url: String): Bitmap? = withContext(Dispatchers.IO) {
    BitmapFactory.decodeFile(openImageFileByUrl(url).absolutePath)
  }

  private suspend fun putBitmapToCaches(bitmap: Bitmap, urlString: String) =
    withContext(Dispatchers.IO) {
      memoryCache[urlString] = SoftReference(bitmap)

      try {
        val file = File(context.cacheDir, urlString.hashCode().toString())
        val bos = BufferedOutputStream(FileOutputStream(file), 65535)
        bitmap.compress(Bitmap.CompressFormat.PNG, 0, bos)
        bos.flush()
        bos.close()
      } catch (ioException: IOException) {
        HyprMXLog.e("Exception storing the image $urlString to disk", ioException)
      }
    }

  private suspend fun fetchBitmap(imageUrl: String): Bitmap? = withContext(Dispatchers.IO) {
    HyprMXLog.d("---fetchBitmap($imageUrl)")
    var bitmap: Bitmap? = null
    var retries = maxRetries
    do {
      val response = networkController.request(imageUrl) { stream ->
        stream.use { BitmapFactory.decodeStream(it) }
      }
      if (response is NetworkResponse.Success && response.isResponseCodeSuccessful()) {
        bitmap = response.value
      } else {
        HyprMXLog.e("error fetching bitmap $imageUrl")
      }
    } while (--retries > 0 && bitmap == null)
    bitmap
  }

  private suspend fun downloadAndCacheBitmap(imageUrl: String): Result<Bitmap> =
    withContext(Dispatchers.IO) {
      inProgressDownloads.add(imageUrl)
      val bitmap = fetchBitmap(imageUrl)
      bitmap?.let { putBitmapToCaches(bitmap, imageUrl) }
      inProgressDownloads.remove(imageUrl)
      return@withContext if (bitmap != null) {
        Result.Success(bitmap)
      } else {
        Result.Failure("Failed to download and cache image", 0)
      }
    }

  override suspend fun downloadAndCacheImage(url: String) {
    withContext(Dispatchers.IO) {
      when {
        inProgressDownloads.contains(url) -> HyprMXLog.d("Image $url download already in progress")
        isMapped(url) -> HyprMXLog.d("Image $url already cached")
        else -> downloadAndCacheBitmap(url)
      }
    }
  }

  override suspend fun retrieveBitmap(url: String): Result<Bitmap> = withContext(Dispatchers.IO) {
    HyprMXLog.d("fetching [$url]")
    if (!isMapped(url)) {
      return@withContext downloadAndCacheBitmap(url)
    }
    val bitmap = getBitmapFromCache(url)
    return@withContext if (bitmap == null) {
      Result.Failure("Failed to retrieve image from cache", 0)
    } else {
      Result.Success(bitmap)
    }
  }
}
