package com.hyprmx.android.sdk.utility

import android.Manifest
import android.annotation.TargetApi
import android.content.ContentValues
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.media.MediaScannerConnection
import android.os.Build
import android.os.Environment
import android.provider.MediaStore
import android.webkit.URLUtil
import android.widget.Toast
import androidx.annotation.VisibleForTesting
import com.hyprmx.android.R
import com.hyprmx.android.sdk.network.NetworkController
import com.hyprmx.android.sdk.network.NetworkResponse
import com.hyprmx.android.sdk.utility.StorePictureManager.FailureCode
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.File
import java.io.IOException
import java.text.SimpleDateFormat
import java.util.Date

/**
 * Stores the provided URI to the phone's gallery. If the URI is remote, the image
 * will be downloaded to external storage and then saved to the phone's download directory.
 */
internal class StorePictureManagerImpl(
  context: Context,
  toastHandlerHandler: ToastHandler = DefaultToastHandlerHandler(),
  private val networkController: NetworkController,
  private val saveImage: SaveImage = SaveImageManager(context),
) : StorePictureManager, ToastHandler by toastHandlerHandler, SaveImage by saveImage {

  /**
   * Request a picture to be downloaded and stored to disk
   */
  override suspend fun storePicture(url: String, context: Context): Result<String> =
    withContext(Dispatchers.IO) {
      if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q &&
        !isPermissionGranted(context, Manifest.permission.WRITE_EXTERNAL_STORAGE)
      ) {
        showToast(context, context.getString(R.string.hyprmx_unable_to_save_image))
        return@withContext Result.Failure(
          "Failed to download. No external storage permission",
          FailureCode.FAILED_TO_STORE_TO_DOWNLOAD.code,
        )
      }

      if (!url.isValidUrl() && !URLUtil.isFileUrl(url)) {
        HyprMXLog.e("Picture URI is invalid")
        showToast(context, context.getString(R.string.hyprmx_unable_to_save_image))
        return@withContext Result.Failure("Picture URI is invalid", FailureCode.INVALID_URI.code)
      }

      try {
        val response = networkController.request(url) { inputStream ->
          inputStream.use { BitmapFactory.decodeStream(inputStream) }
        }

        when (response) {
          is NetworkResponse.Success -> {
            if (response.value.width <= 0 && response.value.height <= 0) {
              return@withContext Result.Failure(
                "Picture failed to decode",
                FailureCode.FAILED_TO_DECODE.code,
              )
            }

            val fileName = getFileName(url)
            saveImage.saveImageToExternalStorage(fileName, response.value)
            showToast(context, context.getString(R.string.hyprmx_saved_image, fileName))
            return@withContext Result.Success("")
          }

          is NetworkResponse.Failure -> return@withContext Result.Failure(
            "Picture failed to download",
            FailureCode.FAILED_TO_STORE_TO_DISK.code,
          )
        }
      } catch (e: Exception) {
        HyprMXLog.e("Error making request to image url: " + e.message)
      }
      return@withContext Result.Failure(
        "Picture failed to download",
        FailureCode.FAILED_TO_STORE_TO_DOWNLOAD.code,
      )
    }

  @VisibleForTesting
  fun getFileName(url: String): String {
    var filename: String = URLUtil.guessFileName(url, null, null)
    filename =
      try {
        "${filename.substringBeforeLast('.')}-${SimpleDateFormat("yyyyMMdd_HHmmss").format(Date())}.${filename.substringAfterLast('.')}"
      } catch (exception: Exception) {
        filename
      }
    return filename
  }
}

internal interface ToastHandler {
  suspend fun showToast(context: Context, message: String)
}

internal class DefaultToastHandlerHandler : ToastHandler {
  override suspend fun showToast(context: Context, message: String) =
    withContext(Dispatchers.Main) {
      Toast.makeText(context, message, Toast.LENGTH_SHORT).show()
    }
}

internal interface SaveImage {
  suspend fun saveImageToExternalStorage(fileName: String, bitmap: Bitmap)
}

internal class SaveImageManager(private val context: Context) : SaveImage {
  override suspend fun saveImageToExternalStorage(fileName: String, bitmap: Bitmap) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
      saveImageAndroidQAndAbove(fileName, bitmap)
    } else {
      saveImagePreAndroidQ(fileName, bitmap)
    }
  }

  @TargetApi(Build.VERSION_CODES.Q)
  private fun saveImageAndroidQAndAbove(fileName: String, bitmap: Bitmap) {
    val file = File(fileName)
    val contentValues = ContentValues().apply {
      put(MediaStore.MediaColumns.DISPLAY_NAME, fileName)
      put(MediaStore.MediaColumns.MIME_TYPE, file.mimeType())
      put(MediaStore.MediaColumns.RELATIVE_PATH, Environment.DIRECTORY_DOWNLOADS)
    }

    val resolver = context.contentResolver
    val uri = resolver.insert(MediaStore.Downloads.EXTERNAL_CONTENT_URI, contentValues)

    if (uri != null) {
      resolver.openOutputStream(uri)?.use { stream ->
        bitmap.compress(Bitmap.CompressFormat.JPEG, 100, stream)

        stream.flush()
      }

      // Tell the media scanner about the new file so that it is immediately available to the user.
      MediaScannerConnection.scanFile(context, arrayOf(file.toString()), null) { path, _ ->
        HyprMXLog.d("Scanned media file: -> $path")
      }
    }
  }

  private suspend fun saveImagePreAndroidQ(fileName: String, bitmap: Bitmap) =
    withContext(Dispatchers.IO) {
// Get the external storage directory path
      val path = File(Environment.getExternalStorageDirectory(), Environment.DIRECTORY_DOWNLOADS)

      if (!path.exists()) {
        path.mkdirs()
      }

      try {
        val file = File(path, fileName)

        file.outputStream().use { stream ->
          bitmap.compress(Bitmap.CompressFormat.JPEG, 100, stream)

          stream.flush()
        }
      } catch (e: IOException) { // Catch the exception
        HyprMXLog.e("Exception when trying to store a picture", e)
        null
      }
    }
}
