@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.utility

import com.hyprmx.android.sdk.presentation.ArgumentKey.IS_MAIN_FRAME
import com.hyprmx.android.sdk.presentation.ArgumentKey.MIME_TYPE
import com.hyprmx.android.sdk.presentation.ArgumentKey.URL
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PublishingEvent.SHOULD_REDIRECT_URL
import com.hyprmx.android.sdk.presentation.PublishingEvent.URL_NAVIGATION_ATTEMPT
import com.hyprmx.android.sdk.presentation.PublishingEvent.WINDOW_OPEN_ATTEMPT
import com.hyprmx.android.sdk.utility.NAVIGATION_DECISION.NAVIGATION_ALLOWED
import com.hyprmx.android.sdk.utility.NAVIGATION_DECISION.NAVIGATION_BLOCKED
import com.hyprmx.android.sdk.utility.NAVIGATION_DECISION.NAVIGATION_DECISION_OPEN_OUTSIDE_APPLICATION
import com.hyprmx.android.sdk.utility.NAVIGATION_DECISION.NAVIGATION_REDIRECTED
import kotlinx.coroutines.CoroutineScope
import org.json.JSONException
import org.json.JSONObject

internal interface URLFilter {
  fun urlNavigationAttempt(url: String, isMainFrame: Boolean): NavigationDecision
  fun windowOpenAttempt(url: String): String?
  fun shouldRedirectURL(url: String, mimeType: String): NavigationDecision
}

internal class DefaultURLFilter(eventPublisher: PresentationEventPublisher, scope: CoroutineScope) :
  PresentationEventPublisher by eventPublisher,
  URLFilter,
  CoroutineScope by scope {
  override fun urlNavigationAttempt(url: String, isMainFrame: Boolean): NavigationDecision {
    val response = publishEvent(URL_NAVIGATION_ATTEMPT, mapOf(URL to url, IS_MAIN_FRAME to isMainFrame)) as String
    val decision = response.toNavigationDecision()
    HyprMXLog.d("urlNavigationAttempt returned with ${decision.id}")
    return decision
  }

  override fun windowOpenAttempt(url: String): String? {
    return publishEvent(WINDOW_OPEN_ATTEMPT, mapOf(URL to url)) as String?
  }

  override fun shouldRedirectURL(url: String, mimeType: String): NavigationDecision {
    val response = publishEvent(SHOULD_REDIRECT_URL, mapOf(URL to url, MIME_TYPE to mimeType)) as String
    val decision = response.toNavigationDecision()
    HyprMXLog.d("shouldRedirectURL returned with ${decision.id}")
    return decision
  }
}

internal sealed class NavigationDecision(val id: String) {
  object NavigationAllowed : NavigationDecision(NAVIGATION_ALLOWED)
  object NavigationBlocked : NavigationDecision(NAVIGATION_BLOCKED)
  object NavigationOpenOutsideApplication : NavigationDecision(NAVIGATION_DECISION_OPEN_OUTSIDE_APPLICATION)
  class NavigationRedirected(val url: String) : NavigationDecision(NAVIGATION_REDIRECTED)
}

internal fun String.toNavigationDecision(): NavigationDecision {
  return try {
    val jsonObject = JSONObject(this)
    val decision = jsonObject.getString("decision")
    val url = jsonObject.getStringOrNull(URL)
    when (decision) {
      NAVIGATION_ALLOWED -> NavigationDecision.NavigationAllowed
      NAVIGATION_BLOCKED -> NavigationDecision.NavigationBlocked
      NAVIGATION_DECISION_OPEN_OUTSIDE_APPLICATION -> NavigationDecision.NavigationOpenOutsideApplication
      NAVIGATION_REDIRECTED -> {
        if (url != null) {
          NavigationDecision.NavigationRedirected(url)
        } else {
          NavigationDecision.NavigationAllowed
        }
      }
      else -> NavigationDecision.NavigationAllowed
    }
  } catch (exception: JSONException) {
    NavigationDecision.NavigationAllowed
  }
}

internal object NAVIGATION_DECISION {
  const val NAVIGATION_ALLOWED = "NAVIGATION_ALLOWED"
  const val NAVIGATION_BLOCKED = "NAVIGATION_BLOCKED"
  const val NAVIGATION_DECISION_OPEN_OUTSIDE_APPLICATION = "OPEN_OUTSIDE_APPLICATION"
  const val NAVIGATION_REDIRECTED = "NAVIGATION_REDIRECTED"
}
