/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.documentdb.internal.directconnectivity;

import com.microsoft.azure.documentdb.ConsistencyLevel;
import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.internal.AuthorizationTokenProvider;
import com.microsoft.azure.documentdb.internal.DatabaseAccountConfigurationProvider;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.directconnectivity.BarrierRequestHelper;
import com.microsoft.azure.documentdb.internal.directconnectivity.ReadPrimaryResult;
import com.microsoft.azure.documentdb.internal.directconnectivity.ReadQuorumResult;
import com.microsoft.azure.documentdb.internal.directconnectivity.ReadQuorumResultKind;
import com.microsoft.azure.documentdb.internal.directconnectivity.StoreReadResult;
import com.microsoft.azure.documentdb.internal.directconnectivity.StoreReader;
import com.microsoft.azure.documentdb.internal.directconnectivity.StoreResponse;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class QuorumReader {
    private static final int MAX_NUMBER_OF_READ_BARRIER_RETRIES = 6;
    private static final int MAX_NUMBER_OF_READ_QUORUM_RETRIES = 6;
    private static final int DELAY_BETWEEN_READ_BARRIER_CALLS_IN_MS = 10;
    private static final int MAX_BARRIER_RETRIES_FOR_MULTI_REGION = 30;
    private static final int BARRIER_RETRY_INTERVAL_IN_MS_FOR_MULTIREGION = 30;
    private static final int MAX_SHORT_BARRIER_RETRIES_FOR_MULTI_REGION = 4;
    private static final int SHORT_BARRIER_RETRY_INTERVAL_IN_MS_FOR_MULTI_REGION = 10;
    private final Logger logger = LoggerFactory.getLogger(QuorumReader.class);
    private StoreReader storeReader;
    private AuthorizationTokenProvider authorizationTokenProvider;
    private final DatabaseAccountConfigurationProvider configurationProvider;

    public QuorumReader(StoreReader storeReader, AuthorizationTokenProvider authorizationTokenProvider, DatabaseAccountConfigurationProvider configurationProvider) {
        this.storeReader = storeReader;
        this.authorizationTokenProvider = authorizationTokenProvider;
        this.configurationProvider = configurationProvider;
    }

    StoreResponse readStrong(DocumentServiceRequest request, int quorumValue) throws DocumentClientException {
        int readQuorumRetry = 6;
        boolean shouldRetryOnSecondary = false;
        boolean hasPerformedReadFromPrimary = false;
        do {
            shouldRetryOnSecondary = false;
            ReadQuorumResult secondaryQuorumReadResult = this.readQuorum(request, quorumValue, false, ConsistencyLevel.Strong);
            switch (secondaryQuorumReadResult.getQuorumResult()) {
                case QuorumMet: {
                    return secondaryQuorumReadResult.getResponse();
                }
                case QuorumSelected: {
                    this.logger.debug("Couldn't converge on the LSN {} after primary read barrier with read quorum {} for strong read.", (Object)secondaryQuorumReadResult.getSelectedLsn(), (Object)quorumValue);
                    request.setQuorumSelectedLSN(secondaryQuorumReadResult.getSelectedLsn());
                    request.setQuorumSelectedStoreResponse(secondaryQuorumReadResult.getStoreReadResult());
                    request.setGlobalCommittedSelectedLSN(secondaryQuorumReadResult.getGlobalCommittedLSN());
                    break;
                }
                case QuorumNotSelected: {
                    if (hasPerformedReadFromPrimary) {
                        this.logger.warn("Primary read already attempted. Quorum couldn't be selected after retrying on secondaries.");
                        throw new DocumentClientException(410, "Primary read already attempted. Quorum couldn't be selected after retrying on secondaries.");
                    }
                    this.logger.debug("Quorum could not be selected with read quorum of {}", (Object)quorumValue);
                    ReadPrimaryResult response = this.readPrimary(request, quorumValue, ConsistencyLevel.Strong);
                    if (response.isSuccessful()) {
                        this.logger.debug("Primary read succeeded");
                        return response.getResponse();
                    }
                    if (response.isShouldRetryOnSecondary()) {
                        this.logger.debug("ReadPrimary did not succeed. Will retry on secondary.");
                        shouldRetryOnSecondary = true;
                        hasPerformedReadFromPrimary = true;
                        break;
                    }
                    this.logger.warn("Could not get successful response from ReadPrimary");
                    throw new DocumentClientException(410, "Could not get successful response from ReadPrimary");
                }
                default: {
                    this.logger.error("Unknown read quorum result {}", (Object)secondaryQuorumReadResult.getQuorumResult().toString());
                    throw new DocumentClientException(500, "Unknown read quorum result.");
                }
            }
        } while (--readQuorumRetry > 0 && shouldRetryOnSecondary);
        this.logger.warn("Could not complete read quorum with read quorum value of {}", (Object)quorumValue);
        throw new DocumentClientException(410, "Could not complete read quorum.");
    }

    private ReadPrimaryResult readPrimary(DocumentServiceRequest request, int readQuorum, ConsistencyLevel consistencyLevel) throws DocumentClientException {
        request.setForceAddressRefresh(false);
        StoreReadResult storeReadResult = this.storeReader.readPrimary(request, true, consistencyLevel);
        if (!storeReadResult.isValid()) {
            throw storeReadResult.getException();
        }
        if (storeReadResult.getCurrentReplicaSetSize() <= 0 || storeReadResult.getLSN() < 0L || storeReadResult.getQuorumAckedLSN() < 0L) {
            this.logger.warn("Invalid value received from response header. CurrentReplicaSetSize {}, StoreLSN {}, QuorumAckedLSN {}. Throwing gone exception", new Object[]{storeReadResult.getCurrentReplicaSetSize(), storeReadResult.getLSN(), storeReadResult.getQuorumAckedLSN()});
            throw new DocumentClientException(410, "Invalid value received from response header.");
        }
        if (storeReadResult.getCurrentReplicaSetSize() > readQuorum) {
            this.logger.debug("Unexpected response. Replica Set size is {} which is greater than min value {}", (Object)storeReadResult.getCurrentReplicaSetSize(), (Object)readQuorum);
            return new ReadPrimaryResult(false, true, null, request.getRequestChargeTracker());
        }
        return new ReadPrimaryResult(true, false, storeReadResult, request.getRequestChargeTracker());
    }

    private ReadQuorumResult readQuorum(DocumentServiceRequest request, int readQuorum, boolean includePrimary, ConsistencyLevel consistencyLevel) throws DocumentClientException {
        long maxLsn = 0L;
        long globalCommittedLSN = -1L;
        StoreReadResult highestLsnResult = null;
        if (request.getQuorumSelectedStoreResponse() == null) {
            List<StoreReadResult> responseResult = this.storeReader.readMultipleReplica(request, includePrimary, readQuorum, consistencyLevel);
            int responseCount = 0;
            for (StoreReadResult result : responseResult) {
                if (!result.isValid()) continue;
                ++responseCount;
            }
            if (responseCount < readQuorum) {
                return new ReadQuorumResult(ReadQuorumResultKind.QuorumNotSelected, -1L, -1L, null, request.getRequestChargeTracker());
            }
            boolean isGlobalStrongReadCandidate = this.configurationProvider.getStoreConsistencyPolicy() == ConsistencyLevel.Strong && (request.getOriginalRequestConsistencyLevel() == null || request.getOriginalRequestConsistencyLevel() == ConsistencyLevel.Strong);
            QuorumMetCheckResult isQuorumMetResult = this.isQuorumMet(responseResult, readQuorum, includePrimary, isGlobalStrongReadCandidate);
            maxLsn = isQuorumMetResult.readLSN;
            globalCommittedLSN = isQuorumMetResult.globalCommittedLSN;
            if (isQuorumMetResult.isQuorumMet) {
                return new ReadQuorumResult(ReadQuorumResultKind.QuorumMet, maxLsn, globalCommittedLSN, isQuorumMetResult.selectedResponse, request.getRequestChargeTracker());
            }
            highestLsnResult = isQuorumMetResult.selectedResponse;
        } else {
            this.logger.warn("wait to catch up max lsn");
            maxLsn = request.getQuorumSelectedLSN();
            globalCommittedLSN = request.getGlobalCommittedSelectedLSN();
            highestLsnResult = request.getQuorumSelectedStoreResponse();
        }
        this.logger.debug("Quorum is not met, sending barrier request to replicas");
        DocumentServiceRequest barrierRequest = BarrierRequestHelper.create(request, this.authorizationTokenProvider);
        if (this.waitForReadBarrier(barrierRequest, false, readQuorum, maxLsn, globalCommittedLSN, consistencyLevel)) {
            return new ReadQuorumResult(ReadQuorumResultKind.QuorumMet, maxLsn, globalCommittedLSN, highestLsnResult, request.getRequestChargeTracker());
        }
        this.logger.warn("Quorum selected with maxLsn {}", (Object)maxLsn);
        return new ReadQuorumResult(ReadQuorumResultKind.QuorumSelected, maxLsn, globalCommittedLSN, highestLsnResult, request.getRequestChargeTracker());
    }

    private QuorumMetCheckResult isQuorumMet(List<StoreReadResult> readResponses, int readQuorum, boolean isPrimaryIncluded, boolean isGlobalStrongRead) {
        long maxLsn = 0L;
        long minLsn = Long.MAX_VALUE;
        int validResponsesCount = 0;
        long numberOfReadRegions = 0L;
        long maxGlobalCommittedLSN = 0L;
        for (StoreReadResult readResponse : readResponses) {
            if (!readResponse.isValid()) continue;
            ++validResponsesCount;
            numberOfReadRegions = Math.max(numberOfReadRegions, readResponse.getNumberOfReadRegions());
            maxGlobalCommittedLSN = Math.max(maxGlobalCommittedLSN, readResponse.getGlobalCommittedLSN());
        }
        if (validResponsesCount == 0) {
            return new QuorumMetCheckResult(false, 0L, -1L, null);
        }
        boolean checkForGlobalStrong = isGlobalStrongRead && numberOfReadRegions > 0L;
        StoreReadResult selectedResponse = null;
        int replicaCountMaxLsn = 0;
        for (StoreReadResult response : readResponses) {
            if (!response.isValid()) continue;
            if (response.getLSN() == maxLsn) {
                ++replicaCountMaxLsn;
            } else if (response.getLSN() > maxLsn) {
                replicaCountMaxLsn = 1;
                maxLsn = response.getLSN();
                selectedResponse = response;
            }
            if (response.getLSN() >= minLsn) continue;
            minLsn = response.getLSN();
        }
        long readLsn = selectedResponse.getItemLSN() == -1L ? maxLsn : Math.min(selectedResponse.getItemLSN(), maxLsn);
        long globalCommittedLSN = checkForGlobalStrong ? readLsn : -1L;
        this.logger.info("QuorumReader: MaxLSN {} ReplicaCountMaxLSN {} bCheckGlobalStrong {} MaxGlobalCommittedLSN {} NumberOfReadRegions {} SelectedResponseItemLSN {}", new Object[]{maxLsn, replicaCountMaxLsn, checkForGlobalStrong, maxGlobalCommittedLSN, numberOfReadRegions, selectedResponse.getItemLSN()});
        boolean isQuorumMet = false;
        if (!(readLsn <= 0L || replicaCountMaxLsn < readQuorum || checkForGlobalStrong && maxGlobalCommittedLSN < maxLsn)) {
            isQuorumMet = true;
        }
        if (!(isQuorumMet || validResponsesCount < readQuorum || selectedResponse.getItemLSN() == -1L || minLsn == Long.MAX_VALUE || selectedResponse.getItemLSN() > minLsn || checkForGlobalStrong && selectedResponse.getItemLSN() > maxGlobalCommittedLSN)) {
            isQuorumMet = true;
        }
        return new QuorumMetCheckResult(isQuorumMet, readLsn, globalCommittedLSN, selectedResponse);
    }

    private boolean matchGlobalCommitted(List<StoreReadResult> responses, long lsn) {
        for (StoreReadResult result : responses) {
            if (result.getGlobalCommittedLSN() < lsn) continue;
            return true;
        }
        return false;
    }

    private boolean waitForReadBarrier(DocumentServiceRequest barrierRequest, boolean allowPrimary, int readQuorum, long readBarrierLsn, long globalCommittedLSN, ConsistencyLevel consistencyLevel) throws DocumentClientException {
        List<StoreReadResult> responses;
        int readBarrierRetryCount = 6;
        int readBarrierRetryCountMultiRegion = 30;
        do {
            barrierRequest.setForceAddressRefresh(false);
            responses = this.storeReader.readMultipleReplica(barrierRequest, allowPrimary, readQuorum, consistencyLevel);
            ArrayList<Long> responseLSNs = new ArrayList<Long>(responses.size());
            int validLsnCount = 0;
            for (StoreReadResult storeReadResult : responses) {
                if (storeReadResult.getLSN() >= readBarrierLsn) {
                    ++validLsnCount;
                }
                responseLSNs.add(storeReadResult.getLSN());
            }
            if (validLsnCount >= readQuorum && (globalCommittedLSN <= 0L || this.matchGlobalCommitted(responses, globalCommittedLSN))) {
                this.logger.debug("secondaries barrier requeest succeeded");
                return true;
            }
            this.logger.warn("Barrier request failed with validLsnCount {}, response LSNs <{}> and readQuorum {} with remaining retries {} and allow primary is {}", new Object[]{validLsnCount, StringUtils.join(responseLSNs, (char)','), readQuorum, readBarrierRetryCount, allowPrimary});
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                throw new IllegalStateException("Delay thread interrupted with exception: ", e);
            }
        } while (--readBarrierRetryCount > 0);
        if (globalCommittedLSN > 0L) {
            while (readBarrierRetryCountMultiRegion-- > 0) {
                responses = this.storeReader.readMultipleReplica(barrierRequest, allowPrimary, readQuorum, consistencyLevel);
                int validLsnCount = 0;
                for (StoreReadResult storeReadResult : responses) {
                    if (storeReadResult.getLSN() < readBarrierLsn) continue;
                    ++validLsnCount;
                }
                if (validLsnCount >= readQuorum && this.matchGlobalCommitted(responses, globalCommittedLSN)) {
                    return true;
                }
                try {
                    if (30 - readBarrierRetryCountMultiRegion > 4) {
                        Thread.sleep(30L);
                        continue;
                    }
                    Thread.sleep(10L);
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException("Delay thread interrupted with exception: ", e);
                }
            }
        }
        return false;
    }

    StoreResponse readBoundedStaleness(DocumentServiceRequest request, int readQuorumValue) throws DocumentClientException {
        int readQuorumRetry = 6;
        boolean shouldRetryOnSecondary = false;
        boolean hasPerformedReadFromPrimary = false;
        block5: do {
            this.logger.warn("remaining retries {}", (Object)readQuorumRetry);
            ReadQuorumResult secondaryQuorumReadResult = this.readQuorum(request, readQuorumValue, false, ConsistencyLevel.BoundedStaleness);
            shouldRetryOnSecondary = false;
            switch (secondaryQuorumReadResult.getQuorumResult()) {
                case QuorumMet: {
                    this.logger.debug("ReadQuorum successful");
                    return secondaryQuorumReadResult.getResponse();
                }
                case QuorumSelected: {
                    this.logger.warn("Could not converge on the LSN {} after read barrier with read quorum {}. Will not perform barrier call on Primary for BoundedStaleness", (Object)secondaryQuorumReadResult.getSelectedLsn(), (Object)readQuorumValue);
                    request.setQuorumSelectedStoreResponse(secondaryQuorumReadResult.getStoreReadResult());
                    request.setQuorumSelectedLSN(secondaryQuorumReadResult.getSelectedLsn());
                    break;
                }
                case QuorumNotSelected: {
                    if (hasPerformedReadFromPrimary) {
                        this.logger.warn("Primary read already attempted. Quorum could not be selected after retrying on secondaries.");
                        throw new DocumentClientException(410, "Primary read already attempted. Quorum could not be selected after retrying on secondaries.");
                    }
                    this.logger.debug("Quorum could not be selected with read quorum of {}", (Object)readQuorumValue);
                    ReadPrimaryResult response = this.readPrimary(request, readQuorumValue, ConsistencyLevel.BoundedStaleness);
                    if (response.isSuccessful() && response.isShouldRetryOnSecondary()) {
                        this.logger.error("PrimaryResult has both Successful and ShouldRetryOnSecondary flags set");
                        assert (false) : "PrimaryResult has both Successful and ShouldRetryOnSecondary flags set";
                        continue block5;
                    }
                    if (response.isSuccessful()) {
                        this.logger.debug("ReadPrimary successful");
                        return response.getResponse();
                    }
                    if (response.isShouldRetryOnSecondary()) {
                        shouldRetryOnSecondary = true;
                        this.logger.debug("ReadPrimary did not succeed. Will retry on secondary.");
                        hasPerformedReadFromPrimary = true;
                        break;
                    }
                    this.logger.warn("Could not get successful response from ReadPrimary");
                    throw new DocumentClientException(410, "Could not get successful response from ReadPrimary");
                }
                default: {
                    this.logger.warn("Unknown ReadQuorum result {}", (Object)secondaryQuorumReadResult.getQuorumResult().toString());
                    throw new DocumentClientException(500, "Unknown ReadQuorum");
                }
            }
        } while (--readQuorumRetry > 0 && shouldRetryOnSecondary);
        this.logger.error("Could not complete read quourm with read quorum value of {}", (Object)readQuorumValue);
        throw new DocumentClientException(410, String.format("Could not complete read quourm with read quorum value of %d", readQuorumValue));
    }

    private static class QuorumMetCheckResult {
        boolean isQuorumMet;
        long readLSN;
        long globalCommittedLSN;
        StoreReadResult selectedResponse;

        QuorumMetCheckResult(boolean isQuorumMet, long readLSN, long globalCommittedLSN, StoreReadResult selectedResponse) {
            this.isQuorumMet = isQuorumMet;
            this.readLSN = readLSN;
            this.globalCommittedLSN = globalCommittedLSN;
            this.selectedResponse = selectedResponse;
        }
    }
}

