package com.talpa.adsilence;

import android.content.Context;

import com.talpa.adsilence.data.DisplayMaterial;
import com.talpa.adsilence.ploy.DefaultAdSlotTime;
import com.talpa.adsilence.ploy.DefaultDisplayCondition;
import com.talpa.adsilence.ploy.DefaultDisplaySwitch;
import com.talpa.adsilence.ploy.DefaultLimitedCondition;
import com.talpa.adsilence.ploy.DefaultMaterialFactory;
import com.talpa.adsilence.ploy.DisplayCondition;
import com.talpa.adsilence.ploy.DisplaySwitch;
import com.talpa.adsilence.ploy.MaterialFactory;

import java.util.concurrent.Future;

import androidx.annotation.AnyThread;
import androidx.annotation.CheckResult;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

/**
 * 广告位 流量 引导流程 接口定义 <br/>
 *
 * <img src="https://s2.loli.net/2022/06/15/ypPqXGw7zI1LbAC.png" alt="广告位流量.png" style="zoom:0.5" />
 *
 * @see <a href="https://transsioner.feishu.cn/sheets/shtcnxnu6apcPbklqd743K7Jbng?sheet=qtpnJc" target="_blank">
 *     静默期 + 离线流量</a>
 *
 * @author 💎 Li Junchao
 */
public interface AdSlotTraffic extends AutoCloseable
{
    /**
     * 获取设备 在 <b>静默期</b> 或者 <b>非静默期</b> 的广告位 流量分配的展示物料, 您必须在 <b>工作线程</b> 调用此方法
     *
     * @return {@link DisplayMaterial}
     */
    @CheckResult
    @WorkerThread
    @Nullable
    DisplayMaterial getDisplayMaterialForAdSlot();

    /**
     * 获取设备 在 <b>静默期</b> 或者 <b>非静默期</b> 的广告位 流量分配的展示物料
     * <p>
     *     慎用此方法，您应该在一个应用层业务线程池里使用 {@link #getDisplayMaterialForAdSlot()},
     *     此方法只会简单的new Thread
     * </p>
     *
     * @return {@link Future}
     */
    @AnyThread
//    Future<DisplayMaterial> getAdSlotDisplayMaterial();

    /**
     * 广告位 流量 展示 功能的构造类,通过此类可重写各个部分的实现逻辑
     */
    final class Builder
    {
        final Context context;
        /** 物料 构造 Factory */
        MaterialFactory materialFactory;
        /** 展示开关 */
        DisplaySwitch displaySwitch;
        /** 展示条件 */
        DisplayCondition displayCondition;
        /** 广告位 时间 */
        AdSlotTime adSlotTime;

        /** 充值快捷方式Id */
        String chargeShortcutId;
        /** 浏览器 快捷方式Id */
        String browserShortcutId;

        boolean buildCalled;

        /**
         * 广告位 流量展示 的构建器
         *
         * <dl>
         *     <dt>使用默认实现:</dt>
         *     <dd>物料构造工厂：{@link DefaultMaterialFactory}</dd>
         *     <dd>流量位展示开关：{@link DefaultDisplaySwitch}</dd>
         *     <dd>展示条件:{@link DefaultDisplayCondition}、限定条件: {@link DefaultLimitedCondition}</dd>
         * </dl>
         *
         * @param context {@link Context}
         */
        public Builder(@NonNull Context context)
        {
            this(context, new DefaultMaterialFactory());
        }

        /**
         * 广告位 流量展示 的构建器
         *
         * @param context           {@link Context}
         * @param materialFactory   {@link MaterialFactory} 展示物料的构造工厂
         */
        public Builder(@NonNull Context context, @NonNull MaterialFactory materialFactory)
        {
//            this(context, materialFactory, new DefaultDisplayCondition(context));
            this(context, materialFactory, new DefaultDisplayCondition(context, new DefaultLimitedCondition()));
        }

        /**
         * 广告位 流量展示 的构建器
         *
         * @param context           {@link Context}
         * @param displaySwitch     {@link DisplaySwitch} 展示开关
         */
        public Builder(@NonNull Context context, @NonNull DisplaySwitch displaySwitch)
        {
            this(context, new DefaultMaterialFactory(), displaySwitch);
        }

        /**
         * 广告位 流量展示 的构建器
         *
         * @param context           {@link Context}
         * @param displayCondition  {@link DisplayCondition} 展示条件
         */
        public Builder(@NonNull Context context, @NonNull DisplayCondition displayCondition)
        {
            this(context, new DefaultMaterialFactory(), displayCondition);
        }

        /**
         * 广告位 流量展示 的构建器
         *
         * @param context           {@link Context}
         * @param materialFactory   {@link MaterialFactory} 展示物料的 构造工厂
         * @param displaySwitch     {@link DisplaySwitch} 展示 开关
         */
        public Builder(@NonNull Context context, @NonNull MaterialFactory materialFactory, @NonNull DisplaySwitch displaySwitch)
        {
            this(context, materialFactory, displaySwitch, new DefaultDisplayCondition(context));
        }

        /**
         * 广告位 流量展示 的构建器
         *
         * @param context           {@link Context}
         * @param materialFactory   {@link MaterialFactory} 物料构造工厂
         * @param displayCondition  {@link DisplayCondition} 展示条件
         */
        public Builder(@NonNull Context context, @NonNull MaterialFactory materialFactory, @NonNull DisplayCondition displayCondition)
        {
            this(context, materialFactory, new DefaultDisplaySwitch(), displayCondition);
        }

        /**
         * 广告位 流量展示 的构建器
         *
         * @param context           {@link Context}
         * @param materialFactory   {@link MaterialFactory} 物料构造工厂
         * @param displaySwitch     {@link DisplaySwitch} 展示 开关
         * @param displayCondition  {@link DisplayCondition} 展示条件
         */
        public Builder(@NonNull Context context, @NonNull MaterialFactory materialFactory,
                       @NonNull DisplaySwitch displaySwitch, @NonNull DisplayCondition displayCondition)
        {
            this(context, materialFactory, displaySwitch, displayCondition, new DefaultAdSlotTime());
        }

        /**
         * 广告位 流量展示 的构建器
         *
         * @param context           {@link Context}
         * @param materialFactory   {@link MaterialFactory} 物料构造工厂
         * @param displaySwitch     {@link DisplaySwitch} 展示 开关
         * @param displayCondition  {@link DisplayCondition} 展示条件
         * @param adSlotTime        {@link AdSlotTime} 用来判断 广告位 是在什么时期(静默期 or 非静默期)
         */
        public Builder(@NonNull Context context,
                       @NonNull MaterialFactory materialFactory,
                       @NonNull DisplaySwitch displaySwitch,
                       @NonNull DisplayCondition displayCondition,
                       @NonNull AdSlotTime adSlotTime)
        {
            this.context = context.getApplicationContext();
            this.materialFactory = materialFactory;
            this.displaySwitch = displaySwitch;
            this.displayCondition = displayCondition;
            this.adSlotTime = adSlotTime;
        }

        /**
         * 配置要展示 <b>物料</b> 的构造工厂
         *
         * @param materialFactory   {@link MaterialFactory} 物料构造工厂
         * @return {@link Builder}
         *
         * @see DefaultMaterialFactory
         */
        public Builder setMaterialFactory(@NonNull MaterialFactory materialFactory)
        {
            this.materialFactory = materialFactory;
            return this;
        }

        /**
         * 配置广告位要展示流量物料的 <b>开关 🔛</b>
         *
         * @param displaySwitch     {@link DisplaySwitch} 展示开关
         * @return {@link Builder}
         *
         * @see DefaultDisplaySwitch
         */
        public Builder setDisplaySwitch(@NonNull DisplaySwitch displaySwitch)
        {
            this.displaySwitch = displaySwitch;
            return this;
        }

        /**
         * 配置广告位要展示流量物料的 <b>展示条件 ❓</b>
         *
         * @param displayCondition  {@link DisplayCondition} 展示条件
         * @return {@link Builder}
         *
         * @see DefaultDisplayCondition
         */
        public Builder setDisplayCondition(@NonNull DisplayCondition displayCondition)
        {
            this.displayCondition = displayCondition;
            return this;
        }

        /**
         * 配置用来判断 广告位 是在什么时期(静默期 or 非静默期)的实现
         *
         * @param adSlotTime    {@link AdSlotTime} 用来判断 广告位 是在什么时期(静默期 or 非静默期)
         * @return {@link Builder}
         *
         * @see DefaultAdSlotTime
         */
        public Builder setAdSlotTime(@NonNull AdSlotTime adSlotTime)
        {
            this.adSlotTime = adSlotTime;
            return this;
        }

        /**
         * 配置 <b>充值 快捷方式Id</b> ,如果您之前有创建自己命名的 <b>充值快捷方式</b>
         *
         * @param chargeShortcutId 充值快捷方式Id
         * @return {@link Builder}
         */
        public Builder setChargeShortcutId(String chargeShortcutId)
        {
            this.chargeShortcutId = chargeShortcutId;
            return this;
        }

        /**
         * 配置 <b>浏览器 快捷方式Id</b> ,如果您之前有创建自己命名的 <b>浏览器快捷方式</b>
         *
         * @param browserShortcutId 浏览器快捷方式Id
         * @return  {@link Builder}
         */
        public Builder setBrowserShortcutId(String browserShortcutId)
        {
            this.browserShortcutId = browserShortcutId;
            return this;
        }

        /**
         * 构建一个广告位流量展示的实现类,此方法 仅能 <b>调用一次 1️⃣</b>
         *
         * @return {@link AdSlotTraffic}
         */
        public AdSlotTraffic build()
        {
            if (buildCalled) {
                throw new IllegalStateException("此构建Builder只能被调用构建一次");
            }

            buildCalled = true;

            return new AdSlotTrafficImpl(this);
        }

        private void checkState(boolean expression, Object errorMessage)
        {
            if (!expression) {
                throw new IllegalStateException(String.valueOf(errorMessage));
            }
        }
    }

    /**
     * 广告位-行为接口
     */
    interface AdSlotTime
    {
        /**
         * 是否在静默期
         *
         * @param context    {@link Context}
         * @return boolean - true:在 静默期
         */
        boolean inSilentPeriod(Context context);
    }
}
