package com.talpa.adsilence.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import androidx.annotation.FloatRange;
import androidx.annotation.NonNull;

/**
 * 随机权重 算法
 *
 * @author 💎 Li Junchao
 */
public final class WeightRandomChooser<T>
{
    /** 随机选取的元素列表 */
    private final List<T> items = new ArrayList<>();
    /** 权重比例范围 */
    private double[] weights;

    public WeightRandomChooser() {}

    public WeightRandomChooser(List<ItemWithWeight<T>> itemsWithWeight)
    {
        this.refresh(itemsWithWeight);
    }

    /**
     * 计算权重，初始化或者重新定义权重时使用
     */
    public void refresh(@NonNull List<ItemWithWeight<T>> itemsWithWeight)
    {
        items.clear();

        // 计算权重总和
        double originWeightSum = 0D;

        for (ItemWithWeight<T> itemWithWeight : itemsWithWeight) {

            double weight = itemWithWeight.getWeight();

            // ignore 权重为 0 的 item
            if (weight <= 0) {
                continue;
            }

            items.add(itemWithWeight.getItem());

            if (Double.isInfinite(weight)) {
                weight = 10000.0D;
            }

            if (Double.isNaN(weight)) {
                weight = 1.0D;
            }

            originWeightSum += weight;
        }

        // 计算每个item的实际权重比例
        double[] actualWeightRatios = new double[items.size()];

        int index = 0;

        for (ItemWithWeight<T> itemWithWeight : itemsWithWeight) {
            double weight = itemWithWeight.getWeight();

            if (weight <= 0) {
                continue;
            }

            actualWeightRatios[index++] = weight / originWeightSum;
        }

        // 计算每个item的权重范围
        weights = new double[items.size()];

        // 权重范围起始位置
        double weightRangeStartPos = 0;

        for (int i = 0; i < index; i++) {
            weights[i] = weightRangeStartPos + actualWeightRatios[i];
            weightRangeStartPos += actualWeightRatios[i];
        }
    }

    /**
     * 基于权重随机算法选择
     */
    @NonNull
    public T randomWithWeight()
    {
        double random = ThreadLocalRandom.current().nextDouble(0d, 1d);

        // 使用二分查找法快速定位随机数处于哪个区间
        int index = Arrays.binarySearch(weights, random);

        if (index < 0) {
            index = -index - 1;
        } else {
            return items.get(index);
        }

        if (index < weights.length && random < weights[index]) {
            return items.get(index);
        }

        // 通常不会走到这里，为了保证能得到正确的返回，这里随便返回一个
        return items.get(0);
    }

    public static class ItemWithWeight<T>
    {
        T item;
        double weight;

        public ItemWithWeight()
        {}

        public ItemWithWeight(@NonNull T item, @FloatRange(from = 1f) double weight)
        {
            this.item = item;
            this.weight = weight;
        }

        public T getItem()
        {
            return item;
        }

        public void setItem(@NonNull T item)
        {
            this.item = item;
        }

        public double getWeight()
        {
            return weight;
        }

        public void setWeight(@FloatRange(from = 1f) double weight)
        {
            this.weight = weight;
        }
    }
}
