package com.talpa.master.repository.net

import com.google.gson.Gson
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import io.reactivex.Flowable
import okhttp3.OkHttpClient
import retrofit2.Retrofit
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory
import retrofit2.converter.gson.GsonConverterFactory
import retrofit2.http.*

/**
 * 网络请求统一管理类
 * @author CY 2018/11/30
 */
object RequestHelper {

    private var mApiService: ApiService? = null

    private var retrofit: Retrofit? = null

    private fun builder(baseUrl: String): ApiService? {

        var builder = retrofit?.newBuilder()?.baseUrl(baseUrl)

        if (builder == null) {
            val client = OkHttpClient.Builder()
                .retryOnConnectionFailure(true)
                //.addNetworkInterceptor(StethoInterceptor())
                .build()
            builder = Retrofit.Builder()
                .baseUrl(baseUrl)
                .addConverterFactory(GsonConverterFactory.create())
                .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
                .client(client)
        }

        retrofit = builder!!.build()

        return retrofit!!.create(ApiService::class.java)
    }


    private fun apiService(host: HOST = HOST.GOOGLE): ApiService {

        val lastHost = retrofit?.baseUrl()?.host()

        when (host) {
            HOST.GOOGLE -> {

                if (lastHost == HOST.GOOGLE.host) {
                    return mApiService!!
                }

                mApiService =
                        builder(HOST.GOOGLE.host)

            }
            HOST.TAL -> {
                if (lastHost == HOST.TAL.host) {
                    return mApiService!!
                }

                mApiService =
                        builder(HOST.TAL.host)
            }
        }

        return mApiService!!
    }


    /**
     * 谷歌翻译
     */
    fun googleTranslate(q: String, targetLanguage: String): Flowable<JsonArray> {
        return apiService().googleTranslate(q, targetLanguage)
    }

    /**
     * 请求翻译语言
     * @param resource 翻译源,默认谷歌
     */
//    fun requestTranslateLanguage(resource: String = "google"): Flowable<Language> {
//
//        val paths = "translate-language"
//        val params = mapOf("resource" to resource)
//        return get(paths, params)
//    }


    /**
     * Get 请求包装，统一调用
     * 内联函数及reified 真实范型
     */
    private inline fun <reified T> get(paths: String, params: Map<String, String>): Flowable<T> =
        gson(apiService(HOST.TAL).get(paths, params))

    /**
     * Json to Bean
     */
    private inline fun <reified T> gson(flowable: Flowable<JsonObject>): Flowable<T> =
        flowable.map { Gson().fromJson(it, T::class.java) }

}

/**
 * Host
 */
enum class HOST(var nikeName: String, var host: String) {

    GOOGLE("GOOGLE", "https://translate.google.cn/"),
    TAL("TAL", "https://api.talpaos.com/")

}

interface ApiService {

    // var currentHost: HOST

    /**
     * 谷歌翻译
     * @param q 关键字，需要翻译的词语
     * @param targetLanguage 翻译目标语言：比如中文翻译为英文，英文为目标语言
     */
    @GET(value = "translate_a/single?client=gtx&dt=t&ie=UTF-8&oe=UTF-8&sl=auto")
    fun googleTranslate(@Query(value = "q") q: String, @Query(value = "tl") targetLanguage: String): Flowable<JsonArray>

    @GET(value = "{paths}")
    fun get(@Path(value = "paths") paths: String): Flowable<JsonObject>

    @GET(value = "{paths}")
    fun get(@Path(value = "paths") paths: String, @QueryMap params: Map<String, String>): Flowable<JsonObject>

    @POST(value = "{paths}")
    fun post(@Path(value = "paths") paths: String, @FieldMap params: Map<String, String>)

}

/*
https://api.talpaos.com/translate-language?resource=google

{
	"code": 200,
	"msg": "success",
	"data": {
		"resource": "google",
		"languages": [{
			"key": "en",
			"name": "English"
		}, {
			"key": "hi",
			"name": "Hindī \/ हिन्दी"
		}, {
			"key": "bn",
			"name": "bāṅlā \/ বাংলা"
		}, {
			"key": "te",
			"name": "తెలుగు"
		}, {
			"key": "mr",
			"name": "मराठी \/ Marāṭh"
		}, {
			"key": "ta",
			"name": "தமிழ் \/ Tamiḻ"
		}, {
			"key": "ur",
			"name": "اُردُو \/ Urdū"
		}, {
			"key": "kn",
			"name": "ಕನ್ನಡ \/ kannaḍa"
		}, {
			"key": "gu",
			"name": "ગુજરાતી \/ Gujarātī"
		}, {
			"key": "ml",
			"name": "മലയാളം \/ malayāḷaṁ"
		}, {
			"key": "pa",
			"name": "ਪੰਜਾਬੀ \/ Shahmukhi"
		}, {
			"key": "zh-CN",
			"name": "中文"
		}, {
			"key": "es",
			"name": "Español \/ Castellano"
		}, {
			"key": "ar",
			"name": "عربي\/عربى"
		}, {
			"key": "ms",
			"name": "Bahasa Melayu"
		}, {
			"key": "pt",
			"name": "Português"
		}, {
			"key": "ru",
			"name": "ру́сский язы́к"
		}, {
			"key": "fr",
			"name": "le français"
		}, {
			"key": "de",
			"name": "Deutsch"
		}, {
			"key": "ha",
			"name": "هَوُسَ"
		}, {
			"key": "ja",
			"name": "にほんご"
		}, {
			"key": "fa",
			"name": "فارسی \/ Fârsî"
		}, {
			"key": "sw",
			"name": "kiswahili"
		}, {
			"key": "jw",
			"name": "Basa Jawa"
		}, {
			"key": "th",
			"name": "ภาษาไทย"
		}, {
			"key": "id",
			"name": "Bahasa Indonesia"
		}, {
			"key": "tr",
			"name": "Türkçe"
		}, {
			"key": "ne",
			"name": "nepali"
		}, {
			"key": "tl",
			"name": "Tagalog"
		}, {
			"key": "vi",
			"name": "Tiếng Việt"
		}, {
			"key": "ko",
			"name": "한국어\/Korean"
		}, {
			"key": "it",
			"name": "Italiano\/Italian"
		}, {
			"key": "am",
			"name": "Amharic"
		}, {
			"key": "cs",
			"name": "Čeština"
		}, {
			"key": "hr",
			"name": "Croatia"
		}, {
			"key": "sv",
			"name": "svenska"
		}, {
			"key": "nl",
			"name": "Nederlands"
		}, {
			"key": "el",
			"name": "Ελληνικά\/Greek"
		}, {
			"key": "ro",
			"name": "limba română"
		}, {
			"key": "pl",
			"name": "Polski"
		}, {
			"key": "fi",
			"name": "suomalainen"
		}, {
			"key": "hu",
			"name": "magyar"
		}, {
			"key": "iw",
			"name": "עברית"
		}, {
			"key": "kk",
			"name": "Қазақша"
		}, {
			"key": "km",
			"name": "ភាសាខ្មែរ"
		}, {
			"key": "ky",
			"name": "Кыргыз тили"
		}, {
			"key": "my",
			"name": "မြန်မာ"
		}, {
			"key": "no",
			"name": "norsk språk"
		}, {
			"key": "ps",
			"name": "پښتو"
		}, {
			"key": "si",
			"name": "සිංහල"
		}, {
			"key": "so",
			"name": "Somali"
		}, {
			"key": "sq",
			"name": "shqiptar"
		}, {
			"key": "uk",
			"name": "Українська"
		}]
	}
}
 */