package com.talpa.master.translate

import android.app.Application
import android.os.Bundle
import android.os.Looper
import android.os.RemoteException
import android.widget.Toast
import androidx.lifecycle.MutableLiveData
import com.cy.sts.EventListener
import com.cy.sts.lang.RegionLang
import com.cy.sts.stt.SpeechHelper
import com.cy.sts.tts.TTSFactory
import com.cy.sts.tts.impl.BaiduTTS
import com.cy.sts.tts.impl.GoogleTTS
import com.hiservice.aidl.HiServiceInterface
import com.hiservice.connect.HiServiceConnection
import com.hiservice.connect.translate
import com.talpa.tengine.lang.LANG
import com.talpa.master.framework.RoomHelper
import com.talpa.master.ui.main.MainViewModel
import io.reactivex.Flowable
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.schedulers.Schedulers
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import okhttp3.ResponseBody
import java.io.File
import java.util.*

/**
 * Create by kgxl on 2021/4/13
 * 翻译中心
 */
open class TranslateService {

    // 初始化
    private lateinit var applicationContext: Application
    private val appKey = "e962487c7a9a"
    private val appSecret = "1d326d19-0b72-eed3-df57-5b7a9155bcd4"
    private val compositeDisposable: CompositeDisposable by lazy { CompositeDisposable() }
    private val bundle = Bundle()
    private var speechLang: LANG = LANG.ZH_CN
    private lateinit var eventListener: EventListener
    private val hiServiceConnection by lazy {
        HiServiceConnection.with(applicationContext).setAppKey(appKey = appKey).setAppSecret(appSecret = appSecret)
    }

    companion object {
        fun getInstance(): TranslateService {
            return TranslateServiceHolder.translateService
        }
    }

    object TranslateServiceHolder {
        val translateService = TranslateService()
    }

    fun setEventListener(eventListener: EventListener) {
        this.eventListener = eventListener
    }

    fun init(application: Application) {
        applicationContext = application
    }

    fun destroy() {
        hiServiceConnection.destroy()
        compositeDisposable.clear()
    }

    @Synchronized
    fun translateFromText(historyId: String, text: String, sourceLanguage: String? = null, targetLanguage: Locale, resultTxt: MutableLiveData<String>) {
        GlobalScope.launch(Dispatchers.IO) {
            speechLang = supportLanguage[targetLanguage]?.second ?: LANG.EN
            try {
                val result = hiServiceConnection.translate(text, sourceLanguage, speechLang.name)
                val state = result.getInt(HiServiceInterface.EXTRA_TRANSLATE_STATE)
                val translationText = result.getString(HiServiceInterface.EXTRA_TRANSLATE_SUCCESS_TRANSLATION)
                val errorText = result.getString(HiServiceInterface.EXTRA_TRANSLATE_FAILURE_MESSAGE)

                val db = Flowable.fromCallable {
                    RoomHelper.historyDao.queryHistory(historyId)
                }.subscribeOn(Schedulers.single()).map {
                    it?.let { history ->
                        when (it.display_type) {
                            MainViewModel.HistoryItemViewType.TARGET.value -> {
                                translationText?.let {
                                    history.source_result = it
                                }
                            }

                            MainViewModel.HistoryItemViewType.SOURCE.value -> {
                                translationText?.let {
                                    history.target_result = it
                                }
                            }
                        }

                        errorText?.let {
                            history.error_msg = errorText
                        }
                        if (state == HiServiceInterface.TRANSLATE_STATE_SUCCESS) {
                            history.result_type = MainViewModel.TransLateResultStatus.SUCCESS.value
                        }
                        RoomHelper.historyDao.updateStarModel(history)
                    }
                    println("speechLang =>${speechLang.name}")
                    Lang(translationText?:"", speechLang)
                }.flatMap(::ttsImpl).map { file ->
                    val h = RoomHelper.historyDao.queryHistory(historyId)
                    h?.let {
                        h.voice_path = file.absolutePath
                        RoomHelper.historyDao.updateStarModel(h)
                    }
                    file
                }.doOnError {
                    it.printStackTrace()
                }.subscribe {
                    eventListener.onEvent(SpeechHelper.CALLBACK_TTS_RESULT, bundle.apply {
                        putString(SpeechHelper.EXTRA_TTS_RESULT, it.path)
                    })
                }
                resultTxt.postValue(translationText)
                compositeDisposable.add(db)
                println("state = $state resultText = $translationText $errorText")
            } catch (e: RemoteException) {
                e.printStackTrace()
            }catch (e:Exception){
                Looper.prepare()
                Toast.makeText(applicationContext,e.message.toString(),Toast.LENGTH_SHORT).show()
            }
        }
    }

//    fun  translatePlayFromText(text: String, targetLanguage : String? = null){
//        try {
//            val language = targetLanguage ?: Locale.ENGLISH.language
//            val speakResult = hiServiceConnection.speak(text = text, language = language)
//
//        } catch (e: Exception) {
//            e.printStackTrace()
//
//        }
//    }

    /**
     * 单纯的翻译文本
     */
    @Synchronized
    fun translateFromText(text: String, sourceLanguage: String? = null, targetLanguage: Locale, resultTxt: MutableLiveData<String>) {
        GlobalScope.launch(Dispatchers.IO) {
            speechLang = supportLanguage[targetLanguage]?.second ?: LANG.EN
            if(sourceLanguage==speechLang.name){
                resultTxt.postValue(text)
                return@launch
            }
            try {
                val result = hiServiceConnection.translate(text, sourceLanguage, speechLang.name)
                val state = result.getInt(HiServiceInterface.EXTRA_TRANSLATE_STATE)
                val translationText = result.getString(HiServiceInterface.EXTRA_TRANSLATE_SUCCESS_TRANSLATION)
                val errorText = result.getString(HiServiceInterface.EXTRA_TRANSLATE_FAILURE_MESSAGE)
                resultTxt.postValue(translationText)
                println("state = $state resultText = $translationText $errorText")
            } catch (e: RemoteException) {
                e.printStackTrace()
            }catch (e:Exception){
                e.printStackTrace()
            }
        }
    }

    /**
     * 存放到本地
     */
    protected fun ttsImpl(langBean: Lang): Flowable<File> {

        val list = arrayOf(GoogleTTS(applicationContext), BaiduTTS(applicationContext))

        val lang: LANG = langBean.textSource

        val supportList = list.filter { it.isSupportLanguage(lang) }

        var flowable: Flowable<File>? = null
        for (tts in supportList) {
            val ttsFlow = getTTSFlowable(tts, langBean.text, lang)
            flowable = flowable?.onErrorResumeNext(ttsFlow) ?: ttsFlow
        }

        return flowable?.onErrorReturn { File("") } ?: Flowable.fromCallable { File("") }

    }

    private fun getTTSFlowable(tts: TTSFactory, text: String, textSpeechSourceLanguage: LANG): Flowable<File> {

        val textSpeechSourceLanguage: LANG = textSpeechSourceLanguage

        return when (tts.javaClass.name) {
            GoogleTTS::class.java.name -> {

                //val googleTk = InputStreamReader(builder.context.resources.assets.open("tk/Google.js"))
                val googleTTS = (tts as GoogleTTS).apply {
                    //tkReader = googleTk
                    setFormData(textSpeechSourceLanguage, text)
                }
                googleTTS.execute().flatMap { res: ResponseBody -> googleTTS.parse(res) }
            }
            else -> {
                val baiduTTS = (tts as BaiduTTS) //BaiduTTS(context!!.applicationContext)
                baiduTTS.setFormData(textSpeechSourceLanguage, text)
                baiduTTS.execute().flatMap { res: ResponseBody -> baiduTTS.parse(res) }
            }
        }
    }

    data class Lang(val text: String, val textSource: LANG)

    val supportLanguage = hashMapOf<Locale, Pair<RegionLang, LANG>>().apply {
        this[Locale.forLanguageTag("bn-IN")] = RegionLang.BN_IN to LANG.BN
        this[Locale.forLanguageTag("hi-IN")] = RegionLang.HI_IN to LANG.HI
        this[Locale.forLanguageTag("en-IN")] = RegionLang.EN_IN to LANG.EN
        this[Locale.forLanguageTag("en-US-POSIX")] = RegionLang.EN_US to LANG.EN
        this[Locale.forLanguageTag("en-GB")] = RegionLang.EN_GB to LANG.EN
        this[Locale.forLanguageTag("fr-FR")] = RegionLang.FR_FR to LANG.FR
        this[Locale.forLanguageTag("es-ES")] = RegionLang.ES_EC to LANG.ES
        this[Locale.forLanguageTag("ar-EG")] = RegionLang.AR_EG to LANG.AR
        this[Locale.forLanguageTag("ar-SA")] = RegionLang.AR_SA to LANG.AR
        this[Locale.forLanguageTag("ar-AE")] = RegionLang.AR_AE to LANG.AR
        this[Locale.forLanguageTag("th-TH")] = RegionLang.TH_TH to LANG.TH
        this[Locale.forLanguageTag("ko-KR")] = RegionLang.KO_KR to LANG.KO
        this[Locale.forLanguageTag("zh-Hans-CN")] = RegionLang.CMN_HANS_CN to LANG.ZH_CN
        this[Locale.forLanguageTag("sw-TZ")] = RegionLang.SW_TZ to LANG.SW
        this[Locale.forLanguageTag("sw-KE")] = RegionLang.SW_KE to LANG.SW
        this[Locale.forLanguageTag("pt-PT")] = RegionLang.PT_PT to LANG.PT
        this[Locale.forLanguageTag("am")] = RegionLang.AM_ET to LANG.AM
        this[Locale.forLanguageTag("yo")] = RegionLang.EN_NG to LANG.EN
        this[Locale.forLanguageTag("ha")] = RegionLang.EN_NG to LANG.EN
        this[Locale.forLanguageTag("ja-JP")] = RegionLang.JA_JP to LANG.JA
    }
}