package com.talpa.master.ui.history

import android.app.Application
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.media.MediaPlayer
import android.view.View
import android.widget.Toast
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import com.google.android.material.snackbar.BaseTransientBottomBar
import com.google.android.material.snackbar.Snackbar
import com.talpa.master.R
import com.talpa.master.common.MediaPlayerHelper
import com.talpa.master.ex.RxEx
import com.talpa.master.framework.RoomHelper
import com.talpa.master.repository.room.model.History
import com.talpa.master.repository.room.dao.HistoryDao
import com.talpa.master.ui.main.MainViewModel
import com.talpa.master.framework.BaseViewModel
import io.reactivex.Flowable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers

/**
 * Create by kgxl on 2021/4/13
 */
class HistoryViewModel(val application: Application, val dao: HistoryDao) : BaseViewModel(), MediaPlayer.OnCompletionListener, MediaPlayer.OnErrorListener {
    val historys = MutableLiveData<List<History>>()
    val isSource = MutableLiveData<Boolean>()
    var speechPath: String = ""
    val sourceBackground =
        arrayListOf(R.drawable.shape_bg_translate_source_blue, R.drawable.shape_bg_translate_source_green, R.drawable.shape_bg_translate_source_orange, R.drawable.shape_bg_translate_source_yellow)
    val targetBackground =
        arrayListOf(R.drawable.shape_bg_translate_target_blue, R.drawable.shape_bg_translate_target_green, R.drawable.shape_bg_translate_target_orange, R.drawable.shape_bg_translate_target_yellow)

    companion object {
        fun factory(application: Application) = object : ViewModelProvider.Factory {
            override fun <T : ViewModel?> create(modelClass: Class<T>): T {
                return HistoryViewModel(application, RoomHelper.historyDao) as T
            }
        }
    }

    fun selectHistory() {
        dao.queryLatestHistory().compose(RxEx.io4mainFlowable()).subscribe {
            historys.postValue(it)
        }
    }

    fun selectHistoryByMark() {
        dao.queryAllHistoryByMark().compose(RxEx.io4mainFlowable()).subscribe {
            historys.postValue(it)
        }
    }

    fun copyText(history: History, view: View) {
        when (history.result_type) {
            MainViewModel.TransLateResultStatus.SUCCESS.value -> {
                when (history.display_type) {
                    MainViewModel.HistoryItemViewType.SOURCE.value -> {
                        copy(view, history.target_result)
                    }
                    MainViewModel.HistoryItemViewType.TARGET.value -> {
                        copy(view, history.source_result)
                    }
                }
            }
            else -> {
                val snackBar = Snackbar.make(view, R.string.main_status_net_error, Snackbar.LENGTH_SHORT)
                snackBar.animationMode = BaseTransientBottomBar.ANIMATION_MODE_FADE
                snackBar.show()
            }
        }
    }

    /**
     * 复制译文
     */
    fun copy(view: View, text: String) {

        val context = view.context

        val cm = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        // 创建普通字符型ClipData
        val mClipData = ClipData.newPlainText("Label", text)
        cm.setPrimaryClip(mClipData)

        //SnackBar
        val snackBar = Snackbar.make(view, R.string.text_copy_complete, Snackbar.LENGTH_SHORT)
        snackBar.animationMode = BaseTransientBottomBar.ANIMATION_MODE_FADE
        snackBar.show()
    }

    fun starMark(history: History, view: View) {
        history.isStartMarker = !history.isStartMarker
        Flowable.fromCallable {
            RoomHelper.historyDao.updateStarModel(history)
        }.subscribeOn(Schedulers.io()).observeOn(AndroidSchedulers.mainThread()).subscribe {
            if (history.isStartMarker) {
                view.setBackgroundResource(R.drawable.ic_collected)
            } else {
                view.setBackgroundResource(R.drawable.ic_collect)
            }
        }
    }

    fun playTranslateVoice(v: View) {
        if (!speechPath.isNullOrEmpty()) {
            stopTranslateVoice()
            //            playState.value = MainViewModel.PlayState.Playing
            MediaPlayerHelper.playMusic(application, speechPath, this, this)
        } else {
            val snackBar = Snackbar.make(v, R.string.read_fail, Snackbar.LENGTH_SHORT)
            snackBar.animationMode = BaseTransientBottomBar.ANIMATION_MODE_FADE
            snackBar.show()
        }
    }

    private fun stopTranslateVoice() {
        speechPath?.let {
            //            playState.value = MainViewModel.PlayState.Enabled
            MediaPlayerHelper.stopMusic()
        }
    }

    override fun onError(mp: MediaPlayer?, what: Int, extra: Int): Boolean {
        Toast.makeText(application, "player error $what", Toast.LENGTH_SHORT).show()
        return true
    }

    override fun onCompletion(mp: MediaPlayer?) {
        //        playState.value = MainViewModel.PlayState.Enabled
    }

}