package com.talpa.master.ui.main

import android.Manifest
import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.ConnectivityManager
import android.net.LinkProperties
import android.net.Network
import android.net.NetworkCapabilities
import android.os.Build
import android.os.Bundle
import android.os.VibrationEffect
import android.os.Vibrator
import android.preference.PreferenceManager
import android.view.View
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProviders
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.cy.sts.EventListener
import com.cy.sts.STSEngine
import com.cy.sts.common.Logcat
import com.cy.sts.lang.RegionLang
import com.cy.sts.stt.SpeechHelper
import com.google.android.material.snackbar.Snackbar
import com.talpa.tengine.Trans
import com.talpa.tengine.lang.LANG
import com.talpa.master.R
import com.talpa.master.common.Constant
import com.talpa.master.databinding.ActivityMainBinding
import com.talpa.master.ex.RxEx
import com.talpa.master.framework.RoomHelper
import com.talpa.master.repository.*
import com.talpa.master.repository.room.model.History
import com.talpa.master.translate.TranslateService
import com.talpa.master.ui.history.HistoryActivity
import com.talpa.master.ui.main.fragment.ChoiceLanguageFragment
import com.talpa.master.ui.main.fragment.ChoiceLanguageListFragment
import com.talpa.master.ui.main.fragment.PermissionFragment
import com.talpa.master.ui.main.fragment.TranslateResultFragment
import com.talpa.master.ui.mark.StarMarkActivity
import com.talpa.master.utils.SharedPreferenceHelper
import com.talpa.master.widget.RecordTextView
import com.talpa.master.databinding.DataBindingActivity
import com.talpa.master.framework.checkPermission
import com.talpa.master.utils.RoomDataUtil
import io.reactivex.Flowable
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import java.io.File
import java.util.*
import java.util.concurrent.TimeUnit

class CommunicationActivity : DataBindingActivity<ActivityMainBinding>(), View.OnClickListener,
    RecordTextView.OnRecordListener, EventListener {
    private val viewModel: MainViewModel by lazy {
        ViewModelProviders.of(this, MainViewModel.factory(application))
            .get(MainViewModel::class.java)
    }

    private val historyDao by lazy {
        RoomHelper.historyDao
    }

    private val stsEngine: STSEngine by lazy {
        STSEngine.Builder(context = application, speechLanguage = RegionLang.CMN_HANS_CN, translateSourceLanguage = LANG.ZH_CN, translateTargetLanguage = LANG.EN, textSpeechSourceLanguage = LANG.EN, eventListener = this)
            .build()
    }

    private val choiceLanguageFragment by lazy { ChoiceLanguageFragment.getInstance() }
    private val choiceLanguageListFragment by lazy { ChoiceLanguageListFragment.getInstance(viewModel) }
    private var isLongPress: Boolean = false

    override fun getContentView(): Int = R.layout.activity_main

    override fun onResume() {
        super.onResume()
        observeNetWork()
    }

    override fun onPause() {
        super.onPause()
        removeObserveNetWork()
        stsEngine.cancelSpeech()
        viewModel.stopTranslateVoice()
    }

    override fun initView() {
        isFirstOpen()
    }

    override fun init() {
        mBinding.viewModel = viewModel
        viewModel.setLanguage()
        translateTips()
        initObserver()
        initListener()
        registerBroadcast()
    }

    private fun translateTips() {
        val sourceLocale = getSourceLocale(applicationContext)
        val targetLocale = getTargetLocale(applicationContext)
        TranslateService.getInstance()
            .translateFromText("翻译失败",sourceLanguage = "ZH_CN", targetLanguage = sourceLocale, resultTxt = viewModel.sourceTranslateError)
        TranslateService.getInstance()
            .translateFromText("翻译失败", sourceLanguage = "ZH_CN",targetLanguage = targetLocale, resultTxt = viewModel.targetTranslateError)
//        TranslateService.getInstance()
//            .translateFromText("你好",sourceLanguage = "ZH_CN", targetLanguage = sourceLocale, resultTxt = viewModel.sourceTranslateHi)
//        TranslateService.getInstance()
//            .translateFromText("你好", sourceLanguage = "ZH_CN",targetLanguage = targetLocale, resultTxt = viewModel.targetTranslateHi)
    }

    private fun initObserver() {
        viewModel.hideDialog.observe(this, {
            if (null != choiceLanguageListFragment && choiceLanguageListFragment.isAdded) choiceLanguageListFragment.dismissAllowingStateLoss()
        })
        viewModel.sourceLang.observe(this, {
            TranslateService.getInstance()
                .translateFromText("翻译失败", targetLanguage = Locale.forLanguageTag(it.key), resultTxt = viewModel.sourceTranslateError)
            saveSourceLocale(this, it.key)
        })
        viewModel.targetLang.observe(this, {
            TranslateService.getInstance()
                .translateFromText("翻译失败", targetLanguage = Locale.forLanguageTag(it.key), resultTxt = viewModel.targetTranslateError)
            saveTargetLocale(this, it.key)
        })
    }

    // 隐藏结果页
    private fun hideResultUi() {
        viewModel.isShowResult.postValue(false)
        mBinding.tvHistory.visibility = View.GONE
    }

    @SuppressLint("CheckResult")
    private fun showResultUi() {
        viewModel.isShowResult.postValue(true)
        mBinding.tvMainResultStart.setBackgroundResource(R.drawable.ic_collect)
        RoomHelper.historyDao.queryHistorySize().compose(RxEx.io4mainFlowable()).subscribe {
            if (it > 1) {
                mBinding.tvHistory.visibility = View.VISIBLE
            } else {
                mBinding.tvHistory.visibility = View.GONE
            }
        }

        mBinding.tvMainResultText.text = getString(R.string.main_status_translating)
        mBinding.tvMainResultSource.text = "..."
        mBinding.tvSourceLanguage.visibility = View.GONE
        mBinding.tvTargetLanguage.visibility = View.GONE
        viewModel.isSource.postValue(currentSpeech == SpeechType.SOURCE)
    }


    private fun initListener() {
        mBinding.tvMainLanguageHeSay.setOnClickListener(this)
        mBinding.tvYouSayLanguage.setOnClickListener(this)
        mBinding.tvHeSayLanguage.setOnClickListener(this)
        mBinding.tvMainLanguageYouSay.setOnRecordListener(this)
        mBinding.tvMainLanguageHeSay.setOnRecordListener(this)
        mBinding.tvMainResultCopy.setOnClickListener(this)
        mBinding.tvMainResultStart.setOnClickListener(this)
        mBinding.tvMainResultText.setOnClickListener(this)
        mBinding.tvMainResultVoice.setOnClickListener(this)
        mBinding.tvHistory.setOnClickListener(this)
        mBinding.tvMainMark.setOnClickListener(this)
        TranslateService.getInstance().setEventListener(this)
    }

    private fun isFirstOpen() {
        if (SharedPreferenceHelper.getBooleanValue(SharedPreferenceHelper.IS_FIRST_OPEN, true)) {
            choiceLanguageFragment.show(supportFragmentManager, ChoiceLanguageFragment.TAG)
        }
    }


    /**
     * 他说、我说、朗读、复制、星标
     */
    override fun onClick(v: View) {
        when (v.id) {

            //朗读
            R.id.tv_main_result_voice -> {
                viewModel.playTranslateVoice(v)
            }
            // 复制
            R.id.tv_main_result_copy -> {
                Flowable.fromCallable {
                    RoomHelper.historyDao.queryLastHistory()
                }.subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread()).subscribe {
                        copyText(it, v)
                    }
            }
            // 星标
            R.id.tv_main_result_start -> {
                Flowable.fromCallable {
                    RoomHelper.historyDao.queryLastHistory()
                }.subscribeOn(Schedulers.io()).map {
                    it.isStartMarker = !it.isStartMarker
                    it
                }.doOnError {
                    showSnackBar(mBinding.tlMain, "star Failure")
                }.subscribe {
                    println("updateStarModel")
                    RoomHelper.historyDao.updateStarModel(it)
                }
                val b = (v.getTag(R.id.view_star_status) ?: false) as Boolean
                if (!b) {
                    v.setTag(R.id.view_star_status, true)
                    v.setBackgroundResource(R.drawable.ic_collected)
                } else {
                    v.setTag(R.id.view_star_status, false)
                    v.setBackgroundResource(R.drawable.ic_collect)
                }
            }
            R.id.tv_main_mark -> {
                startActivity(Intent(MainActivity@ this, StarMarkActivity::class.java))
            }

            // 点击我说选择语言列表
            R.id.tv_you_say_language -> {
                viewModel.currentLangType = MainViewModel.LanguageType.SOURCE
                choiceLanguageListFragment.show(supportFragmentManager, ChoiceLanguageListFragment.TAG)
            }
            R.id.tv_language_you_say_arrow -> {
                viewModel.currentLangType = MainViewModel.LanguageType.SOURCE
                choiceLanguageListFragment.show(supportFragmentManager, ChoiceLanguageListFragment.TAG)
            }

            // 点击他说选择语言列表
            R.id.tv_he_say_language -> {
                viewModel.currentLangType = MainViewModel.LanguageType.TARGET
                choiceLanguageListFragment.show(supportFragmentManager, ChoiceLanguageListFragment.TAG)
            }
            R.id.tv_history -> {
                startActivity(Intent(MainActivity@ this, HistoryActivity::class.java))
            }
            R.id.tv_main_result_text -> {
                //点击翻译结果
                Flowable.fromCallable {
                    RoomHelper.historyDao.queryLastHistory()
                }.subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread()).subscribe {
                        TranslateResultFragment.getInstance(it.id)
                            .show(supportFragmentManager, TranslateResultFragment.TAG)
                    }
            }
        }
    }


    /**
     * 开始录音
     */
    private fun startVoice() {
        hideResultUi()
        showListenerUi()
        viewModel.stopTranslateVoice()
        //开始录音
        stsEngine.cancelSpeech()
        //setLanguage()
        stsEngine.startSpeech()
    }

    private fun stopRecordUi() {
        composite.clear()
        setSpeechState()
        showResultUi()
    }

    private fun errorRecordUi() {
        composite.clear()
        setSpeechState()
        hideResultUi()
        if (currentSpeech == SpeechType.SOURCE) {
            mBinding.tvSourceLanguage.visibility = View.VISIBLE
            mBinding.tvTargetLanguage.visibility = View.GONE
            mBinding.tvSourceLanguage.text = viewModel.sourceTranslateError.value
        } else {
            mBinding.tvSourceLanguage.visibility = View.GONE
            mBinding.tvTargetLanguage.visibility = View.VISIBLE
            mBinding.tvTargetLanguage.text = viewModel.targetTranslateError.value
        }
    }

    private fun setSpeechState(isRecording: Boolean = false) {
        mBinding.tvMainLanguageYouSay.setTag(R.id.view_speech_status, isRecording)
        mBinding.tvMainLanguageHeSay.setTag(R.id.view_speech_status, isRecording)
    }

    /**
     * 正在听
     */
    private fun showListenerUi() {
        val subscribe = Observable.interval(1, TimeUnit.SECONDS).take(Constant.LISTENER_TIME)
            .observeOn(AndroidSchedulers.mainThread()).doOnSubscribe {
                if (currentSpeech == SpeechType.SOURCE) {
                    mBinding.tvTargetLanguage.visibility = View.GONE
                    mBinding.tvSourceLanguage.visibility = View.VISIBLE
                    mBinding.tvSourceLanguage.text =
                        "${getString(R.string.main_status_listening)}...(${Constant.LISTENER_TIME})"
                } else {
                    mBinding.tvSourceLanguage.visibility = View.GONE
                    mBinding.tvTargetLanguage.visibility = View.VISIBLE
                    mBinding.tvTargetLanguage.text =
                        "${getString(R.string.main_status_listening)}...(${Constant.LISTENER_TIME})"
                }
            }.doOnComplete {
                //倒计时结束
                println("倒计时结束")
                setSpeechState()
                showResultUi()
                mBinding.tvSourceLanguage.visibility = View.GONE
                mBinding.tvTargetLanguage.visibility = View.GONE
            }.doOnError {
                setSpeechState()
                it.printStackTrace()
            }.subscribe {
                if (currentSpeech == SpeechType.SOURCE) {
                    mBinding.tvSourceLanguage.visibility = View.VISIBLE
                    mBinding.tvTargetLanguage.visibility = View.GONE
                    mBinding.tvSourceLanguage.text =
                        "${getString(R.string.main_status_listening)}...(${Constant.LISTENER_TIME - it})"
                } else {
                    mBinding.tvTargetLanguage.visibility = View.VISIBLE
                    mBinding.tvSourceLanguage.visibility = View.GONE
                    mBinding.tvTargetLanguage.text =
                        "${getString(R.string.main_status_listening)}...(${Constant.LISTENER_TIME - it})"
                }
            }
        composite.add(subscribe)

    }


    /**
     * 结束语音
     */
    private fun stopVoice() {
        Logcat.d(msg = "停止录音")

        //停止录音
        stopRecorder()
        // 停止播報翻譯
        //        viewModel.stopMusic()
    }

    /**
     * 停止录音
     */
    private fun stopRecorder() {
        stsEngine.stopSpeech()
        //isRecording = false
    }

    /**
     * 对引擎设置语言
     */
    private fun setLanguage() {
        val sourceLocale = getSourceLocale(applicationContext)
        val targetLocale = getTargetLocale(applicationContext)

        val (sourceSpeechLanguage, translateSourceLanguage) = viewModel.supportLanguage[sourceLocale] as Pair<RegionLang, LANG>
        val (targetSpeechLanguage, translateTargetLanguage) = viewModel.supportLanguage[targetLocale] as Pair<RegionLang, LANG>
        //val speechLanguage = if (sourceSpeech) sourceSpeechLanguage else targetSpeechLanguage

        when (currentSpeech) {
            SpeechType.SOURCE -> {
                stsEngine.setLanguage(speechLanguage = sourceSpeechLanguage, translateSourceLanguage = translateSourceLanguage, translateTargetLanguage = translateTargetLanguage, textSpeechSourceLanguage = translateTargetLanguage)
            }
            SpeechType.TARGET -> {
                stsEngine.setLanguage(speechLanguage = targetSpeechLanguage, translateSourceLanguage = translateTargetLanguage, translateTargetLanguage = translateSourceLanguage, textSpeechSourceLanguage = translateSourceLanguage)
            }
        }

    }

    override fun onEvent(code: Int, bundle: Bundle?) {
        when (code) {
            SpeechHelper.CALLBACK_STT_READY -> { //准备识别
                Logcat.d("EventListener", "CALLBACK_STT_READY")
                onAudioStartListener()
            }
            SpeechHelper.CALLBACK_STT_BEGIN -> { //开始识别
                Logcat.d("EventListener", "CALLBACK_STT_BEGIN")
            }
            SpeechHelper.CALLBACK_STT_PARTIAL -> { //识别部分结果返回，同时根据参数是否进行翻译结果返回
                val text = bundle?.getString(SpeechHelper.EXTRA_STT_PARTIAL)
                Logcat.d("EventListener", "CALLBACK_STT_PARTIAL  text=$text")
            }
            SpeechHelper.CALLBACK_STT_END -> { //识别结束
                Logcat.d("EventListener", "CALLBACK_STT_END")
            }
            SpeechHelper.CALLBACK_STT_FINISH -> { //识别完成，同时根据参数是否返回播报音频数据
                val text = bundle?.getString(SpeechHelper.EXTRA_STT_PARTIAL)
                if (text != null && text.isNotBlank()) {
                    val id = RoomDataUtil.getOrderIdByTime()
                    println("currentSpeech->$currentSpeech")
                    when (currentSpeech) {
                        SpeechType.SOURCE -> {
                            insertHistory(id, text, "", MainViewModel.HistoryItemViewType.SOURCE.value, MainViewModel.TransLateResultStatus.IN_TRANSLATE.value)
                        }
                        SpeechType.TARGET -> {
                            insertHistory(id, "", text, MainViewModel.HistoryItemViewType.TARGET.value, MainViewModel.TransLateResultStatus.IN_TRANSLATE.value)
                        }
                    }
                    setLanguage()
                    viewModel.sourceResult.postValue(text)
                    val key = if (currentSpeech == SpeechType.SOURCE) {
                        viewModel.targetLang.value?.key
                    } else {
                        viewModel.sourceLang.value?.key
                    }
                    TranslateService.getInstance()
                        .translateFromText(id, text, null, Locale.forLanguageTag(key), viewModel.targetResult)
                }
                Logcat.d("EventListener", "CALLBACK_STT_FINISH  text=$text")
                stopRecordUi()
            }
            SpeechHelper.CALLBACK_TRANSLATE_RESULT -> { //单独翻译结果
                //                val trans = bundle?.getSerializable(SpeechHelper.EXTRA_TRANSLATE_RESULT) as? Trans
                //                Logcat.d("EventListener", "CALLBACK_TRANSLATE_RESULT  text=$trans")
                //
                //                if (trans == null) {
                //                    Snackbar.make(mBinding.clMainParent, R.string.translate_status_fail, Snackbar.LENGTH_SHORT)
                //                        .show()
                //                    return
                //                }
                //                translateSuccess(trans)
            }
            SpeechHelper.CALLBACK_TRANS_FAIL -> { //翻译失败
                Snackbar.make(mBinding.clMainParent, R.string.translate_status_fail, Snackbar.LENGTH_SHORT)
                    .show()
                val history = bundle?.getString(SpeechHelper.EXTRA_TTS_HISTORYID)
                history?.let {
                    upHistoryByError(it, resources.getString(R.string.translate_status_fail), MainViewModel.TranslateError.TRANS_FAILE.value)
                }
            }
            SpeechHelper.CALLBACK_TTS_RESULT -> { //播报结果

                val path = bundle?.getString(SpeechHelper.EXTRA_TTS_RESULT)
                //                val history = bundle?.getString(SpeechHelper.EXTRA_TTS_HISTORYID)
                //                println("id=>"+history)
                if (!path.isNullOrEmpty() && File(path).exists()) {
                    viewModel.speechPath = path
                    //播放状态更改ui
                    viewModel.playState.postValue(MainViewModel.PlayState.Enabled)
                } else {
                    viewModel.playState.postValue(MainViewModel.PlayState.Default)
                }
                //                updateHistory(
                //                    history!!,
                //                    "",
                //                    null,
                //                    "",
                //                    null,
                //                    path
                //                )

                Logcat.d("EventListener", "CALLBACK_TTS_RESULT  path=$path")
            }
            SpeechHelper.CALLBACK_ERROR -> { //发生错误

                //                stopUIState()
                errorRecordUi()

                val error = bundle?.getInt(SpeechHelper.EXTRA_ERROR_CODE)
                val history = bundle?.getString(SpeechHelper.EXTRA_TTS_HISTORYID)
                history?.let {
                    upHistoryByError(it, resources.getString(R.string.translate_status_fail), MainViewModel.TranslateError.TRANS_FAILE.value)
                }
                Logcat.d("EventListener", "CALLBACK_ERROR  error=$error")
                when (error) {

                    /** Network operation timed out. */
                    SpeechHelper.ERROR_NETWORK_TIMEOUT -> {
                        Snackbar.make(mBinding.clMainParent, R.string.main_status_net_error, Snackbar.LENGTH_SHORT)
                            .show()

                    }

                    /** Other network related errors. */
                    SpeechHelper.ERROR_NETWORK -> {
                        Snackbar.make(mBinding.clMainParent, R.string.main_status_net_error, Snackbar.LENGTH_SHORT)
                            .show()
                    }

                    /** Audio recording error. */
                    SpeechHelper.ERROR_AUDIO -> {
                        Snackbar.make(mBinding.clMainParent, "error code=$error", Snackbar.LENGTH_SHORT)
                            .show()
                    }

                    /** Server sends error status. */
                    SpeechHelper.ERROR_SERVER -> {
                        Snackbar.make(mBinding.clMainParent, "error code=$error", Snackbar.LENGTH_SHORT)
                            .show()
                    }

                    /** Other client side errors. */
                    SpeechHelper.ERROR_CLIENT -> {
                        //Snackbar.make(mBinding.dragView, "error code=$error", Snackbar.LENGTH_SHORT).show()
                    }

                    /** No speech input */
                    SpeechHelper.ERROR_NO_SPEECH_INPUT -> {
                        //Snackbar.make(mBinding.dragView, "error code=$error", Snackbar.LENGTH_SHORT).show()
                        //Snackbar.make(mBinding.dragView, R.string.error_speech_timeout, Snackbar.LENGTH_SHORT).show()

                    }

                    /** No recognition result matched. */
                    SpeechHelper.ERROR_NO_MATCH -> {
                        //                        updateStatus(status = TranslateStatus.UN_RECOGNIZE, autoDismiss = true)
                        logEvent(context = application, code = Preferences.TB_003)
                    }

                    /** RecognitionService busy. */
                    SpeechHelper.ERROR_RECOGNIZER_BUSY -> {
                        Snackbar.make(mBinding.clMainParent, "error code=$error", Snackbar.LENGTH_SHORT)
                            .show()
                    }

                    /** Insufficient permissions */
                    SpeechHelper.ERROR_INSUFFICIENT_PERMISSIONS -> {
                        Snackbar.make(mBinding.clMainParent, "error code=$error", Snackbar.LENGTH_SHORT)
                            .show()
                    }

                    SpeechHelper.ERROR_OTHER -> {
                        //                        val throwable = bundle.getSerializable(SpeechHelper.EXTRA_THROWABLE) as? Throwable
                        //                        throwable?.printStackTrace()
                        //                        val log = File(externalCacheDir, "throwable_${System.currentTimeMillis()}.txt")
                        //                        throwable?.printStackTrace(PrintWriter(log))
                        Snackbar.make(mBinding.clMainParent, "error code=$error", Snackbar.LENGTH_SHORT)
                            .show()
                    } //其它错误


                }
            }
            SpeechHelper.CALLBACK_STT_VOLUME -> { //声波振幅
                val volume = bundle?.getFloat(SpeechHelper.EXTRA_STT_VOLUME) ?: 0f
                var volumePercent = bundle?.getInt(SpeechHelper.EXTRA_STT_VOLUME_PERCENT) ?: 1
                volumePercent = if (volumePercent <= 0) 1 else volumePercent
                Logcat.d("EventListener", "CALLBACK_STT_VOLUME  volume=$volume  volumePercent=$volumePercent")

            }
            SpeechHelper.CALLBACK_STT_AUDIO -> { //音频数据
                val audio = bundle?.getByteArray(SpeechHelper.EXTRA_STT_AUDIO)
            }
        }
    }

    /**
     * 插入新数据
     */
    private fun insertHistory(id: String, sourceText: String, targetText: String, viewType: Int, resultType: Int) {
        val db = Observable.fromCallable {
            viewModel.isInsert = true
            viewModel.cando = false
            Logcat.d("EventListener", id)
            val history =
                History(id = id, source_result = sourceText, target_result = targetText, display_type = viewType, error_code = null, error_msg = "", is_delete = false, mills = System.currentTimeMillis(), sourcelanguage_name = viewModel.sourceLang.value!!.name, result_type = resultType, sourcelanguage_tag = viewModel.sourceLang.value!!.key, targetlanguage_tag = viewModel.targetLang.value!!.key, targetlanguage_name = viewModel.targetLang.value!!.name)
            history

        }.subscribeOn(Schedulers.single()).subscribe({
            historyDao.insert(it)
        }, Throwable::printStackTrace)

        composite.add(db)
    }

    /**
     * 更新数据
     */
    private fun updateHistory(id: String, result: String, errorCode: Int?, errorMsg: String, resultType: Int?, voicePath: String?) {

        val db = Observable.fromCallable {
            val queryHistory = historyDao.queryHistory(id)
            queryHistory
        }.subscribeOn(Schedulers.single()).subscribe({
            it?.let {
                if (result.isNotBlank()) it.target_result = result

                it.error_code = errorCode ?: it.error_code
                if (errorMsg.isNotBlank()) {
                    it.error_msg = errorMsg
                }
                it.result_type = resultType ?: it.result_type
                it.voice_path = voicePath ?: it.voice_path
                viewModel.isInsert = false
                viewModel.cando = true
                historyDao.updateStarModel(it)
            }

        }, Throwable::printStackTrace)
        composite.add(db)
    }

    private fun upHistoryByError(id: String, msg: String, errorcode: Int) {
        updateHistory(id, "", errorcode, msg, MainViewModel.TransLateResultStatus.ERROR.value, null)
    }

    /**
     * 翻译成功结果处理
     * @param text 译文
     */
    private fun translateSuccess(trans: Trans) {
        val text = trans.result!!.translation //translationList.joinToString("\n")
        println("text = $text")
        // 显示结果页
    }

    /**
     * 说话类型
     */
    enum class SpeechType {
        SOURCE, TARGET
    }

    //网络状态
    private val netWorkLiveData = MutableLiveData<Boolean>()

    //通知广播
    private val notificationReceiver = NotificationReceiver(netWorkLiveData)

    private val composite by lazy { io.reactivex.disposables.CompositeDisposable() }

    private var currentSpeech = SpeechType.SOURCE

    enum class TranslateStatus(val stringRes: Int) {

        //正在聆听
        LISTENER(R.string.main_status_listening),

        //录入超时
        RECORD_TIMEOUT(R.string.main_status_record_timeout),

        //翻译中
        TRANSLATING(R.string.main_status_translating),

        //网络错误，请重试
        NETWORK_ERROR(R.string.main_status_net_error),

        //没网络
        NETWORK_NOT(R.string.main_status_net_not),

        //未能识别您的声音
        UN_RECOGNIZE(R.string.main_status_unrecognize),

        //空状态
        EMPTY(R.string.empty_text)

    }

    /**
     * 开始录音监听
     */
    private fun onAudioStartListener() {
        //        updateStatus(TranslateStatus.LISTENER)
        // 震动
        startVibrate()

        //        isTimeOut = false

        //        if (speechDisposable?.isDisposed == false) {
        //            speechDisposable?.dispose()
        //        }
        //
        //        mHandler.removeCallbacks(timeOutRecorder)
        //        mHandler.postDelayed(timeOutRecorder, RECORDER_TIME_OUT)


    }

    private fun startVibrate() {
        val preferences = PreferenceManager.getDefaultSharedPreferences(this)
        val preferenceVibrator = preferences.getBoolean("vibration_preference", false)

        if (preferenceVibrator) {

            val vibrator = getSystemService(Context.VIBRATOR_SERVICE) as Vibrator

            if (vibrator.hasVibrator()) {

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    vibrator.vibrate(VibrationEffect.createOneShot(100, VibrationEffect.DEFAULT_AMPLITUDE))
                } else {
                    vibrator.vibrate(100)
                }

            }
        }

    }

    /**
     * 所有权限处理
     */
    private fun checkAppPermissions(showDialog: Boolean = true): Boolean {
        if (!checkPermission(Manifest.permission.RECORD_AUDIO)) {
            if (showDialog) {
                PermissionFragment().show(supportFragmentManager, "permission")
            }
            return false
        }
        return true
    }

    /**
     * 观察网络状态
     */
    private fun observeNetWork() {
        netWorkLiveData.observe(this, netWorkObserver)
        netWorkLiveData.value = networkIsConnected(this)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val connectivityManager =
                getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
            connectivityManager.registerDefaultNetworkCallback(networkCallback)
        }
    }

    private val networkCallback = object : ConnectivityManager.NetworkCallback() {
        override fun onAvailable(network: Network) {
            super.onAvailable(network)

            Logcat.d(msg = "NetworkCallback#onAvailable  network=$network")
            runOnUiThread { netWorkLiveData.value = networkIsConnected(applicationContext) }
        }

        override fun onUnavailable() {
            super.onUnavailable()
            Logcat.d(msg = "NetworkCallback#onUnavailable")
        }

        override fun onLost(network: Network) {
            super.onLost(network)
            Logcat.d(msg = "NetworkCallback#onLost")
            runOnUiThread { netWorkLiveData.value = networkIsConnected(applicationContext) }
        }

        override fun onLosing(network: Network, maxMsToLive: Int) {
            super.onLosing(network, maxMsToLive)
            Logcat.d(msg = "NetworkCallback#onLosing")
        }

        override fun onCapabilitiesChanged(network: Network, networkCapabilities: NetworkCapabilities) {
            super.onCapabilitiesChanged(network, networkCapabilities)
            Logcat.d(msg = "NetworkCallback#onCapabilitiesChanged")
        }

        override fun onLinkPropertiesChanged(network: Network, linkProperties: LinkProperties) {
            super.onLinkPropertiesChanged(network, linkProperties)
            Logcat.d(msg = "NetworkCallback#onLinkPropertiesChanged")
        }

    }

    /**
     * 移除网络状态观察
     */
    private fun removeObserveNetWork() {
        netWorkLiveData.removeObserver(netWorkObserver)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val connectivityManager =
                getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
            connectivityManager.unregisterNetworkCallback(networkCallback)
        }
    }


    private val netWorkObserver = Observer<Boolean> { connected ->
        if (!connected) {
            logEvent(context = application, code = Preferences.TB_002)
            showSnackBar(mBinding.tlMain, R.string.main_net_error)
            stopVoice()
        }
    }

    /**
     * 本地动态广播注册
     */
    private fun registerBroadcast() {
        val filter = IntentFilter(NotificationReceiver.ACTION_NOTIFICATION)

        filter.addAction(ConnectivityManager.CONNECTIVITY_ACTION)
        val broadcastManager = LocalBroadcastManager.getInstance(this)
        broadcastManager.registerReceiver(notificationReceiver, filter)
        broadcastManager.sendBroadcast(Intent(NotificationReceiver.ACTION_NOTIFICATION))
    }

    override fun onDestroy() {
        val broadcastManager = LocalBroadcastManager.getInstance(this)
        broadcastManager.unregisterReceiver(notificationReceiver)
        composite.clear()
        TranslateService.getInstance().destroy()
        super.onDestroy()
    }

    override fun onRecordClick(v: View) {
        when (v.id) {
            R.id.tv_main_language_you_say -> {
                startTargetVoice(v, SpeechType.SOURCE)
            }
            R.id.tv_main_language_he_say -> {
                startTargetVoice(v, SpeechType.TARGET)
            }
        }
    }

    override fun onRecordLongClick(v: View) {
        isLongPress = true
        when (v.id) {
            R.id.tv_main_language_you_say -> {
                startTargetVoice(v, SpeechType.SOURCE)
            }
            R.id.tv_main_language_he_say -> {
                startTargetVoice(v, SpeechType.TARGET)
            }
        }
    }

    override fun onEventUp(v: View) {
        if (isLongPress) {
            isLongPress = false
            when (v.id) {
                R.id.tv_main_language_you_say -> {
                    startTargetVoice(v, SpeechType.SOURCE)
                }
                R.id.tv_main_language_he_say -> {
                    startTargetVoice(v, SpeechType.TARGET)
                }
            }
        }
    }

    private fun startTargetVoice(v: View, speechType: SpeechType) {
        val speechStatus = v.getTag(R.id.view_speech_status)
        if (speechStatus == true) { //正在speech，停止录音
            composite.clear()
            showResultUi()
            stopVoice()
            v.setTag(R.id.view_speech_status, false)
        } else {
            //处于未说话状态,开始录音
            stopVoice()
            currentSpeech = speechType
            setLanguage()
            // 权限
            if (!checkAppPermissions()) {
                return
            }
            //网络状态
            if (netWorkLiveData.value != true) {
                return
            }
            v.setTag(R.id.view_speech_status, true)
            startVoice()
        }
    }

    private fun copyText(history: History, view: View) {
        when (history.result_type) {
            MainViewModel.TransLateResultStatus.SUCCESS.value -> {
                when (history.display_type) {
                    MainViewModel.HistoryItemViewType.SOURCE.value -> {
                        viewModel.copy(view, history.target_result)
                    }
                    MainViewModel.HistoryItemViewType.TARGET.value -> {
                        viewModel.copy(view, history.source_result)
                    }
                }
            }
            else -> {
                showSnackBar(mBinding.tlMain, R.string.main_status_net_error)
            }
        }
    }
}