package com.talpa.master.ui.main

import android.app.Application
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.graphics.drawable.Drawable
import android.media.MediaPlayer
import android.util.SparseArray
import android.view.View
import android.widget.Toast
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.ViewModelProviders
import com.cy.sts.lang.RegionLang
import com.google.android.material.snackbar.BaseTransientBottomBar
import com.google.android.material.snackbar.Snackbar
import com.talpa.tengine.lang.LANG
import com.talpa.master.R
import com.talpa.master.common.MediaPlayerHelper
import com.talpa.master.framework.RoomHelper
import com.talpa.master.repository.MainRepository
import com.talpa.master.repository.room.model.LanguageModel
import com.talpa.master.framework.BaseViewModel
import com.talpa.master.framework.Resource
import io.reactivex.schedulers.Schedulers
import java.util.*
import kotlin.properties.Delegates

/**
 * Create by kgxl on 2021/4/5
 */
class MainViewModel(val repository: MainRepository, private val ctx: Application) : BaseViewModel(),
    MediaPlayer.OnCompletionListener,
    MediaPlayer.OnErrorListener {
    companion object {
        fun factory(application: Application) = object : ViewModelProvider.Factory {
            override fun <T : ViewModel> create(modelClass: Class<T>): T {
                return MainViewModel(MainRepository(), application) as T
            }
        }

        fun viewModel(fragmentActivity: FragmentActivity) = ViewModelProviders.of(
            fragmentActivity,
            factory(fragmentActivity.application)
        ).get(MainViewModel::class.java)
    }

    //本地支持语言列表
    val localLanguageLiveData = MutableLiveData<ArrayList<LanguageModel.LanguageList>>()

    var isInsert = true
    var cando = true

    enum class HistoryItemViewType(val value: Int) {
        SOURCE(0),//源语言  绿色背景
        TARGET(1)//目标语言  蓝色背景
    }

    /**
     * 翻译结果状态
     * 0异常 1成功 2翻译中
     */
    enum class TransLateResultStatus(val value: Int) {
        ERROR(0),//异常
        SUCCESS(1),//成功
        IN_TRANSLATE(2) //翻译中
    }

    enum class TranslateError(val value: Int) {
        NETWORK_FAILE(0),
        TRANS_FAILE(1)
    }

    //源语言logo
    val observableSourceLogo = MutableLiveData<Drawable>()

    //源语言文本
    val observableSourceText = MutableLiveData<String>()

    //目标语言缩写
    val observableSourceShortLang = MutableLiveData<String>()


    //当前源语言
    var sourceLanguage: LanguageModel.LanguageList? by Delegates.observable(null) { prop, old: LanguageModel.LanguageList?, new: LanguageModel.LanguageList? ->
        if (new != null) {
            val locale = Locale.forLanguageTag(new.key)
            val lang = locale?.language?.toUpperCase(locale)
            observableSourceShortLang.postValue(lang)
        }
    }

    //目标语言logo
    val observableTargetLogo = MutableLiveData<Drawable>()

    //目标语言文本
    val observableTargetText = MutableLiveData<String>()

    //目标语言缩写
    val observableTargetShortLang = MutableLiveData<String>()

    //当前目标语言
//    var targetLanguage: LanguageModel.LanguageList? by Delegates.observable(null) { prop, old: LanguageModel.LanguageList?, new: LanguageModel.LanguageList? ->
//        if (new != null) {
//            val locale = Locale.forLanguageTag(new.key)
//            val lang = locale?.language?.toUpperCase(locale)
//            observableTargetShortLang.postValue(lang)
//        }
//    }


    //当前选中的语言类型,默认目标语言
    var currentLanguageType: LanguageType = LanguageType.TARGET


    enum class LanguageType(val value: Int) {
        SOURCE(1),//源语言
        TARGET(2)//目标语言
    }

    enum class PlayState {
        /**
         * 默认禁用状态
         */
        Default,

        /**
         * 播放状态
         */
        Playing,

        /**
         * 可播放状态
         */
        Enabled
    }

    val source = MutableLiveData<Resource<String>>()
    val haveGoogleVoice = MutableLiveData<Boolean>()
    val languages = MutableLiveData<List<LanguageModel.LanguageList>>()
    val hideDialog = MutableLiveData<Boolean>()
    val sourceLang = MutableLiveData<LanguageModel.LanguageList>()
    val targetLang = MutableLiveData<LanguageModel.LanguageList>()
    var currentLangType: LanguageType = LanguageType.SOURCE
    var langCheck: SparseArray<Boolean> = SparseArray()
    val isShowResult = MutableLiveData<Boolean>()
    val isSource = MutableLiveData<Boolean>()
    val sourceResult = MutableLiveData<String>()
    val targetResult = MutableLiveData<String>()
    val sourceTranslateError = MutableLiveData<String>()
    val targetTranslateError = MutableLiveData<String>()
    val sourceTranslateHi = MutableLiveData<String>()
    val targetTranslateHi = MutableLiveData<String>()

    /**
     * 语音路径
     */
    var speechPath: String = ""

    val playState = MutableLiveData<PlayState>()

    fun getLocalAllLanguage() {
        if (currentLangType == LanguageType.SOURCE) {
            repository.getLocalAllLanguage(languages, targetLang.value)
        } else if (currentLangType == LanguageType.TARGET) {
            repository.getLocalAllLanguage(languages, sourceLang.value)
        }
    }

    /**
     * 复制译文
     */
    fun copy(view: View, text: String) {

        val context = view.context

        val cm = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        // 创建普通字符型ClipData
        val mClipData = ClipData.newPlainText("Label", text)
        cm.setPrimaryClip(mClipData)

        //SnackBar
        val snackBar = Snackbar.make(view, R.string.text_copy_complete, Snackbar.LENGTH_SHORT)
        snackBar.animationMode = BaseTransientBottomBar.ANIMATION_MODE_FADE
        snackBar.show()
    }

    fun playTranslateVoice(v: View) {
        if (!speechPath.isNullOrEmpty()) {
            stopTranslateVoice()
            playState.value = PlayState.Playing
            MediaPlayerHelper.playMusic(ctx, speechPath, this, this)
        } else {
            val snackBar = Snackbar.make(v, R.string.read_fail, Snackbar.LENGTH_SHORT)
            snackBar.animationMode = BaseTransientBottomBar.ANIMATION_MODE_FADE
            snackBar.show()
        }
    }

    fun stopTranslateVoice() {
        speechPath?.let {
            playState.value = PlayState.Enabled
            MediaPlayerHelper.stopMusic()
        }
    }

    override fun onError(mp: MediaPlayer?, what: Int, extra: Int): Boolean {
        Toast.makeText(ctx, "player error $what", Toast.LENGTH_SHORT).show()
        return true
    }

    override fun onCompletion(mp: MediaPlayer?) {
        playState.value = PlayState.Enabled
    }

    fun setLanguage() {
        RoomHelper.appDao.queryLanguageBySelect(languageType = LanguageType.SOURCE.value)
            .subscribeOn(Schedulers.io()).subscribe {
                if (it.isNotEmpty())
                    sourceLang.postValue(it[0])
            }
        RoomHelper.appDao.queryLanguageBySelect(languageType = LanguageType.TARGET.value)
            .subscribeOn(Schedulers.io()).subscribe {
                if (it.isNotEmpty())
                    targetLang.postValue(it[0])
            }
    }

    val supportLanguage = hashMapOf<Locale, Pair<RegionLang, LANG>>().apply {
        this[Locale.forLanguageTag("bn-IN")] = RegionLang.BN_IN to LANG.BN
        this[Locale.forLanguageTag("hi-IN")] = RegionLang.HI_IN to LANG.HI
        this[Locale.forLanguageTag("en-IN")] = RegionLang.EN_IN to LANG.EN
        this[Locale.forLanguageTag("en-US-POSIX")] = RegionLang.EN_US to LANG.EN
        this[Locale.forLanguageTag("en-GB")] = RegionLang.EN_GB to LANG.EN
        this[Locale.forLanguageTag("fr-FR")] = RegionLang.FR_FR to LANG.FR
        this[Locale.forLanguageTag("es-ES")] = RegionLang.ES_EC to LANG.ES
        this[Locale.forLanguageTag("ar-EG")] = RegionLang.AR_EG to LANG.AR
        this[Locale.forLanguageTag("ar-SA")] = RegionLang.AR_SA to LANG.AR
        this[Locale.forLanguageTag("ar-AE")] = RegionLang.AR_AE to LANG.AR
        this[Locale.forLanguageTag("th-TH")] = RegionLang.TH_TH to LANG.TH
        this[Locale.forLanguageTag("ko-KR")] = RegionLang.KO_KR to LANG.KO
        this[Locale.forLanguageTag("zh-Hans-CN")] = RegionLang.CMN_HANS_CN to LANG.ZH_CN
        this[Locale.forLanguageTag("sw-TZ")] = RegionLang.SW_TZ to LANG.SW
        this[Locale.forLanguageTag("sw-KE")] = RegionLang.SW_KE to LANG.SW
        this[Locale.forLanguageTag("pt-PT")] = RegionLang.PT_PT to LANG.PT
        this[Locale.forLanguageTag("am")] = RegionLang.AM_ET to LANG.AM
        this[Locale.forLanguageTag("yo")] = RegionLang.EN_NG to LANG.EN
        this[Locale.forLanguageTag("ha")] = RegionLang.EN_NG to LANG.EN
        this[Locale.forLanguageTag("ja-JP")] = RegionLang.JA_JP to LANG.JA
    }
}