package com.talpa.master.ui.history

import android.app.Activity
import android.content.Intent
import android.graphics.drawable.AnimationDrawable
import android.media.AudioManager
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.observe
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.SimpleItemAnimator
import com.blankj.utilcode.util.SizeUtils
import com.talpa.master.R
import com.talpa.master.common.Constant
import com.talpa.master.common.Constant.HISTORY_LIST
import com.talpa.master.databinding.DataBindingActivity
import com.talpa.master.databinding.MolActivityHistoryBinding
import com.talpa.master.ui.main.CommunicationActivity
import com.talpa.master.ui.main.MainViewModel
import com.talpa.master.utils.AppUtil
import com.talpa.translate.face.TranslateService
import kotlinx.coroutines.launch

class HistoryActivity : DataBindingActivity<MolActivityHistoryBinding>() {
    private val viewModel: HistoryViewModel by lazy {
        ViewModelProvider(
            this@HistoryActivity,
            HistoryViewModel.factory()
        )[HistoryViewModel::class.java]
    }

    private var mAudioManger: AudioManager? = null
    private var afChangeListener: AudioManager.OnAudioFocusChangeListener? = null
    private var voiceAnimationDrawable: AnimationDrawable = AnimationDrawable()
    override fun getContentView(): Int = R.layout.mol_activity_history
    override fun initView() {
        val historyId = intent.getStringExtra(Constant.HISTORY_ID)
        historyId?.let { viewModel.translateHistoryId = it }
        mBinding.rvHistory.layoutManager = LinearLayoutManager(this)
        mBinding.rvHistory.adapter = HistoryAdapter(viewModel, HISTORY_LIST)
        (mBinding.rvHistory.itemAnimator as SimpleItemAnimator).supportsChangeAnimations = false
        mBinding.rvHistory.addItemDecoration(HistoryAdapter.SpaceItemDecoration(SizeUtils.dp2px(16F)))
        val darkTheme = AppUtil.isDarkTheme(applicationContext)
        viewModel.isDark.postValue(darkTheme)

        setSupportActionBar(mBinding.toolBar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.setDisplayShowTitleEnabled(false)
        mBinding.toolBar.setNavigationOnClickListener {
            backToIsStartMark()
            finish()
        }
        viewModel.registerCall()
        initOnAudioFocus()

        mBinding.toolBar.navigationContentDescription = null

        // 切换暗黑模式暂停播放
        viewModel.voicePlayStatus.value = MainViewModel.VoicePlayStatus.END
        lifecycleScope.launch {
            TranslateService.Companion.getInstance().stopSpeak()
        }
    }

    private  fun initOnAudioFocus() {
        mAudioManger = this.getSystemService(AppCompatActivity.AUDIO_SERVICE) as AudioManager
        afChangeListener = AudioManager.OnAudioFocusChangeListener { focusChange ->
            when (focusChange) {
                AudioManager.AUDIOFOCUS_LOSS_TRANSIENT -> {
                    // Pause playback
                    Log.d(CommunicationActivity.TAG, "init: 音頻焦點——————————AUDIOFOCUS_LOSS_TRANSIENT")
                    viewModel.stopTranslateVoice()
                }
            }
        }
        mAudioManger!!.requestAudioFocus(
                afChangeListener,
                AudioManager.STREAM_MUSIC,
                AudioManager.AUDIOFOCUS_GAIN
        )
    }

    override fun init() {
        mBinding.viewModel = viewModel
        viewModel.selectHistory()
        viewModel.historys.observe(this, {
            mBinding.rvHistory.adapter?.notifyDataSetChanged()
        })

        viewModel.voicePlayStatus.observe(this, {
            viewModel.currentPlayStatus = it
            if (viewModel.lastPlayIndex == viewModel.currentPlayClickIndex) {
                mBinding.rvHistory.adapter?.notifyItemChanged(viewModel.currentPlayClickIndex)
            } else {
                mBinding.rvHistory.adapter?.notifyItemChanged(viewModel.lastPlayIndex)
                mBinding.rvHistory.adapter?.notifyItemChanged(viewModel.currentPlayClickIndex)
            }
        })

    }

    // 退到后台或者息屏停止播放
    override fun onPause() {
        super.onPause()
        viewModel.currentPlayStatus = MainViewModel.VoicePlayStatus.END
        mBinding.rvHistory.adapter?.notifyItemChanged(viewModel.currentPlayClickIndex)
        lifecycleScope.launch {
            TranslateService.Companion.getInstance().stopSpeak()
        }
    }

    override fun onDestroy() {
        viewModel.unregisterCall()
        voiceAnimationDrawable.stop()
        mAudioManger?.abandonAudioFocus(afChangeListener)
        super.onDestroy()
    }

    override fun onBackPressed() {
        backToIsStartMark()
        super.onBackPressed()
    }

    // 返回
    fun backToIsStartMark() {
        val resultIntent = Intent()
        resultIntent.putExtra(Constant.IS_STAR_MARK, viewModel.isStarMark)
        setResult(Activity.RESULT_OK, resultIntent)
        lifecycleScope.launch {
            TranslateService.Companion.getInstance().stopSpeak()
        }
    }

}