package com.talpa.master.ui.history

import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.view.View
import android.widget.ImageView
import android.widget.Toast
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewModelScope
import com.hiservice.aidl.UtteranceProgressCallback
import com.hiservice.connect.HiServiceConnection
import com.hiservice.connect.registerUtteranceProgress
import com.hiservice.connect.unregisterUtteranceProgress
import com.talpa.master.R
import com.talpa.master.common.Constant.STARMARK_LIST
import com.talpa.master.framework.RoomHelper
import com.talpa.master.repository.room.dao.HistoryDao
import com.talpa.master.repository.room.model.History
import com.talpa.master.ui.main.MainViewModel
import com.talpa.master.utils.event.Event
import com.talpa.translate.face.TranslateService
import com.talpa.translate.framework.BaseViewModel
import io.reactivex.Flowable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.*

/**
 * Create by kgxl on 2021/4/13
 */
class HistoryViewModel(val dao: HistoryDao) : BaseViewModel() {
    val historys by lazy { MutableLiveData<MutableList<History>>() }
    val isSource = MutableLiveData<Boolean>()
    val voicePlayStatus = MutableLiveData<MainViewModel.VoicePlayStatus>()
    var isStarMark = false
    var lastPlayIndex = -1
    var currentPlayClickIndex = -1
    var currentPlayStatus = MainViewModel.VoicePlayStatus.START
    var translateHistoryId: String = ""

    val enlargeDialogLiveData = MutableLiveData<Event<History>>()

    private var markDisposable: Disposable? = null
    private var historyDisposable: Disposable? = null

    companion object {
        fun factory() = object : ViewModelProvider.Factory {
            override fun <T : ViewModel> create(modelClass: Class<T>): T {
                return HistoryViewModel(RoomHelper.historyDao) as T
            }
        }
    }

    // 历史列表数据
    fun selectHistory() {

        historyDisposable = dispose(historyDisposable)

        historyDisposable = dao.queryLatestHistory().subscribeOn(Schedulers.io()).subscribe {
            historys.postValue(it)
        }
    }

    // 星标列表数据
    fun selectHistoryByMark() {

        markDisposable = dispose(markDisposable)

        markDisposable = dao.queryAllHistoryByMark().subscribeOn(Schedulers.io()).subscribe {
            historys.postValue(it)
        }
    }

    /**
     * 复制译文
     */
    fun copyText(history: History, view: View) {
//        when (history.display_type) {
//            MainViewModel.HistoryItemViewType.SOURCE.value -> {
        copy(view, history.target_result)
//            }
//            MainViewModel.HistoryItemViewType.TARGET.value -> {
//                copy(view, history.source_result)
//            }
//        }
    }


    fun copy(view: View, text: String) {
        val context = view.context

        val cm = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        // 创建普通字符型ClipData
        val mClipData = ClipData.newPlainText("Label", text)
        cm.setPrimaryClip(mClipData)

        Toast.makeText(context, R.string.text_copy_complete, Toast.LENGTH_SHORT).show()
    }

    /**
     * 点击星标
     */
    fun starMark(history: History, type: String, position: Int, imageView: ImageView) {
        history.isStartMarker = !history.isStartMarker
        if (translateHistoryId == history.id.toString().trim()) {
            isStarMark = history.isStartMarker
        }
        if (history.isStartMarker) {
            history.end_mills = System.currentTimeMillis()
        } else {
            history.end_mills = 0L
        }
        Flowable.fromCallable {
            RoomHelper.historyDao.updateStarModel(history)
        }.subscribeOn(Schedulers.io()).observeOn(AndroidSchedulers.mainThread()).subscribe {
            if (history.isStartMarker) {
//                imageView.setImageResource(R.drawable.mol_ic_collected)
//                Toast.makeText(application, R.string.text_starred_startMark, Toast.LENGTH_SHORT)
//                    .show()
            } else {
//                imageView.setImageResource(R.drawable.mol_ic_collect)
//                Toast.makeText(application, R.string.text_unstarred_startMark, Toast.LENGTH_SHORT)
//                    .show()
                if (STARMARK_LIST == type) {
                    stopTranslateVoice()
                }
            }
        }
    }

    /**
     * 播报
     */
    fun playTranslateVoice(history: History) {
        viewModelScope.launch(Dispatchers.IO) {
            if (currentPlayClickIndex == lastPlayIndex) {
                if (TranslateService.Companion.getInstance().isSpeaking()) {
                    stopSpeakTranslate()
                } else {
                    val isSuccess: Boolean = TranslateService.Companion.getInstance().startSpeak(
                        history.target_result,
                        Locale.forLanguageTag(history.targetlanguage_tag).language
                    )
                }
            } else {
                stopTranslateVoice()
                val isSuccess: Boolean = TranslateService.Companion.getInstance().startSpeak(
                    history.target_result,
                    Locale.forLanguageTag(history.targetlanguage_tag).language
                )
            }
        }


    }

    /**
     * 停止播报
     */
    fun stopTranslateVoice() {
        viewModelScope.launch(Dispatchers.IO) {
            if (TranslateService.Companion.getInstance().isSpeaking()) {
                TranslateService.Companion.getInstance().stopSpeak()
                voicePlayStatus.postValue(MainViewModel.VoicePlayStatus.END)
            }
        }
    }

    private fun stopSpeakTranslate() {
        viewModelScope.launch(Dispatchers.IO) {
            TranslateService.Companion.getInstance().stopSpeak()
            voicePlayStatus.postValue(MainViewModel.VoicePlayStatus.END)
        }
    }


    fun registerCall() = viewModelScope.launch(context = Dispatchers.IO) {
        HiServiceConnection.registerUtteranceProgress(callback = utteranceProgressCallback)
    }

    fun unregisterCall() = viewModelScope.launch(context = Dispatchers.IO) {
        HiServiceConnection.unregisterUtteranceProgress(callback = utteranceProgressCallback)
    }

    private val map = HashMap<Int, Boolean>()
    val updateListEnlarge = MutableLiveData<Pair<Int, Int>>()

    fun enlarge(mHistory: History, b: Boolean) {
        updateListEnlarge.value = Pair(0, 0)
        map.clear()
        map[mHistory.id] = b
    }

    fun isEnlarge(it: History): Boolean {
        return map[it.id] ?: false
    }

    fun itemEnlarge(info: History, position: Int) {
        stopTranslateVoice()
        enlargeDialogLiveData.value = Event(info)
    }

    override fun onCleared() {
        super.onCleared()
        markDisposable = dispose(markDisposable)
        historyDisposable = dispose(historyDisposable)

        map.clear()
    }

    private fun dispose(disposable: Disposable?): Disposable? {
        disposable?.apply {
            if (!isDisposed) {
                dispose()
            }
        }
        return null
    }

    val utteranceProgressCallback = object :
        UtteranceProgressCallback.Stub() {
        override fun onStart(p0: String?) {
            voicePlayStatus.postValue(MainViewModel.VoicePlayStatus.START)
        }

        override fun onDone(p0: String?) {
            voicePlayStatus.postValue(MainViewModel.VoicePlayStatus.END)
        }

        override fun onError(p0: String?) {

        }
    }

    enum class VoicePlayStatus(val value: Int) {
        START(0),//开始
        PAUSE(1),//暂停
        END(2),//结束
    }


}