package com.talpa.master.ui.main

import android.content.Intent
import android.os.Bundle
import android.os.Handler
import android.util.Log
import android.view.KeyEvent
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.window.layout.WindowInfoTracker
import com.talpa.master.R
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * val intent = Intent()
intent.component =
ComponentName("com.transsion.mol", "com.talpa.master.ui.main.CommunicationActivity")
intent.putExtra("_key_input_source_content", "外面传进来的默认源语言内容")
intent.putExtra("_key_input_source_code", "zh")
intent.putExtra("_key_input_target_code", "en")
startActivity(intent)
 */
class CommunicationActivity : CommunicationActivityV2() {

    companion object {
        const val TAG = "FaceToFace"

        const val KEY_INPUT_SOURCE_CONTENT = "_key_input_source_content"
        const val KEY_INPUT_SOURCE_LANGUAGE_CODE = "_key_input_source_code"
        const val KEY_INPUT_TARGET_LANGUAGE_CODE = "_key_input_target_code"

        var keyCode = 0
    }

    /**
     * 外面传进来的默认源语言内容
     */
    private var mInputSourceContent: String? = null

    /**
     * 外面传进来的默认源语言内容的 语言码
     */
    private var mInputSourceLanguageCode: String? = null

    /**
     * 外面传进阿里的默认需要翻译的语言码
     */
    private var mInputTargetLanguageCode: String? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        parseParam(intent)
        super.onCreate(savedInstanceState)

        lifecycleScope.launch(Dispatchers.Main) {
            lifecycle.repeatOnLifecycle(Lifecycle.State.STARTED) {
                WindowInfoTracker.getOrCreate(this@CommunicationActivity)
                    .windowLayoutInfo(this@CommunicationActivity)
                    .collect { newLayoutInfo ->
                        dismissLanguageDialog()
                        Log.d(TAG, "newLayoutInfo = $newLayoutInfo,keyCode= $keyCode")
                        if (keyCode == 307) {
                            keyCode = 0
                            reStart()
                        }
                    }
            }
        }
    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        Companion.keyCode = keyCode
        Log.d(TAG, "keyCode= $keyCode")
        if (keyCode == 308) {
            reStart()
        }
        return super.onKeyDown(keyCode, event)
    }

    private fun reStart() {
        finish()
        Handler().postDelayed({
            intent.addFlags(Intent.FLAG_ACTIVITY_NO_ANIMATION)
            overridePendingTransition(0, 0)
            startActivity(intent)
        }, 300)
    }

    override fun onNewIntent(intent: Intent?) {
        parseParam(intent)
        super.onNewIntent(intent)
    }

    override fun initView() {
        super.initView()


        lifecycleScope.launch(Dispatchers.Main) {
            initInputContent()
        }
    }
    private suspend fun initInputContent() {
        mInputSourceLanguageCode?.let {
            if (it.isNotEmpty() && localCheck(it)) {
                viewModel.updateSourceLanguage(it)
            }
        }

        mInputTargetLanguageCode?.let {
            if (it.isNotEmpty() && localCheck(it)) {
                viewModel.updateTargetLanguage(it)
            }
        }

        mInputSourceContent?.let {
            if (it.isNotEmpty()) {
                lifecycleScope.launch(Dispatchers.Main) {
                    //开始翻译中
                    viewModel.startTranslateText(it, true)
                }
            }
        }
    }

    private suspend fun localCheck(it: String): Boolean = withContext(Dispatchers.IO) {
        try {
            val item = viewModel.languageList.findLast { lang -> it == lang.key }
            null != item
        } catch (e: Exception) {
            e.printStackTrace()
            false
        }
    }
    private fun parseParam(intent: Intent?) {
        intent?.let {
            mInputSourceContent = it.getStringExtra(KEY_INPUT_SOURCE_CONTENT)
            mInputSourceLanguageCode =
                it.getStringExtra(KEY_INPUT_SOURCE_LANGUAGE_CODE)
            mInputTargetLanguageCode =
                it.getStringExtra(KEY_INPUT_TARGET_LANGUAGE_CODE)
        }
        Log.i(
            TAG,
            "input param:srcCode:$mInputSourceLanguageCode, dstCode:$mInputTargetLanguageCode, content:$mInputSourceContent"
        )
    }

    override fun finish() {
        super.finish()
        Log.i("MOL_SDK", "face_finish:" + Log.getStackTraceString(Throwable()))
    }

    /**
     * 说话类型
     */
    enum class SpeechType {
        SOURCE, TARGET
    }

    enum class TranslateStatus(val stringRes: Int) {
        //正在听
        LISTENER(R.string.main_status_listening),

        //录入超时
        RECORD_TIMEOUT(R.string.main_status_record_timeout),

        //网络错误，请重试
        NETWORK_ERROR(R.string.main_net_error),

        //空状态
        EMPTY(R.string.empty_text),

        // 识别完成
        COMPLATE(R.string.empty_text),

    }
}

