package com.talpa.master.ui.main

import android.Manifest
import android.content.Intent
import android.content.pm.ActivityInfo
import android.media.AudioManager
import android.os.Bundle
import android.util.Log
import android.view.View
import android.widget.Toast
import androidx.activity.viewModels
import androidx.core.view.isVisible
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.lifecycleScope
import com.blankj.utilcode.util.SizeUtils
import com.hiservice.aidl.HiServiceInterface
import com.talpa.datareport.report.*
import com.talpa.lib.language.bean.LanguageInfo
import com.talpa.lib.language.open.LanguageSelectDialog
import com.talpa.lib.language.util.FoldUtils
import com.talpa.lib.language.viewModel.LDLanguageViewModel
import com.talpa.master.R
import com.talpa.master.common.Constant
import com.talpa.master.common.VibrateUtils
import com.talpa.master.common.getString
import com.talpa.master.databinding.DataBindingActivity
import com.talpa.master.databinding.MolActivityConversationV2Binding
import com.talpa.master.databinding.isDarkFontDefault
import com.talpa.master.databinding.setDarkMode
import com.talpa.master.framework.RoomHelper
import com.talpa.master.repository.room.model.History
import com.talpa.master.ui.history.HistoryAdapter
import com.talpa.master.ui.main.fragment.TextTranslateFragment
import com.talpa.master.ui.main.v2.CommunicationViewModelV2
import com.talpa.master.ui.main.v2.ConversationAdapter
import com.talpa.master.ui.main.v2.TranslateEnlargeActivity
import com.talpa.master.ui.mark.StarMarkActivity
import com.talpa.master.utils.AppUtil
import com.talpa.master.utils.MyLinearLayoutManager
import com.talpa.master.utils.NetWorkUtil
import com.talpa.master.utils.event.Event
import com.talpa.translate.face.TranslateService
import com.talpa.translate.face.common.Constants
import com.talpa.translate.face.common.Logcat
import com.talpa.translate.face.stt.STSEngine
import com.talpa.translate.face.stt.SpeechHelper
import com.talpa.translate.face.stt.EventListener
import com.talpa.translate.framework.checkPermission
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.*
import java.util.concurrent.TimeUnit

open class CommunicationActivityV2 : DataBindingActivity<MolActivityConversationV2Binding>(),
    EventListener {
    companion object {
        const val REQUEST_CODE = 100
        const val TAG = "FaceToFace"
    }

    protected val viewModel: CommunicationViewModelV2 by viewModels {
        ViewModelProvider.AndroidViewModelFactory.getInstance(application)
    }
    private val roomViewModel: RoomViewModel by viewModels {
        ViewModelProvider.AndroidViewModelFactory.getInstance(application)
    }


    private var mConversationAdapter: ConversationAdapter? = null

    private var mLanguageSelectDialog:LanguageSelectDialog? = null
    private var mLDLanguageViewModel: LDLanguageViewModel? = null


    private var mAudioManger: AudioManager? = null
    private var afChangeListener: AudioManager.OnAudioFocusChangeListener? = null

    private var currentTranslateStatus: CommunicationActivity.TranslateStatus =
        CommunicationActivity.TranslateStatus.EMPTY

    private val stsEngine: STSEngine by lazy {
        STSEngine.Builder(
            context = application,
            speechLanguage = "cmn-Hans-CN",
            eventListener = this
        ).build()
    }


    private var sourceError: String = ""
    private var targetError: String = ""
    private var voiceSourceError: String = ""
    private var voicetargetError: String = ""
    private var voiceSourceBusy: String = ""
    private var voiceTargetBusy: String = ""
    private var voiceSourceNoSpeechPut: String = ""
    private var voiceTargetNoSpeechPut: String = ""
    private var sourceListening: String = ""
    private var targetListening: String = ""
    private var sourceRecognizing: String = ""
    private var targetRecognizing: String = ""
    private var sourceTranslating: String = ""
    private var targetTranslating: String = ""

    ////////
    private val composite by lazy { io.reactivex.disposables.CompositeDisposable() }

    private var isinit = false
    private var mLatestHistory: Disposable? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        if (AppUtil.isHiOS()) {
            this.setTheme(R.style.Theme_Translate)
        } else {
            this.setTheme(R.style.XOSTheme_Translate)
        }
    }

    override fun getContentView(): Int = R.layout.mol_activity_conversation_v2

    override fun skipOrientation(): Boolean {
        // 中等宽度 以上的 屏幕
        if (FoldUtils.isTabletType(this)) {
            // CUT: 2022/11/10 设计文稿 不区分横竖屏,测试让适配横竖屏
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_FULL_USER
        } else {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        }

        return true
    }

    override fun onResume() {
        super.onResume()
        viewModel.registerCall()
        inputDialogResume()
    }

    override fun onPause() {
        mTextTranslateFragment?.pause()
        super.onPause()
        stsEngine.cancelSpeech()
        viewModel.stopTranslateVoice()
        viewModel.unregisterCall()
    }

    override fun initView() {
        Log.d(TAG, "onCreate")
//        val darkTheme = AppUtil.isDarkTheme(applicationContext)
//        viewModel.isDark.postValue(darkTheme)

//        isChangeLanguage()

        mBinding.faceList.let { recycler ->
            if (null == mConversationAdapter) {
                mConversationAdapter = ConversationAdapter(getListData(), viewModel).apply {
                    recycler.layoutManager = MyLinearLayoutManager(applicationContext)
                    recycler.addItemDecoration(
                        HistoryAdapter.SpaceItemDecoration(
                            SizeUtils.dp2px(16F), SizeUtils.dp2px(20F)
                        )
                    )
                    recycler.adapter = this
                }

                updateDbData()
            }
        }
        Locale.getDefault().toLanguageTag().let {
            if (it.contains("-")) {
                mBinding.faceTextInput.text =
                    R.string.v_enten_text.getString(this,it.split("-")[0], "Enter the text")
            } else {
                mBinding.faceTextInput.text = R.string.v_enten_text.getString(this,it, "Enter the text")
            }
        }
    }

    override fun finish() {
        super.finish()
        mLatestHistory?.apply {
            if (!isDisposed) {
                dispose()
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        mLatestHistory?.apply {
            if (!isDisposed) {
                dispose()
            }
        }
        dismissLanguageDialog()
    }

     fun dismissLanguageDialog() {
        mLanguageSelectDialog?.dismiss()
        mLanguageSelectDialog = null
    }

    private fun updateDbData() {
        lifecycleScope.launch(Dispatchers.IO) {
            if (null == mLatestHistory) {
                RoomHelper.historyDao.clearHistory()
            }
            mLatestHistory?.apply {
                if (!isDisposed) {
                    dispose()
                }
            }
            mLatestHistory =
                RoomHelper.historyDao.queryLatestHistory()
                    .subscribeOn(Schedulers.io())
                    .subscribe {
                        val order = ArrayList<History>()
                        val size = it.size
                        for (i in size - 1 downTo 0) {
                            order.add(it[i])
                        }
                        lifecycleScope.launch(Dispatchers.Main) {
                            mConversationAdapter?.apply {
                                if (order.isEmpty()) {
                                    order.addAll(getListData())
                                }
                                updateList(order, notifyData = false)
                                if (!isinit && itemCount > 1) {
                                    mBinding.faceList.scrollToPosition(itemCount - 1)
                                }
                                cusNotifyDataSetChanged()
                                if (!isinit) {
                                    isinit = true
                                }
                            }
                        }
                    }
        }
    }


    override fun init() {
        mBinding.viewModel = viewModel
        initObserver()
        initListener()
        mAudioManger = this.getSystemService(AUDIO_SERVICE) as AudioManager

        DataAnalysisHelper.logEvent(this, Face_enter)
    }

    private fun getListData(): ArrayList<History> {
        return ArrayList<History>().apply {
            viewModel.mGuideUtil?.leftGuide?.let {
                add(it)
            }
            viewModel.mGuideUtil?.rightGuide?.let {
                add(it)
            }
        }
    }


    private fun initObserver() {
        // 我说文案
        viewModel.sourceLanguageLiveData.observe(this, Observer {
            viewModel.lastSelectType = CommunicationActivity.SpeechType.SOURCE
//            mBinding.tvSourceLanguage.text = Constant.hiMap[it.toLanguageTag()]
            sourceError = R.string.v_error.getString(this,it.toLanguageTag())
            sourceListening = R.string.v_listen.getString(this,it.toLanguageTag())
            voiceSourceError = R.string.v_speech_recognition.getString(this,it.toLanguageTag())
            sourceTranslating = R.string.v_translating.getString(this,it.toLanguageTag())
            voiceSourceBusy = R.string.v_recognition_busy.getString(this,it.toLanguageTag())
            voiceSourceNoSpeechPut = R.string.v_speech_no_put.getString(this,it.toLanguageTag())
            sourceRecognizing = R.string.v_recognizing.getString(this,it.toLanguageTag())
        })


        // 他说文案
        viewModel.targetLanguageLiveData.observe(this, Observer {
//            mBinding.tvTargetLanguage.text = Constant.hiMap[it.toLanguageTag()]
            targetError = R.string.v_error.getString(this,it.toLanguageTag())
            targetListening = R.string.v_listen.getString(this,it.toLanguageTag())
            voicetargetError = R.string.v_speech_recognition.getString(this,it.toLanguageTag())
            targetTranslating = R.string.v_translating.getString(this,it.toLanguageTag())
            voiceTargetBusy = R.string.v_recognition_busy.getString(this,it.toLanguageTag())
            voiceTargetNoSpeechPut = R.string.v_speech_no_put.getString(this,it.toLanguageTag())
            targetRecognizing = R.string.v_recognizing.getString(this,it.toLanguageTag())
        })

        viewModel.txtTranslateLiveData.observe(this, Observer { event ->
            val pair: Pair<String, Boolean> = event.getContentIfNotHandled() ?: return@Observer
            val text = pair.first
            val isLeft = pair.second
            mConversationAdapter?.addTranslating(
                isLeft,
                text,
                if (isLeft) sourceTranslating else targetTranslating,
                mBinding.faceList
            )
            //
            val src = viewModel.sourceLanguageLiveData.value?.toLanguageTag() ?: return@Observer
            val dst = viewModel.targetLanguageLiveData.value?.toLanguageTag() ?: return@Observer
            val srcParam = if (isLeft) src else dst

            val dstParam = if (isLeft) dst else src

            viewModel.sourceResult.postValue(text)
            viewModel.currentSpeech =
                if (isLeft) CommunicationActivity.SpeechType.SOURCE else CommunicationActivity.SpeechType.TARGET
            viewModel.saveLanguageUsed(srcParam, true)
            viewModel.saveLanguageUsed(dstParam, false)
            TranslateService.Companion.getInstance()
                .translateFromText(text, srcParam, dstParam)
                .observe(this, Observer {
                    viewModel.textTranslateResult.postValue(Event(it))
                })
        })

        viewModel.textTranslateResult.observe(this, Observer { event ->
            event.getContentIfNotHandled()?.let {
                translateServiceCompleteResult(
                    it.getOrNull()?.state ?: HiServiceInterface.TRANSLATE_STATE_FAILURE,
                    it.getOrNull()?.translationText,
                    it.getOrNull()?.targetLanguage,
                    it.getOrNull()?.sourceLanguage
                )
            }
        })

        viewModel.voicePlayStatusV2.observe(this, Observer { event ->
            val it = event.getContentIfNotHandled() ?: return@Observer
            Log.i(TAG, "voice play status:$it, position:${viewModel.currentPlayClickIndex}")
            when (it) {
                MainViewModel.VoicePlayStatus.START -> {
//                    mConversationAdapter?.notifyItemChanged(viewModel.currentPlayClickIndex)
                    mConversationAdapter?.cusNotifyDataSetChanged()
                }
                MainViewModel.VoicePlayStatus.PAUSE, MainViewModel.VoicePlayStatus.END -> {
//                    mConversationAdapter?.notifyItemChanged(viewModel.currentPlayClickIndex)
                    mConversationAdapter?.cusNotifyDataSetChanged()
                }
            }
        })

        viewModel.sourceLanguageLiveData.observe(this, Observer {
            mBinding.faceTitleSrc.text = Locale.forLanguageTag(it.toLanguageTag()).displayLanguage
            mBinding.faceISayLanguage.text = Locale.forLanguageTag(it.toLanguageTag()).displayLanguage
        })
        viewModel.targetLanguageLiveData.observe(this, Observer {
            mBinding.faceTitleDst.text = Locale.forLanguageTag(it.toLanguageTag()).displayLanguage
            mBinding.faceHeSayLanguage.text = Locale.forLanguageTag(it.toLanguageTag()).displayLanguage
        })
//        viewModel.resultText.observe(this, Observer {
//            mBinding.tvMainResultText.text = it
//        })

//        viewModel.isResultStart.observe(this, Observer {
//            if (mBinding.tvMainResultStart.isEnabled) {
//                if (it) {
//                    mBinding.tvMainResultStart.setImageResource(R.drawable.mol_ic_collected)
//                } else {
//                    mBinding.tvMainResultStart.setImageResource(R.drawable.mol_ic_collect)
//                }
//            }

//        })

//        viewModel.historyBtn.observe(this, Observer {
//            mBinding.tvHistory.isVisible = it
//        })

        viewModel.enlargeDialogLiveData.observe(this, Observer { event ->
            val it = event.getContentIfNotHandled() ?: return@Observer
            TranslateEnlargeActivity.launch(this, it.target_result.trim(), it.targetlanguage_tag)
        })

        viewModel.updateListEnlarge.observe(this, Observer {
            it?.let {
                mConversationAdapter?.cusNotifyDataSetChanged()
            }
        })

        viewModel.guideLiveData.observe(this, Observer {
            if (!it) {
                return@Observer
            }
            mConversationAdapter?.apply {
                if (itemCount == 0) {
                    updateList(getListData())
                }
            }
        })

        viewModel.checkStopLiveData.observe(this, Observer {
            it.getContentIfNotHandled()?.let {
                if (mBinding.faceISayLayout.getTag(R.id.view_speech_status) == true
                    || mBinding.faceHeSayLayout.getTag(R.id.view_speech_status) == true
                ) {
                    stopVoice()
                    stopRecordUi()
                    recognizing()
                }
            }
        })

        viewModel.hideInputDialogLiveData.observe(this, Observer {
            it.getContentIfNotHandled()?.let { hide ->
                if (hide) {
                    hideInputDialogImpl()
                } else {
                    showInputDialogImpl()
                }
            }
        })
    }

    private fun initOnAudioFocus() {
        if (null != afChangeListener) {
            return
        }
        afChangeListener = AudioManager.OnAudioFocusChangeListener { focusChange ->
            when (focusChange) {
                AudioManager.AUDIOFOCUS_LOSS_TRANSIENT -> {
                    // Pause playback
                    Log.d(
                        CommunicationActivity.TAG,
                        "init: 音頻焦點——————————AUDIOFOCUS_LOSS_TRANSIENT, currentTranslateStatus:$currentTranslateStatus, "
                    )
                    if (currentTranslateStatus != CommunicationActivity.TranslateStatus.LISTENER) {
                        viewModel.stopTranslateVoice()
                    }
                }
                AudioManager.AUDIOFOCUS_LOSS -> {
                    Log.d(CommunicationActivity.TAG, "init: 音頻焦點——————————AUDIOFOCUS_LOSS")
                }
            }
        }

        mAudioManger?.requestAudioFocus(
            afChangeListener,
            AudioManager.STREAM_MUSIC,
            AudioManager.AUDIOFOCUS_GAIN
        )
    }


//    private fun isChangeLanguage() {
//        if (SharedPreferenceHelper.getBooleanValue(
//                SharedPreferenceHelper.IS_CHANGE_LANGUAGE,
//                true
//            )
//        ) {
//            viewModel.languageList = this.loadLanguageList()
//            SharedPreferenceHelper.putBooleanValue(SharedPreferenceHelper.IS_CHANGE_LANGUAGE, false)
//            val sourceLanguage = viewModel.sourceLanguageLiveData.value?.displayLanguage
//            val targetLanguage = viewModel.targetLanguageLiveData.value?.displayLanguage
//            viewModel.sourceLang.value = sourceLanguage
//            viewModel.targetLang.value = targetLanguage
//        }
//    }

    private fun initListener() {
        mBinding.faceTitleClose.setOnClickListener {
            onClickBack()
        }

        mBinding.faceTitleSrc.setOnClickListener {
            onClickChangeSrcLanguage()

        }
        mBinding.faceTitleDst.setOnClickListener {
            onClickChangeDstLanguage()
        }

        mBinding.faceTitleExchange.setOnClickListener {

            onClickExchangeSrcDstLanguage()
        }

        mBinding.faceTitleMore.setOnClickListener {
            onClickMore()

        }

        mBinding.faceISayLayout.setOnClickListener {
            onClickRecordISay(it)

        }
        mBinding.faceHeSayLayout.setOnClickListener {
            onClickRecordHeSay(it)

        }

        mBinding.faceTextInput.setOnClickListener {
            onClickInputText()
        }
        mBinding.faceTextInputIcon.setOnClickListener {
            onClickInputText()
        }
    }

    private fun onClickBack() {
        stopPlayingVoice()
//        stopVoice()
        onBackPressed()
    }

    private fun onClickChangeSrcLanguage() {
        if (isRecording()) {
            Toast.makeText(this, sourceListening, Toast.LENGTH_SHORT).show()
            return
        }
        //1, 当前正在听 -> toast
        //2, 停止播报
        //3,
        showLanguageDialog(true)
    }

    private fun onClickChangeDstLanguage() {
        if (isRecording()) {
            Toast.makeText(this, targetListening, Toast.LENGTH_SHORT).show()
            return
        }
        showLanguageDialog(false)
    }

    private fun showLanguageDialog(isSelectLeft: Boolean) {
        initLanguageDialog().let {
            viewModel.sourceLanguageLiveData.value?.let { local->
                it.sourceLang.value =  Locale.forLanguageTag(local.toLanguageTag()).displayLanguage
            }
            viewModel.targetLanguageLiveData.value?.let { local->
                it.targetLang.value =  Locale.forLanguageTag(local.toLanguageTag()).displayLanguage
            }
            dismissLanguageDialog()
            mLanguageSelectDialog = LanguageSelectDialog()


            mLanguageSelectDialog?.dismiss()

            mLanguageSelectDialog?.show(this, it, isSelectLeft)
        }
    }

    private fun initLanguageDialog(): LDLanguageViewModel {
        mLDLanguageViewModel?.let {
            return it
        }
        return ViewModelProvider(
            this,
            ViewModelProvider.AndroidViewModelFactory.getInstance(application)
        ).get(LDLanguageViewModel::class.java).apply {
            mLDLanguageViewModel = this

            initLanguageList(getOrderedLanguages(Constants.supportLanguages))

            sourceLanguageLiveData.observe(this@CommunicationActivityV2, Observer {
                it?.let {
                    viewModel.sourceLang.value = it.name
                    viewModel.updateSourceLanguage(
                        languageTag = it.code,
                        languageName = it.name
                    )
                }
            })
            targetLanguageLiveData.observe(this@CommunicationActivityV2, Observer {
                it?.let {
                    viewModel.targetLang.value = it.name
                    viewModel.updateTargetLanguage(
                        languageTag = it.code,
                        languageName = it.name
                    )
                }
            })
        }
    }

    private fun onClickExchangeSrcDstLanguage() {
        if (isRecording()) {
            Toast.makeText(this, sourceListening, Toast.LENGTH_SHORT).show()
            return
        }
        val selectedSrcName = viewModel.sourceLang.value
        val selectedDstName = viewModel.targetLang.value
        val selectedSrc = viewModel.sourceLanguageLiveData.value
        val selectedDst = viewModel.targetLanguageLiveData.value

        viewModel.sourceLang.value = selectedDstName
        viewModel.targetLang.value = selectedSrcName

        viewModel.sourceLanguageLiveData.value = selectedDst
        viewModel.targetLanguageLiveData.value = selectedSrc

        mLDLanguageViewModel?.updateSourceLanguage(LanguageInfo(selectedDst!!.language, selectedDstName))
        mLDLanguageViewModel?.updateTargetLanguage(LanguageInfo(selectedSrc!!.language, selectedSrcName))

    }

    private fun isRecording(): Boolean {
        val recording = isRecordingStatus()
        val tagISay = mBinding.faceISayLayout.getTag(R.id.view_speech_status)
        val tagHeSay = mBinding.faceHeSayLayout.getTag(R.id.view_speech_status)
        if (recording && tagISay == false && tagHeSay == false) {
            //状态不对， 可能由于音频回收慢导致. 再次stop下录音.
            Log.i(TAG, "isRecording, status($currentTranslateStatus) error.")
            safe {
                stopVoice()
            }
        }

        return recording
                || tagISay == true
                || tagHeSay == true
    }

    private fun safe(function: (() -> Unit)?) {
        try {
            function?.invoke()
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun onClickMore() {
        if (viewModel.currentSpeech == CommunicationActivity.SpeechType.SOURCE) {
            if (isRecordingStatus()) {
                Toast.makeText(this, sourceListening, Toast.LENGTH_SHORT).show()
                return
            }
        } else {
            if (isRecordingStatus()) {
                Toast.makeText(this, targetListening, Toast.LENGTH_SHORT).show()
                return
            }
        }

        stopPlayingVoice()

        updateDbData()

        val intent = Intent(this, StarMarkActivity::class.java)
        intent.putExtra(Constant.HISTORY_ID, "")
        startActivityForResult(intent, REQUEST_CODE)
    }

    private fun isRecordingStatus():Boolean {
        return currentTranslateStatus == CommunicationActivity.TranslateStatus.LISTENER
    }


    private fun onClickRecordISay(v: View) {
        //开始录音
        val isOk = startTargetVoice(v, CommunicationActivity.SpeechType.SOURCE)
        if (isOk) {
            viewModel.currentLangType = MainViewModel.LanguageType.SOURCE

            mBinding.faceISayLanguage.setText(R.string.mol_face_click_translate)

            viewModel.targetLang.value?.let {
                mBinding.faceHeSayLanguage.text = it
            }
            mBinding.faceHeSayLayout.isEnabled = false
        }
        DataAnalysisHelper.logEvent(this, FA_voice_input_success, hashMapOf("type" to "1"))
    }

    private fun onClickRecordHeSay(v: View) {
        val isOk = startTargetVoice(v, CommunicationActivity.SpeechType.TARGET)
        if (isOk) {
            viewModel.currentLangType = MainViewModel.LanguageType.TARGET

            mBinding.faceHeSayLanguage.setText(R.string.mol_face_click_translate)

            viewModel.sourceLang.value?.let {
                mBinding.faceISayLanguage.text = it
            }
            mBinding.faceISayLayout.isEnabled = false
        }
        DataAnalysisHelper.logEvent(this, FA_voice_input_success, hashMapOf("type" to "2"))
    }

    private fun onClickInputText() {
        if (isRecording()) {
            Toast.makeText(this, sourceListening, Toast.LENGTH_SHORT).show()
            return
        }
        if (!NetWorkUtil.getActiveNetworkConnected()) {
            Toast.makeText(this, R.string.main_net_error, Toast.LENGTH_SHORT).show()
            DataAnalysisHelper.logEvent(this, FA_keyboard_input_fail)
        } else {
            DataAnalysisHelper.logEvent(this, FA_keyboard_input_success)
        }

        stopPlayingVoice()

        mTextTranslateFragment = TextTranslateFragment.newInstance()
        mTextTranslateFragment?.show(this, layoutInflater, mBinding.faceDialogLayout, viewModel) ?: return

    }

    private var mTextTranslateFragment: TextTranslateFragment? = null

    private fun showInputDialogImpl() {
        mBinding.faceDialogDimAmount.isVisible = true
        mBinding.faceDialogLayout.isVisible = true
        mBinding.faceDialogDimAmount.isFocusable = true
        mBinding.faceDialogDimAmount.isClickable = true


        val isDark = this.isDarkFontDefault()
        setDarkMode(
            R.color.face_dialog_dim_amount_bg_color,
            R.color.face_color_bottom_dialog,
            isDark
        )

    }

    private fun hideInputDialogImpl() {
//        hideAnimation(mBinding.faceDialogDimAmount)
        mBinding.faceDialogLayout.isVisible = false
        mBinding.faceDialogDimAmount.isVisible = false
        setDarkMode(getStatusBarColorResId(), getNavigationBarColorResId(), isDarkFont())
    }

    private fun isInputDialogShowing(): Boolean {
        return mBinding.faceDialogDimAmount.isVisible
    }

    override fun onBackPressed() {
        if (mTextTranslateFragment?.isShowing() == true) {
            mTextTranslateFragment?.dismiss()
            return
        }

        if (isInputDialogShowing()) {
            hideInputDialogImpl()
            return
        }
        super.onBackPressed()
    }


    private var isShow = false

    private fun inputDialogResume() {
        if (isShow) {
            isShow = false
        } else {
            mTextTranslateFragment?.resume()
        }
    }

    override fun onRestoreInstanceState(savedInstanceState: Bundle) {
        super.onRestoreInstanceState(savedInstanceState)
         isShow = savedInstanceState.getBoolean("isShow", false)
       val inputText = savedInstanceState.getString("inputText", "")
        val inputSelection = savedInstanceState.getInt("inputSelection", 0)
        val isKeyboardShown = savedInstanceState.getBoolean("isKeyboardShown", true)
        val isSelectLeft = savedInstanceState.getBoolean("isSelectLeft", true)
        Log.d("Sky", "onRestoreInstanceState, isShow:$isShow")
        if (isShow) {
            Log.d("Sky", "onRestoreInstanceState, inputText:$inputText")
            Log.d("Sky", "onRestoreInstanceState, inputSelection:$inputSelection")
            Log.d("Sky", "onRestoreInstanceState, isKeyboardShown:$isKeyboardShown")
            Log.d("Sky", "onRestoreInstanceState, isSelectLeft:$isSelectLeft")
            mTextTranslateFragment = TextTranslateFragment.newInstance()
            mTextTranslateFragment?.show(
                this,
                layoutInflater,
                mBinding.faceDialogLayout,
                viewModel,
                inputText,
                inputSelection,
                isKeyboardShown,
                isSelectLeft
            )

        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        mTextTranslateFragment?.let {
            val isShow = it.isShowing()
            val inputText = it.inputText()
            val inputSelection = it.inputSelection()
            val isKeyboardShown = it.isKeyboardShown()
            val isSelectLeft = it.isSelectLeft()
            outState.putBoolean("isShow", isShow)
            outState.putString("inputText", inputText)
            outState.putInt("inputSelection", inputSelection)
            outState.putBoolean("isKeyboardShown", isKeyboardShown)
            outState.putBoolean("isSelectLeft", isSelectLeft)

            Log.d("Sky", "onSaveInstanceState, isShow:$isShow")
            Log.d("Sky", "onSaveInstanceState, inputText:$inputText")
            Log.d("Sky", "onSaveInstanceState, inputSelection:$inputSelection")
            Log.d("Sky", "onSaveInstanceState, isKeyboardShown:$isKeyboardShown")
            Log.d("Sky", "onSaveInstanceState, isSelectLeft:$isSelectLeft")

        }
    }

    /**
     * 停止播报
     */
    private fun stopPlayingVoice() {
        viewModel.stopTranslateVoice()
    }

    /**
     * 开始语音说话
     */
    private fun startTargetVoice(v: View, speechType: CommunicationActivity.SpeechType): Boolean {
        val isHlsInstall = AppUtil.checkApkExist(applicationContext, Constants.HLS_PACK_NAME)
        if (!isHlsInstall) {
            Toast.makeText(this, R.string.bing_servive_failure, Toast.LENGTH_SHORT).show()
            return false
        }
        val speechStatus = v.getTag(R.id.view_speech_status)
        if (speechStatus == true) { //正在speech，停止录音
            stopVoice()
            stopRecordUi()
            recognizing()
            return false
        } else {
            // 权限
            if (!checkAppPermissions()) {
                return false
            }

            if (!NetWorkUtil.getActiveNetworkConnected()) {
                Toast.makeText(this, R.string.main_net_error, Toast.LENGTH_SHORT).show()
                DataAnalysisHelper.logEvent(
                    this,
                    FA_voice_input_fail,
                    hashMapOf("errorcode" to "1")
                )
                return false
            }


            if (checkAudioManger()) {
                Toast.makeText(this, R.string.Audio_busy, Toast.LENGTH_SHORT).show()
                return false
            }

//            if (!AppUtil.validateMicAvailability()) {
//                Toast.makeText(this, R.string.Audio_busy, Toast.LENGTH_SHORT).show()
//                return
//            }

            if (AppUtil.phoneIsInUses()) {
                Toast.makeText(this, R.string.Audio_busy, Toast.LENGTH_SHORT).show()
                return false
            }


            //处于未说话状态,开始录音
            //stopVoice()

            viewModel.currentSpeech = speechType
            setSpeechState(true)
            setLanguage()
            startVoice()
            return true
        }
    }

    private fun setSpeechState(isRecording: Boolean = false) {
        mBinding.faceISayLayout.setTag(R.id.view_speech_status, isRecording)
        mBinding.faceHeSayLayout.setTag(R.id.view_speech_status, isRecording)
        Log.v(TAG, "setSpeechState:isRecording:$isRecording")

        if (!isRecording) {
            mBinding.faceISayLayout.isEnabled = true
            mBinding.faceHeSayLayout.isEnabled = true

            viewModel.targetLang.value?.let {
                mBinding.faceHeSayLanguage.text = it
            }

            viewModel.sourceLang.value?.let {
                mBinding.faceISayLanguage.text = it
            }
        }
    }

    private fun youSayStartRecodingAnimation() {
//        mBinding.tvMainLanguageYouSay.setImageResource(R.drawable.mol_recording_start_anim)
//        youSayAnimationDrawable = mBinding.tvMainLanguageYouSay.drawable as AnimationDrawable
//        youSayAnimationDrawable.start()
        mConversationAdapter?.apply {
            addSaying(true, Constant.LISTENER_TIME.toInt(), mBinding.faceList)
        }
    }

    private fun heSayStartRecodingAnimation() {
//        mBinding.tvMainLanguageHeSay.setImageResource(R.drawable.mol_recording_hesay_anim)
//        heSayAnimationDrawable = mBinding.tvMainLanguageHeSay.drawable as AnimationDrawable
//        heSayAnimationDrawable.start()
        mConversationAdapter?.apply {
            addSaying(false, Constant.LISTENER_TIME.toInt(), mBinding.faceList)
        }
    }

    /**
     * 开始录音
     */
    private fun startVoice() {
        initOnAudioFocus()
        if (viewModel.currentSpeech == CommunicationActivity.SpeechType.SOURCE) {
            youSayStartRecodingAnimation()
        } else {
            heSayStartRecodingAnimation()
        }
        // 开始倒计时
        showListenerUi()
        viewModel.stopTranslateVoice()
        //开始录音
        stsEngine.cancelSpeech()
        stsEngine.startSpeech()
        setSpeechState(isRecording = true)
    }

    /**
     * 停止录音UI
     */

    private fun stopRecordUi() {
        composite.clear()
        setSpeechState(false)
    }

    /**
     * 正在听
     */
    private fun showListenerUi() {
        composite.clear()
        val subscribe = Observable.interval(0, 1, TimeUnit.SECONDS).take(Constant.LISTENER_TIME + 1)
            .observeOn(AndroidSchedulers.mainThread()).doOnSubscribe {

                updateStatus(CommunicationActivity.TranslateStatus.LISTENER)
//                if (viewModel.currentSpeech == CommunicationActivity.SpeechType.SOURCE) {
//                    mBinding.tvTargetLanguage.visibility = View.GONE
//                    mBinding.tvSourceLanguage.visibility = View.VISIBLE
//                    mBinding.tvSourceLanguage.text =
//                        "$sourceListening...(${Constant.LISTENER_TIME})"

//                } else {
//                    mBinding.tvSourceLanguage.visibility = View.GONE
//                    mBinding.tvTargetLanguage.visibility = View.VISIBLE
//                    mBinding.tvTargetLanguage.text =
//                        "$targetListening...(${Constant.LISTENER_TIME})"
//                }
            }.doOnComplete {
                //倒计时结束
                setSpeechState(true)
                stopVoice()
            }.doOnError {
                setSpeechState(false)
                it.printStackTrace()
            }.subscribe {
                Log.i(TAG, "倒计时: it:$it")
                setSpeechState(true)
                mConversationAdapter?.updateCountDown(
                    (Constant.LISTENER_TIME - it).toInt(),
                    mBinding.faceList
                )
//                if (viewModel.currentSpeech == CommunicationActivity.SpeechType.SOURCE) {
//                    mBinding.tvSourceLanguage.visibility = View.VISIBLE
//                    mBinding.tvTargetLanguage.visibility = View.GONE
//                    mBinding.tvSourceLanguage.text =
//                        "$sourceListening...(${Constant.LISTENER_TIME - it})"
//                } else {
//                    mBinding.tvTargetLanguage.visibility = View.VISIBLE
//                    mBinding.tvSourceLanguage.visibility = View.GONE
//                    mBinding.tvTargetLanguage.text =
//                        "$targetListening...(${Constant.LISTENER_TIME - it})"
//                }

                if (it == Constant.LISTENER_TIME) {
//                    mBinding.tvSourceLanguage.visibility = View.GONE
//                    mBinding.tvTargetLanguage.visibility = View.GONE
//                    showResultUi()
                    setSpeechState(false)
                }

            }
        composite.add(subscribe)
    }

    private fun updateStatus(status:CommunicationActivity.TranslateStatus) {
        currentTranslateStatus = status
        Log.v(TAG, "----------------------------------------")
        Log.i(TAG, "status:$status, :::")//${Log.getStackTraceString(Throwable())}
        Log.v(TAG, "----------------------------------------")

    }

    /**
     * 所有权限处理
     */
    private fun checkAppPermissions(): Boolean {
        if (!checkPermission(Manifest.permission.RECORD_AUDIO)
            || !checkPermission(Manifest.permission.READ_PHONE_STATE)) {
            //logEvent(this, FA_Recording_permission)
            requestPermissions(arrayOf(Manifest.permission.RECORD_AUDIO, Manifest.permission.READ_PHONE_STATE), 200)
            return false
        }
        return true
    }

    /**
     * 音频通道是否被占用
     */
    private fun checkAudioManger(): Boolean {
        return mAudioManger?.activeRecordingConfigurations?.isNotEmpty() == true
    }

    /**
     * 停止录音
     */
    private fun stopVoice() {
        Log.d(CommunicationActivity.TAG, "stopVoice:停止录音 ")
        //停止录音
        stsEngine.stopSpeech()
    }

    /**
     * 识别中...
     */
    private fun recognizing() {
        when (viewModel.currentSpeech) {
            CommunicationActivity.SpeechType.SOURCE -> {
                mConversationAdapter?.recordError(sourceRecognizing, mBinding.faceList)
            }
            CommunicationActivity.SpeechType.TARGET -> {
                mConversationAdapter?.recordError(targetRecognizing, mBinding.faceList)
            }
        }
    }

    /**
     * 对引擎设置语言
     */
    private fun setLanguage() {
        val sourceLocale = viewModel.sourceLanguageLiveData.value ?: return//getSourceLocale()
        val targetLocale = viewModel.targetLanguageLiveData.value ?: return//getTargetLocale()
        when (viewModel.currentSpeech) {
            CommunicationActivity.SpeechType.SOURCE -> {
                stsEngine.setLanguage(
                    speechLanguage = sourceLocale.language
                )
            }
            CommunicationActivity.SpeechType.TARGET -> {
                stsEngine.setLanguage(
                    speechLanguage = targetLocale.language
                )
            }
        }
    }

    /**
     * 开始文字翻译
     */
    private fun startTextTranslate(text: String) {
        viewModel.sourceResult.postValue(text)
        stopRecordUi()
//        showResultUi()
        viewModel.translateText(text).observe(this) {
            translateServiceCompleteResult(
                it.getOrNull()?.state ?: HiServiceInterface.TRANSLATE_STATE_FAILURE,
                it.getOrNull()?.translationText,
                it.getOrNull()?.targetLanguage,
                it.getOrNull()?.sourceLanguage
            )
        }
    }

    /**
     * 翻译的回调
     * stateCode 状态码
     * result 翻译结果
     */

    private fun translateServiceCompleteResult(
        stateCode: Int,
        result: String?,
        targetLanguages: String?,
        sourceLanguages: String?
    ) {
        if (targetLanguages.isNullOrEmpty()) {
            translaterFailure()
            return
        }
//        viewModel.isResultStart.value = false

        Log.i(TAG, "translate result--->state:$stateCode result:$result " +
                "targetLanguages:$targetLanguages sourceLanguages:$sourceLanguages")

        lifecycleScope.launch(Dispatchers.IO) {
            when (stateCode) {
                HiServiceInterface.TRANSLATE_STATE_SUCCESS -> { //成功
                    val history = roomViewModel.insertHistoryList(
                        result ?: "",
                        viewModel,
                        sourceLanguages,
                        targetLanguages
                    )
//                    viewModel.resultText.postValue(result)
                    withContext(context = Dispatchers.Main) {
                        handleTranslateSuccessUI(history)
                    }

                }
                HiServiceInterface.TRANSLATE_STATE_FAILURE -> { // 失败
                    withContext(Dispatchers.Main) {
                        translaterFailure()
                    }
                }
            }
        }
    }

    private fun handleTranslateSuccessUI(history: History?) {
        mConversationAdapter?.addSayingResult(history, mBinding.faceList)
    }

    /**
     * 翻译失败
     */
    private fun translaterFailure() {
//        textResultShow(false)
        if (viewModel.isTextTranslate) {
            val text =
                R.string.v_text_translat_error.getString(this,
                    viewModel.sourceLangType,
                    "Please try again later"
                )
            Log.i(TAG, "translaterFailure, text:$text, type:${viewModel.sourceLangType}")
            mConversationAdapter?.recordError(text, mBinding.faceList)
//            viewModel.resultText.postValue(text)
        } else {
            if (viewModel.currentSpeech == CommunicationActivity.SpeechType.SOURCE) {
//                viewModel.resultText.postValue(sourceError)
                mConversationAdapter?.recordError(sourceError, mBinding.faceList)
            } else {
//                viewModel.resultText.postValue(targetError)
                mConversationAdapter?.recordError(targetError, mBinding.faceList)
            }
        }
    }

    /**
     * 显示结果页
     * quickShowText 文字翻译显示源语言
     */

//    private fun showResultUi(quickShowText: Boolean = false, sourceLangType: String = "") {
//        if (quickShowText) {
//            if (sourceLangType == viewModel.targetLanguageLiveData.value?.language) {
//                viewModel.isSource.postValue(false)
//            } else {
//                viewModel.isSource.postValue(true)
//            }
//        } else {
//            viewModel.isSource.postValue(viewModel.currentSpeech == CommunicationActivity.SpeechType.SOURCE)
//        }
//
//    }


    // 语音识别失败
    private fun errorRecordUi(error: Int) {
        updateStatus(CommunicationActivity.TranslateStatus.EMPTY)
        composite.clear()
        setSpeechState(false)

        if (viewModel.currentSpeech == CommunicationActivity.SpeechType.SOURCE) {
//            mBinding.tvSourceLanguage.visibility = View.VISIBLE
//            mBinding.tvTargetLanguage.visibility = View.GONE
            when (error) {
                SpeechHelper.ERROR_NO_MATCH -> showSoureceLineTextView(voiceSourceError)
                SpeechHelper.ERROR_RECOGNIZER_BUSY -> showSoureceLineTextView(voiceSourceBusy)
                SpeechHelper.ERROR_NO_SPEECH_INPUT -> showSoureceLineTextView(voiceSourceNoSpeechPut)
                else -> showSoureceLineTextView(voiceSourceError)
            }

        } else {
//            mBinding.tvSourceLanguage.visibility = View.GONE
//            mBinding.tvTargetLanguage.visibility = View.VISIBLE
            when (error) {
                SpeechHelper.ERROR_NO_MATCH -> showTargetLineTextView(voicetargetError)
                SpeechHelper.ERROR_RECOGNIZER_BUSY -> showTargetLineTextView(voiceTargetBusy)
                SpeechHelper.ERROR_NO_SPEECH_INPUT -> showTargetLineTextView(voiceTargetNoSpeechPut)
                else -> showTargetLineTextView(voicetargetError)
            }
        }

    }

    private fun showSoureceLineTextView(errorMsg: String) {
        mConversationAdapter?.recordError(errorMsg, mBinding.faceList)
    }

    private fun showTargetLineTextView(errorMsg: String) {
        mConversationAdapter?.recordError(errorMsg, mBinding.faceList)
    }


    override fun onEvent(code: Int, bundle: Bundle?) {
        when (code) {
            SpeechHelper.CALLBACK_STT_READY -> { //准备识别
                Logcat.d("EventListener", "CALLBACK_STT_READY")
                // 震动
                VibrateUtils.vibrate(100)
            }
            SpeechHelper.CALLBACK_STT_BEGIN -> { //开始识别
                Logcat.d("EventListener", "CALLBACK_STT_BEGIN")
                updateStatus(CommunicationActivity.TranslateStatus.LISTENER)
                setSpeechState(isRecording = true)
            }
            SpeechHelper.CALLBACK_STT_PARTIAL -> { //识别部分结果返回，同时根据参数是否进行翻译结果返回
                val text = bundle?.getString(SpeechHelper.EXTRA_STT_PARTIAL)
                if (!text.isNullOrEmpty()) {
                    Logcat.d("EventListener", "CALLBACK_STT_PARTIAL  text=$text")
                }
                setSpeechState(isRecording = true)
            }
            SpeechHelper.CALLBACK_STT_END -> { //识别结束
                Logcat.d("EventListener", "CALLBACK_STT_END")
                setSpeechState(isRecording = false)
                updateStatus(CommunicationActivity.TranslateStatus.EMPTY)
            }
            SpeechHelper.CALLBACK_STT_FINISH -> { //识别完成，同时根据参数是否返回播报音频数据
                updateStatus(CommunicationActivity.TranslateStatus.COMPLATE)
                val text = bundle?.getString(SpeechHelper.EXTRA_STT_PARTIAL)
                if (text != null && text.isNotBlank()) {
                    viewModel.isTextTranslate = false
                    startTextTranslate(text)
                }
                Logcat.d("EventListener", "CALLBACK_STT_FINISH  text=$text")
                setSpeechState(isRecording = false)
                mConversationAdapter?.apply {
                    isLeftSaying?.let { isLeft ->
                        addTranslating(
                            isLeft,
                            text,
                            if (isLeft) sourceTranslating else targetTranslating,
                            mBinding.faceList
                        )
                    }
                }
            }

            SpeechHelper.CALLBACK_ERROR -> { //发生错误
                setSpeechState(isRecording = false)
                updateStatus(CommunicationActivity.TranslateStatus.EMPTY)
                // 如果当前翻译成功，显示结果UI
                if (currentTranslateStatus == CommunicationActivity.TranslateStatus.COMPLATE) return

                val error = bundle?.getInt(SpeechHelper.EXTRA_ERROR_CODE)
                Logcat.d("EventListener", "CALLBACK_ERROR  error=$error")
                // 识别错误码
                when (error) {
                    /** Network operation timed out. */
                    SpeechHelper.ERROR_NETWORK_TIMEOUT -> {
                        Toast.makeText(this, R.string.main_net_error, Toast.LENGTH_SHORT).show()
                        errorRecordUi(SpeechHelper.ERROR_NETWORK_TIMEOUT)
                    }

                    /** Other network related errors. */
                    SpeechHelper.ERROR_NETWORK -> {

                        Toast.makeText(this, R.string.main_net_error, Toast.LENGTH_SHORT).show()
                        errorRecordUi(SpeechHelper.ERROR_NETWORK)
                    }

                    /** Audio recording error. */
                    SpeechHelper.ERROR_AUDIO -> {
                        errorRecordUi(SpeechHelper.ERROR_AUDIO)
                    }

                    /** Server sends error status. */
                    SpeechHelper.ERROR_SERVER -> {
                        errorRecordUi(SpeechHelper.ERROR_SERVER)
                    }

                    /** Other client side errors. */
                    SpeechHelper.ERROR_CLIENT -> {
                        errorRecordUi(SpeechHelper.ERROR_CLIENT)
                    }

                    /** No speech input */
                    SpeechHelper.ERROR_NO_SPEECH_INPUT -> {
                        errorRecordUi(SpeechHelper.ERROR_NO_SPEECH_INPUT)
                    }

                    /** No recognition result matched. */
                    SpeechHelper.ERROR_NO_MATCH -> {
                        errorRecordUi(SpeechHelper.ERROR_NO_MATCH)
                    }

                    /** RecognitionService busy. */
                    SpeechHelper.ERROR_RECOGNIZER_BUSY -> {
                        errorRecordUi(SpeechHelper.ERROR_RECOGNIZER_BUSY)
                    }

                    /** Insufficient permissions */
                    SpeechHelper.ERROR_INSUFFICIENT_PERMISSIONS -> {
                    }

                    else -> { //其它错误
                        errorRecordUi(SpeechHelper.ERROR_OTHER)
                    }

                }
            }
            SpeechHelper.CALLBACK_STT_VOLUME -> { //声波振幅
//                val volume = bundle?.getFloat(SpeechHelper.EXTRA_STT_VOLUME) ?: 0f
//                var volumePercent = bundle?.getInt(SpeechHelper.EXTRA_STT_VOLUME_PERCENT) ?: 1
//                volumePercent = if (volumePercent <= 0) 1 else volumePercent
//                Logcat.d(
//                    "EventListener",
//                    "CALLBACK_STT_VOLUME  volume=$volume  volumePercent=$volumePercent"
//                )
            }
        }
    }

    // 结果页是否start
//    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
//        super.onActivityResult(requestCode, resultCode, data)
//        if (requestCode == REQUEST_CODE && resultCode == Activity.RESULT_OK) {
//            data?.let {
//                val isStar = data.getBooleanExtra(Constant.IS_STAR_MARK, false)
//                viewModel.isResultStart.value = isStar
//            }
//        }
//    }

//    internal fun onTextTranslateFragmentDismiss(isTranslate: Boolean, sourceLangType: String) {
//        if (isTranslate) {
//            viewModel.isTextTranslate = isTranslate
//            viewModel.sourceLangType = sourceLangType
//            showResultUi(true, sourceLangType)
//        }
//    }

}

fun getOrderedLanguages(vararg languages: String): List<String> {
    val supportList = StringBuilder()
        .also { builder ->
            languages.forEach {
                builder.append(it)
                builder.append(",")
            }
        }.toString()

    val tags: Array<String> =
        supportList.split(",".toRegex()).dropLastWhile { it.isEmpty() || it == "," }.toTypedArray()

    val result = ArrayList<String>()
    tags.forEach {
        result.add(it)
    }
    return result
}