package com.talpa.master.ui.main

import android.app.Application
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.SharedPreferences
import android.os.Bundle
import android.util.ArraySet
import android.util.Log
import android.util.SparseArray
import android.view.View
import android.widget.Toast
import androidx.core.os.LocaleListCompat
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import com.blankj.utilcode.util.AppUtils
import com.hiservice.aidl.UtteranceProgressCallback
import com.hiservice.connect.HiServiceConnection
import com.hiservice.connect.isLanguageAvailable
import com.hiservice.connect.registerUtteranceProgress
import com.hiservice.connect.unregisterUtteranceProgress
import com.hiservice.text2speech.core.TAG
import com.talpa.master.R
import com.talpa.master.repository.*
import com.talpa.master.repository.room.model.History
import com.talpa.master.repository.room.model.LanguageModel
import com.talpa.master.ui.main.v2.GuideUtil
import com.talpa.master.utils.AppUtil
import com.talpa.master.utils.event.Event
import com.talpa.translate.face.CompleteTextResult
import com.talpa.translate.face.TranslateService
import com.talpa.translate.face.common.Constants
import com.talpa.translate.face.common.CustomLocale
import com.talpa.translate.framework.Resource
import com.transsion.trancare.Athena
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.*


/**
 * Create by kgxl on 2021/4/5
 */
open class MainViewModel(private val ctx: Application) :
    AndroidViewModel(ctx) {

    companion object {
        private const val SP_NAME = "sp_language_src_dst"
        private const val KEY_SRC = "key_src"
        private const val KEY_DST = "key_dst"
    }

    private var mSp: SharedPreferences? = null

    //    private val srcUsedList = Collections.synchronizedList(ArrayList<LanguageModel.LanguageList>())
//    private val dstUsedList = Collections.synchronizedList(ArrayList<LanguageModel.LanguageList>())
    private val srcUsedList = Collections.synchronizedList(ArrayList<String>())
    private val dstUsedList = Collections.synchronizedList(ArrayList<String>())

    var isDark = MutableLiveData<Boolean>()

//    var isHios = MutableLiveData<Boolean>()

    val sourceLanguageLiveData = MutableLiveData<Locale>()
    val targetLanguageLiveData = MutableLiveData<Locale>()

    var languageList = ctx.loadLanguageList()
    var isTextTranslate: Boolean = false  // 是否是文字翻译
//    var isVoicePlay: Boolean = false

    //    val isResultStart by lazy { MutableLiveData<Boolean>(false) }
//    var isHistoryBtn: Boolean = false
//    var isHistoryCount: Boolean = false
//    var isTranslating: Boolean = false  //是否正在翻译中
//    var isFirstChoiceLanguage: Boolean = false  //是否已显示
//    var uiState = UiState.Default
    val source = MutableLiveData<Resource<String>>()
    val sourceLang = MutableLiveData<String>()
    val targetLang = MutableLiveData<String>()
    //    val targetLangStatus = MutableLiveData<String>()
    val sourceResult = MutableLiveData<String>()
    val targetResult = MutableLiveData<String>()
//    val isShowResult = MutableLiveData<Boolean>()

    val isSource = MutableLiveData<Boolean>()
    val voicePlayStatus = MutableLiveData<VoicePlayStatus>()
    val textTranslateResult = MutableLiveData<Event<Result<CompleteTextResult>>>()
    var currentLangType: LanguageType = LanguageType.SOURCE
    var langCheck: SparseArray<Boolean> = SparseArray()
    var currentSpeech = CommunicationActivity.SpeechType.SOURCE
    var lastSelectType = CommunicationActivity.SpeechType.SOURCE
    var sourceLangType: String = ""
//    var resultText = MutableLiveData<String>()
//    var historyBtn = MutableLiveData<Boolean>(false)


    var editTextStr: String = ""

    private val MOL_TAG = "mol"
    private val VERSION = 1

    var mGuideUtil: GuideUtil? = null
    val guideLiveData = MutableLiveData<Boolean>()

    var currentPlayClickIndex: Int = -1
    val voicePlayStatusV2 = MutableLiveData<Event<VoicePlayStatus>>()

    //tid
    private val TID_TRANSLATE = 256660000009L

    enum class HistoryItemViewType(val value: Int) {
        SOURCE(0),//源语言  绿色背景
        TARGET(1)//目标语言  蓝色背景
    }

    enum class LanguageType(val value: Int) {
        SOURCE(1),//源语言
        TARGET(2)//目标语言
    }

    enum class VoicePlayStatus(val value: Int) {
        START(0),//开始
        PAUSE(1),//暂停
        END(2),//结束
    }

    enum class UiState {
        /**
         * 初始化HI页面
         */
        Default,

        /**
         * 播放状态
         */
        Listening,

        /**
         * 可播放状态
         */
        Result
    }


    init {
        initLanguage()
    }

    fun initLanguage() {
        viewModelScope.launch(context = Dispatchers.IO) {
            ctx.initLanguage()
            val sl = ctx.getSourceLocale()
            val tl = if ("en" == sl.toLanguageTag()) {
                ctx.getTargetLocale("et")
            } else {
                ctx.getTargetLocale()
            }

            sourceLanguageLiveData.postValue(sl)
            targetLanguageLiveData.postValue(tl)

            mGuideUtil = GuideUtil().apply {
                initGuide(
                    ctx.applicationContext,
                    sl.language,
                    tl.language,
                    AppUtils.getAppName(getApplication<Application>().packageName)
                )
                guideLiveData.postValue(true)
            }
            updateCurrentLanguageName(
                sourceLanguage = sl.language,
                targetLanguage = tl.language
            )

            initUsedLanguage()
        }
    }

    fun saveLanguageUsed(languageTag: String, isSrc: Boolean = true) {
        viewModelScope.launch(Dispatchers.IO) {
            languageList.findLast { it.key == languageTag }?.let {
                saveLanguageUsed(it, isSrc)
            }
        }
    }

    fun saveLanguageUsed(info: LanguageModel.LanguageList, isSrc: Boolean = true) {
        if (isSrc) {
            addSrcUsed(info.key)
        } else {
            addDstUsed(info.key)
        }
    }

    //    private fun addSrcUsed(info: LanguageModel.LanguageList) {
    private fun addSrcUsed(info: String) {
        viewModelScope.launch(Dispatchers.IO) {

            if (!srcUsedList.contains(info)) {
                srcUsedList.add(info)
                saveToDb(KEY_SRC, srcUsedList)
            }
        }
    }

    //    private fun addDstUsed(info: LanguageModel.LanguageList) {
    private fun addDstUsed(info: String) {
        viewModelScope.launch(Dispatchers.IO) {
            if (!dstUsedList.contains(info)) {
                dstUsedList.add(info)
                saveToDb(KEY_DST, dstUsedList)
            }
        }
    }


    private fun sp(): SharedPreferences {
        mSp?.let {
            return it
        }
        ctx.getSharedPreferences(SP_NAME, Context.MODE_PRIVATE).apply {
            mSp = this
            return this
        }
    }

    private suspend fun initUsedLanguage() {
        withContext(Dispatchers.IO) {
            sp().getStringSet(KEY_SRC, null)?.forEach { item ->
                if (!srcUsedList.contains(item)) {
                    srcUsedList.add(item)
                }
            }

            sp().getStringSet(KEY_DST, null)?.forEach { item ->
                if (!dstUsedList.contains(item)) {
                    dstUsedList.add(item)
                }
            }
        }
    }

    private fun add(list: ArrayList<String>, item: String?): Boolean {
        if (item.isNullOrEmpty()) {
            return false
        }
        if (!list.contains(item)) {
            list.add(item)
            return true
        }
        return false
    }

    private suspend fun saveToDb(key: String, list: List<String>) {
        withContext(Dispatchers.IO) {
            val set = ArraySet<String>()
            list.forEach {
                set.add(it/*.key*/)
            }
            sp().edit().putStringSet(key, set).apply()
        }
    }


    fun updateCurrentLanguageName(sourceLanguage: String? = null, targetLanguage: String? = null) {

        val updateSourceLanguage = sourceLanguage ?: sourceLanguageLiveData.value?.language
        val updateTargetLanguage = targetLanguage ?: targetLanguageLiveData.value?.language

        viewModelScope.launch(context = Dispatchers.IO) {
            val languages = languageList
            val sourceName =
                languages.findLast { it.key == updateSourceLanguage }?.name
                    ?: ""
            val targetName =
                languages.findLast { it.key == updateTargetLanguage }?.name
                    ?: ""
            sourceLang.postValue(sourceName)
            targetLang.postValue(targetName)
        }
    }

    // 更新我说语言
    fun updateSourceLanguage(languageTag: String, languageName: String? = null) {
        sourceLanguageLiveData.value = Locale.forLanguageTag(languageTag)

        viewModelScope.launch(context = Dispatchers.IO) {
            ctx.saveSourceLocale(languageTag)

            if (languageName == null) {
                updateCurrentLanguageName()
            }
            val tl = ctx.getTargetLocale()
            athenaTranslate(languageTag, tl.language)
        }


    }

    // 更新他说语言
    fun updateTargetLanguage(languageTag: String, languageName: String? = null) {
        targetLanguageLiveData.value = Locale.forLanguageTag(languageTag)

        viewModelScope.launch(context = Dispatchers.IO) {
            ctx.saveTargetLocale(languageTag)
            if (languageName == null) {
                updateCurrentLanguageName()
            }
            val sl = ctx.getSourceLocale()
            athenaTranslate(sl.language, languageTag)
        }

    }

    /**
     * 复制译文
     */
    fun copy(view: View? = null, text: String) {
        val context = ctx//view.context
        val cm = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        // 创建普通字符型ClipData
        val mClipData = ClipData.newPlainText("Label", text)
        cm.setPrimaryClip(mClipData)
        Toast.makeText(getApplication(), R.string.copied_toast_translation, Toast.LENGTH_SHORT)
            .show()

    }

    /**
     * 播报
     */
    fun playTranslateVoice(history: History, position: Int = -1) {
        viewModelScope.launch(Dispatchers.IO) {
            if (TranslateService.Companion.getInstance().isSpeaking()) {
                stopTranslateVoice()
            } else {
                val isSuccess: Boolean = TranslateService.Companion.getInstance()
                    .startSpeak(
                        history.target_result,
                        Locale.forLanguageTag(history.targetlanguage_tag).language
                    )
                if (isSuccess) {
                    currentPlayClickIndex = position
                    voicePlayStatusV2.postValue(Event(VoicePlayStatus.START))
                } else {
                    //播放失败
                    viewModelScope.launch(Dispatchers.Main) {
                        getApplication<Application>().let {
                            Toast.makeText(it, R.string.play_error, Toast.LENGTH_LONG).show()
                        }
                    }
                }
            }
        }

    }

    /**
     * 停止播报
     */
    fun stopTranslateVoice() {
        viewModelScope.launch(context = Dispatchers.IO) {
            TranslateService.Companion.getInstance().stopSpeak()
            voicePlayStatus.postValue(VoicePlayStatus.PAUSE)
            voicePlayStatusV2.postValue(Event(VoicePlayStatus.PAUSE))
        }
    }


    /**
     * 文字翻译
     */
    fun translateText(txt: String): LiveData<Result<CompleteTextResult>> {
//        isTranslating = true
        val sourceLanguage = sourceLanguageLiveData.value?.language ?: ""
        val targetLanguage = targetLanguageLiveData.value?.language ?: ""
        var targetLanguageTag = ""
        var sourceLanguageTag = ""//目標語言的tag
        if (currentSpeech == CommunicationActivity.SpeechType.SOURCE) {
            sourceLanguageTag = sourceLanguage
            targetLanguageTag = targetLanguage
        } else {
            targetLanguageTag = sourceLanguage
            sourceLanguageTag = targetLanguage
        }
        saveLanguageUsed(sourceLanguageTag, true)
        saveLanguageUsed(targetLanguageTag, false)

        Log.i("FaceToFace","translateText--->sourceLanguageTag:$sourceLanguageTag targetLanguageTag:$targetLanguageTag txt:$txt")

        // 判断是否能播放
//        isVoicePlay = isCanPlayVoice(targetLanguageTag)
        return TranslateService.Companion.getInstance()
            .translateFromText(
                txt,
                sourceLanguageTag,
                targetLanguageTag,
            )
    }

    /**
     * 埋点
     */

    fun athenaTranslate(sourceLanguage: String?, targetLanguage: String?) {
        val enable: Boolean = Athena.getInstance().isEnable(TID_TRANSLATE)
        if (!enable) {
            return
        }
        val bundle = Bundle()
        bundle.putString("act", "face") //面对面翻译固定传"face"，因为其他的翻译场景也使用这个tid，所以需要这个act来区分
        bundle.putString("lan1", sourceLanguage) //language1：面对面翻译中第一个语言的语言码
        bundle.putString("lan2", targetLanguage) //language2：面对面翻译中第二个语言的语言码
        Athena.getInstance().log(TID_TRANSLATE, MOL_TAG, VERSION, bundle)
    }

    fun registerCall() = viewModelScope.launch(context = Dispatchers.IO) {
        HiServiceConnection.registerUtteranceProgress(callback = utteranceProgressCallback)
    }

    fun unregisterCall() = viewModelScope.launch(context = Dispatchers.IO) {
        HiServiceConnection.unregisterUtteranceProgress(callback = utteranceProgressCallback)
    }

    val utteranceProgressCallback = object :
        UtteranceProgressCallback.Stub() {
        override fun onStart(p0: String?) {
            voicePlayStatus.postValue(VoicePlayStatus.START)
            voicePlayStatusV2.postValue(Event(VoicePlayStatus.START))
        }

        override fun onDone(p0: String?) {
            voicePlayStatus.postValue(VoicePlayStatus.END)
            voicePlayStatusV2.postValue(Event(VoicePlayStatus.END))

        }

        override fun onError(p0: String?) {
            voicePlayStatusV2.postValue(Event(VoicePlayStatus.END))

        }
    }

    fun destroy() = viewModelScope.launch(context = Dispatchers.IO) {
        HiServiceConnection.destroy()
    }


    override fun onCleared() {
        super.onCleared()
        destroy()
    }


    private fun isZhCn(): Boolean {
        val curLanguage = Locale.getDefault().language
        return "zh".equals(curLanguage, ignoreCase = true)
    }

    /**
     * 能否播报
     */
    fun isCanPlayVoice(targetLanguage: String): Boolean {
        var isPlay: Boolean = false
        isPlay = if (Constants.noSupportVoiceLanguages.contains(targetLanguage)) {
            false
        } else {
//            if (HiServiceConnection.isLanguageAvailable(targetLanguage)) {
//                true
//            }else{
//                Constant.supportVoiceLanguages.contains(targetLanguage)
//            }

            HiServiceConnection.isLanguageAvailable(targetLanguage)
        }
        Log.d(
            CommunicationActivity.TAG,
            "isPlay: ${HiServiceConnection.isLanguageAvailable(targetLanguage)},language:${targetLanguage}"
        )
        return isPlay
    }


}


fun Context.loadLanguageList(langMap: HashMap<String, String>? = null): List<LanguageModel.LanguageList> {
    val supportSourceLanguages = getSupportSourceLanguage()
    val supportSourceLanguageSize = supportSourceLanguages.size()
    val sourceValues = List(supportSourceLanguageSize) {
        supportSourceLanguages[it]?.let { it1 ->
            CustomLocale(
                it1.language,
                it1.displayLanguage
            )
        }
    }
    var array: MutableList<LanguageModel.LanguageList> = mutableListOf<LanguageModel.LanguageList>()
    var language: String = resources.configuration.locale.language
    for (CustomLocale in sourceValues) {
        if (CustomLocale == null) {
            continue
        }
        val languageList = LanguageModel.LanguageList()
        languageList.key = CustomLocale.key
        if (language == "si" && CustomLocale.key == "no") { // 挪威语兼容
            languageList.name = "නෝර්වීජියානු"
        } else {
            languageList.name = CustomLocale.name
        }
        array.add(languageList)
        langMap?.put(CustomLocale.key, CustomLocale.name)
    }
    return array.toList()
}


private fun getSupportSourceLanguage(): LocaleListCompat {
    return AppUtil.getLocaleListCompat(Constants.supportLanguages)
}

//private fun noSupportVoiceLanguages(): LocaleListCompat {
//    return AppUtil.getLocaleListCompat(Constant.noSupportVoiceLanguages)
//}