package com.talpa.master.ui.main

import android.app.Application
import android.content.ComponentName
import android.content.Intent
import android.content.pm.ResolveInfo
import android.os.Bundle
import android.speech.RecognitionListener
import android.speech.RecognitionService
import android.speech.RecognizerIntent
import android.speech.SpeechRecognizer
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.MutableLiveData
import java.util.*

class SpeechViewModel(application: Application) : AndroidViewModel(application) {

    val rmsdBState by lazy { MutableLiveData<Float>() }

    val recognitionResults by lazy { MutableLiveData<ArrayList<String>>() }

    val recognitionState by lazy { MutableLiveData<RecognitionState>() }

    private val speechRecognizer by lazy {
        val list: List<ResolveInfo> =
            getApplication<Application>().packageManager.queryIntentServices(
                Intent(RecognitionService.SERVICE_INTERFACE), 0
            )
        val componentName =
            list.firstOrNull()?.serviceInfo?.let { ComponentName(it.packageName, it.name) }

        //com.google.android.voicesearch.serviceapi.GoogleRecognitionService
        //com.google.android.voicesearch.serviceapi.GoogleRecognitionService
        println("list=====${list.map { "${it.serviceInfo?.name}_${it.serviceInfo?.isEnabled}" }}")
        SpeechRecognizer.createSpeechRecognizer(getApplication(), componentName)//componentName
    }

    fun speechRecognizeAvailable() : Boolean {
        val list: List<ResolveInfo> =
            getApplication<Application>().packageManager.queryIntentServices(
                Intent(RecognitionService.SERVICE_INTERFACE), 0
            )
        val componentName =
            list.firstOrNull()?.serviceInfo?.let { ComponentName(it.packageName, it.name) }
        return componentName != null && list.firstOrNull()?.serviceInfo?.enabled ?: false
    }

    fun startRecognize(locale: Locale = Locale.getDefault()): Boolean {


        try {
            val isRecognitionAvailable = SpeechRecognizer.isRecognitionAvailable(getApplication())

            println("isRecognitionAvailable=$isRecognitionAvailable")

            speechRecognizer.setRecognitionListener(OnRecognitionListener())

            val recognizerIntent = recognizerIntent(locale)

            speechRecognizer.startListening(recognizerIntent)
            return true
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return false

    }

    fun finishRecognize() {
        try {
            speechRecognizer.stopListening()
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    fun stopRecognize() {
        try {
            //speechRecognizer.setRecognitionListener(null)
            speechRecognizer.stopListening()
            speechRecognizer.cancel()
            speechRecognizer.destroy()
            recognitionState.value = OnEndOfSpeech
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    fun isFinishRecoginze() : Boolean {
        return recognitionState.value != OnReadyForSpeech && recognitionState.value != OnBeginningOfSpeech
                && recognitionState.value != OnRmsChanged
    }

    private fun recognizerIntent(locale: Locale = Locale.getDefault()): Intent {
        val intent = Intent(RecognizerIntent.ACTION_RECOGNIZE_SPEECH)

        intent.putExtra(
            RecognizerIntent.EXTRA_LANGUAGE_MODEL,
            RecognizerIntent.LANGUAGE_MODEL_FREE_FORM
        )

        intent.putExtra(RecognizerIntent.EXTRA_PARTIAL_RESULTS, true)
        intent.putExtra(RecognizerIntent.EXTRA_SPEECH_INPUT_MINIMUM_LENGTH_MILLIS, 10000000 * 60)

        intent.putExtra(RecognizerIntent.EXTRA_LANGUAGE, locale.toLanguageTag())


        return intent
    }

    private inner class OnRecognitionListener : RecognitionListener {

        override fun onReadyForSpeech(params: Bundle?) {
            val state = OnReadyForSpeech.init<OnReadyForSpeech> {
                OnReadyForSpeech.params = params
            }
            recognitionState.value = state
        }

        override fun onBeginningOfSpeech() {
            recognitionState.value = OnBeginningOfSpeech
        }

        override fun onRmsChanged(rmsdB: Float) {
            rmsdBState.value = rmsdB

            val state = OnRmsChanged.init<OnRmsChanged> {
                OnRmsChanged.rmsdB = rmsdB
            }
            recognitionState.value = state
        }

        override fun onBufferReceived(buffer: ByteArray?) {

            val state = OnBufferReceived.init<OnBufferReceived> {
                OnBufferReceived.buffer = buffer
            }
            recognitionState.value = state
        }

        override fun onEndOfSpeech() {
            recognitionState.value = OnEndOfSpeech
        }

        override fun onError(error: Int) {

            val state = OnError.init<OnError> { OnError.error = error }
            recognitionState.value = state

        }

        override fun onResults(results: Bundle?) {
            val recognitionTempResults =
                results?.getStringArrayList(SpeechRecognizer.RESULTS_RECOGNITION)

            recognitionTempResults?.let {
                recognitionResults.value = it
            }

            recognitionState.value = OnResults.init<OnResults> { OnResults.results = results }
        }

        override fun onPartialResults(results: Bundle?) {

            val partialResults =
                results?.getStringArrayList(SpeechRecognizer.RESULTS_RECOGNITION)

            val state = OnPartialResults.init<OnPartialResults> {
                OnPartialResults.results = results
            }
            recognitionState.value = state
        }

        override fun onEvent(eventType: Int, params: Bundle?) {
            val state = OnEvent.init<OnEvent> {
                OnEvent.eventType = eventType
                OnEvent.params = params
            }
            recognitionState.value = state
        }
    }


}

sealed class RecognitionState {
    fun <T : RecognitionState> init(init: T.() -> Unit): T {
        init.invoke(this as T)
        return this
    }
}

object OnReadyForSpeech : RecognitionState() {
    var params: Bundle? = null
}

object OnBeginningOfSpeech : RecognitionState()

object OnRmsChanged : RecognitionState() {
    var rmsdB: Float = 0f
}

object OnBufferReceived : RecognitionState() {
    var buffer: ByteArray? = null
}

object OnEndOfSpeech : RecognitionState()

object OnError : RecognitionState() {
    var error: Int = 0
}

object OnResults : RecognitionState() {
    var results: Bundle? = null
}

object OnPartialResults : RecognitionState() {
    var results: Bundle? = null
}

object OnEvent : RecognitionState() {
    var eventType: Int = 0
    var params: Bundle? = null
}


fun log(message: String) {
    //com.zaz.translate.log.log(debug = BuildConfig.DEBUG, tag = "Speech", message = message)
    println(message)
}

