package com.talpa.master.ui.main.v2

import android.annotation.SuppressLint
import android.util.Log
import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.databinding.DataBindingUtil
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.talpa.master.BuildConfig
import com.talpa.master.R
import com.talpa.master.common.Constant
import com.talpa.master.common.getString
import com.talpa.master.databinding.*
import com.talpa.master.repository.room.model.History
import com.talpa.master.ui.main.CommunicationActivityV2
import com.talpa.master.ui.main.MainViewModel

class ConversationAdapter(
    private var list: ArrayList<History>? = null,
    private val mViewModelV2: CommunicationViewModelV2
) :
    RecyclerView.Adapter<ConversationHolder>() {

    companion object {
        const val VIEW_TYPE_UNKNOWN = 0
        const val VIEW_TYPE_LEFT_TEXT = 1
        const val VIEW_TYPE_RIGHT_TEXT = 2
        const val VIEW_TYPE_LEFT_SAYING = 3
        const val VIEW_TYPE_RIGHT_SAYING = 4
    }

    //记录是否是在录音, true=我说， false=他说, null=没有录音
    var isLeftSaying: Boolean? = null

    //记录录音的倒计时时间
    private var countDownCur: Int? = null

    //记录录音出错信息
    private var countDownCurErrorMsg: String? = null

    //记录录音识别到的文字
    private var recordContent: String? = null

    //记录录音 我说的Holder 用于更新信息时不用闪一下
    private var mCountDownConversationLeftHolder: ConversationHolder? = null

    //记录录音 他说的Holder 用于更新信息时不用闪一下
    private var mCountDownConversationRightHolder: ConversationHolder? = null

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ConversationHolder {
        val inflater = LayoutInflater.from(parent.context)
        return when (viewType) {
            VIEW_TYPE_LEFT_TEXT -> {
                ConversationLeftHolder(
                    DataBindingUtil.inflate<MolItemConversationLeftTextBinding?>(
                        inflater,
                        R.layout.mol_item_conversation_left_text,
                        parent,
                        false
                    ).apply {
                        viewModel = mViewModelV2
                    }
                )
            }
            VIEW_TYPE_RIGHT_TEXT -> {
                ConversationRightHolder(
                    DataBindingUtil.inflate<MolItemConversationRightTextBinding?>(
                        inflater,
                        R.layout.mol_item_conversation_right_text,
                        parent,
                        false
                    ).apply {
                        viewModel = mViewModelV2
                    }
                )
            }
            VIEW_TYPE_LEFT_SAYING -> {

                ConversationLeftSayingHolder(
                    DataBindingUtil.inflate<MolItemConversationLeftSayingBinding?>(
                        inflater,
                        R.layout.mol_item_conversation_left_saying,
                        parent,
                        false
                    ).apply {
                        viewModel = mViewModelV2
                    }
                ).apply {
                    mCountDownConversationLeftHolder = this
                }
            }
            VIEW_TYPE_RIGHT_SAYING -> {
                ConversationRightSayingHolder(
                    DataBindingUtil.inflate<MolItemConversationRightSayingBinding?>(
                        inflater,
                        R.layout.mol_item_conversation_right_saying,
                        parent,
                        false
                    ).apply {
                        viewModel = mViewModelV2
                    }
                ).apply {
                    mCountDownConversationRightHolder = this
                }
            }
            else -> {
                ConversationUnknownHolder(
                    DataBindingUtil.inflate<MolItemConversationUnknownBinding?>(
                        inflater,
                        R.layout.mol_item_conversation_unknown,
                        parent,
                        false
                    ).apply {
                        viewModel = mViewModelV2
                    }
                )
            }
        }
    }

    override fun onBindViewHolder(holder: ConversationHolder, position: Int) {
        holder.bind(itemData(position), position)

        //如果是在录音中
        isLeftSaying?.let {
            countDownCur?.let {
                holder.bindCount(it)
            }
            countDownCurErrorMsg?.let {
                holder.recordTipMessage(recordContent, it)
            }
        }
    }

    override fun getItemCount(): Int {
        return size()
    }

    override fun getItemViewType(position: Int): Int {
        itemData(position)?.let {
            return when (it.display_type) {
                MainViewModel.HistoryItemViewType.SOURCE.value -> {
                    VIEW_TYPE_LEFT_TEXT
                }
                MainViewModel.HistoryItemViewType.TARGET.value -> {
                    VIEW_TYPE_RIGHT_TEXT
                }
                else ->
                    VIEW_TYPE_UNKNOWN
            }
        }
        isLeftSaying?.let {
            return if (it) {
                VIEW_TYPE_LEFT_SAYING
            } else {
                VIEW_TYPE_RIGHT_SAYING
            }
        }
        return VIEW_TYPE_UNKNOWN
    }

    private fun infoSize() = list?.size ?: 0

    private fun size(): Int {
        isLeftSaying?.let {
            return infoSize() + 1
        }
        return infoSize()
    }

    private fun itemData(position: Int): History? {
        if (position in 0 until infoSize()) {
            return list?.get(position)
        }
        return null
    }

    @SuppressLint("NotifyDataSetChanged")
    fun updateList(it: ArrayList<History>?, notifyData: Boolean = true) {
        if (BuildConfig.DEBUG) {
            Log.i(
                CommunicationActivityV2.TAG,
                "updateList, adapter, isLeftSaying:$isLeftSaying,  it size:${it?.size},  size:${this.list?.size}"
            )
        }

        list = null
        list = it
        if (notifyData) {
            cusNotifyDataSetChanged()
        }
    }

    @SuppressLint("NotifyDataSetChanged")
    fun addSaying(
        isLeftSaying: Boolean? = null,
        countDownMax: Int? = Constant.LISTENER_TIME.toInt(),
        recyclerView: RecyclerView
    ) {
        if (BuildConfig.DEBUG) {
            Log.i(
                CommunicationActivityV2.TAG,
                "addSaying, adapter, isLeftSaying:$isLeftSaying, countDownMax:$countDownMax, size:${this.list?.size}"
            )
        }
        removeGuideItem()

        this.isLeftSaying = isLeftSaying
        countDownCur = countDownMax
        countDownCurErrorMsg = null

        recyclerView.scrollToPosition(itemCount - 1)

        cusNotifyDataSetChanged()
    }

    private fun removeGuideItem() {
        val listGuide = ArrayList<History>()
        list?.forEach {
            if (it.pid == GuideUtil.LEFT_GUIDE_PID || it.pid == GuideUtil.RIGHT_GUIDE_PID) {
                listGuide.add(it)
            }
        }
        if (BuildConfig.DEBUG) {
            Log.i(
                CommunicationActivityV2.TAG,
                "removeGuideItem:${listGuide.isNotEmpty()}, size:${this.list?.size}"
            )
        }

        if (listGuide.isNotEmpty()) {
            list?.removeAll(listGuide)
        }
    }

    @SuppressLint("NotifyDataSetChanged")
    fun addTranslating(
        isLeftSaying: Boolean? = null,
        content: String?,
        tipMsg: String?,
        recyclerView: RecyclerView
    ) {

        if (BuildConfig.DEBUG) {
            Log.i(
                CommunicationActivityV2.TAG,
                "addTranslating, adapter, isLeftSaying:$isLeftSaying, content:$content, tipMsg:$tipMsg"
            )
        }
        removeGuideItem()
        this.isLeftSaying = isLeftSaying
        countDownCur = null
        recordContent = content
        countDownCurErrorMsg = tipMsg

        recyclerView.scrollToPosition(itemCount - 1)

        cusNotifyDataSetChanged()
    }

    @SuppressLint("NotifyDataSetChanged")
    fun addSayingResult(history: History?, recyclerView: RecyclerView) {
        if (BuildConfig.DEBUG) {
            Log.i(
                CommunicationActivityV2.TAG,
                "addSayingResult, adapter, isLeftSaying:$isLeftSaying, size:${this.list?.size}"
            )
        }
        removeGuideItem()

        isLeftSaying = null
        countDownCur = null
        recordContent = null
        countDownCurErrorMsg = null

        history?.let { historyId ->
            this.list?.apply {
                findLast { it.id == historyId.id }?.let {
                    return
                }
                add(historyId)
            }
        }

        recyclerView.scrollToPosition(itemCount - 1)

        cusNotifyDataSetChanged()
    }

    /**
     * 录制过长中倒计时
     */
    fun updateCountDown(count: Int?, recyclerView: RecyclerView) {
        countDownCur = count
        if (BuildConfig.DEBUG) {
            Log.i(
                CommunicationActivityV2.TAG,
                "updateCountDown: countDownCur:$countDownCur, is null:${mCountDownConversationLeftHolder == null}, is:$mCountDownConversationLeftHolder"
            )
            Log.i(
                CommunicationActivityV2.TAG,
                "updateCountDown: countDownCur:$countDownCur, r is null:${mCountDownConversationRightHolder == null}, r is:$mCountDownConversationRightHolder"
            )
        }

        isLeftSaying?.let { isLeft ->
            recyclerView.layoutManager?.let { lm ->
                if (lm is LinearLayoutManager) {
                    val lastPosition = lm.findLastCompletelyVisibleItemPosition()
                    Log.v(
                        CommunicationActivityV2.TAG,
                        "updateCountDown, lastPosition:$lastPosition, count:${itemCount - 1}"
                    )
                    if (lastPosition != itemCount - 1) {
                        notifyItemChanged(itemCount - 1)
                        return
                    }
                }
            }

            if (isLeft) {
                countDownCur?.let {
                    mCountDownConversationLeftHolder?.bindCount(it)
                }
            } else {
                countDownCur?.let {
                    mCountDownConversationRightHolder?.bindCount(it)
                }
            }
        }
    }

    /**
     * 录制出错
     */
    fun recordError(errorMsgParam: String, recyclerView: RecyclerView) {
        var errorMsg = errorMsgParam
        if (BuildConfig.DEBUG) {
            Log.i(CommunicationActivityV2.TAG, "recordError, adapter, errorMsg:$errorMsg")
        }
        if (errorMsg.isEmpty()) {
            errorMsg =
                R.string.v_text_translat_error.getString(recyclerView.context,
                    mViewModelV2.sourceLangType,
                    "Please try again later"
                )
        }
        removeGuideItem()
        countDownCurErrorMsg = errorMsg
        isLeftSaying?.let { isLeft ->
            recyclerView.layoutManager?.let { lm ->
                if (lm is LinearLayoutManager) {
                    val lastPosition = lm.findLastCompletelyVisibleItemPosition()
                    Log.v(
                        CommunicationActivityV2.TAG,
                        "recordError, lastPosition:$lastPosition, count:${itemCount - 1}"
                    )
                    if (lastPosition != itemCount - 1) {
                        notifyItemChanged(itemCount - 1)
                        return
                    }
                }
            }
            if (isLeft) {
                mCountDownConversationLeftHolder?.apply {
                    recordTipMessage(errorMsg, null)
                }
            } else {
                mCountDownConversationRightHolder?.recordTipMessage(errorMsg, null)
            }
        }
    }

    @SuppressLint("NotifyDataSetChanged")
    fun cusNotifyDataSetChanged() {
        notifyDataSetChanged()
    }
}