package com.talpa.master.ui.main.v2

import android.app.Activity
import android.content.Intent
import android.content.pm.ActivityInfo
import android.graphics.Color
import android.graphics.drawable.AnimationDrawable
import android.util.TypedValue
import android.widget.Toast
import androidx.core.widget.TextViewCompat
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProvider
import com.talpa.lib.language.open.getAttrColor
import com.talpa.lib.language.util.FoldUtils
import com.talpa.master.R
import com.talpa.master.databinding.DataBindingActivity
import com.talpa.master.databinding.MolFragmentTranslateResultBinding
import com.talpa.master.ui.main.ResultViewModel
import com.talpa.master.utils.AppUtil

class TranslateEnlargeActivity : DataBindingActivity<MolFragmentTranslateResultBinding>() {

    companion object {
        const val KEY_INTENT_CONTENT = "key_intent_content"
        const val KEY_INTENT_LANGUAGE_CODE = "key_intent_language_code"
        fun launch(activity: Activity, msg: String, languageCode: String) {
            activity.startActivity(Intent(activity, TranslateEnlargeActivity::class.java).apply {
                putExtra(KEY_INTENT_CONTENT, msg)
                putExtra(KEY_INTENT_LANGUAGE_CODE, languageCode)
            })
        }
    }


    private var isHiOS = true
    private var mContent: String? = null
    private var mLanguageCode: String? = null
    private var voiceAnimationDrawable: AnimationDrawable = AnimationDrawable()


    override fun skipOrientation(): Boolean {

        // 中等宽度 以上的 屏幕
        if (FoldUtils.isTabletType(this)) {
            // CUT: 2022/11/10 设计文稿 不区分横竖屏,测试让适配横竖屏
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_FULL_USER
        } else {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE
        }

        return true
    }

    override fun skipTheme(): Boolean {
        return true
    }
    override fun getStatusBarColorResId():Int {
        return android.R.color.transparent
    }
    override fun getNavigationBarColorResId() :Int {
        return if(isHiOS) {
            R.color.mol_color_F8F8F8
        } else {
            com.transsion.widgetslib.R.color.os_platform_basic_color_xos
        }
    }

    override fun isDarkFont(): Boolean {
        return true
    }

    override fun getContentView(): Int = R.layout.mol_fragment_translate_result

    override fun initView() {

        mBinding.faceDialogCloseClick.setOnClickListener {
            onBackPressed()
        }

        mBinding.tvResultSpeechsClick.setOnClickListener {
            if (mContent.isNullOrEmpty()) {
                finish()
                return@setOnClickListener
            }
            if (mLanguageCode.isNullOrEmpty()) {
                return@setOnClickListener
            }
            mBinding.viewModel?.playTranslateVoice(mContent, mLanguageCode)
        }
    }

    override fun onResume() {
        super.onResume()
        mBinding.viewModel?.registerCall()
    }

    override fun onPause() {
        super.onPause()
        mBinding.viewModel?.stopTranslateVoice()
        mBinding.viewModel?.unregisterCall()
    }

    override fun init() {
        isHiOS = AppUtil.isHiOS()
        if (isHiOS) {
            setTheme(R.style.MolFullscreenHiOs)
        }
        parseIntent(intent)
        if (mContent.isNullOrEmpty()) {
            finish()
            return
        }
        initOsView()

        ViewModelProvider(
            this,
            ViewModelProvider.AndroidViewModelFactory.getInstance(application)
        ).get(ResultViewModel::class.java).apply {
            mBinding.viewModel = this
            initObserver(this)
        }
        mBinding.tvResultContent.text = mContent
        TextViewCompat.setAutoSizeTextTypeUniformWithConfiguration(
            mBinding.tvResultContent,
            21,
            120,
            1,
            TypedValue.COMPLEX_UNIT_PX
        )
    }

    private fun initOsView() {
        if (isHiOS) {
            mBinding.faceDialogClose.setImageResource(R.drawable.mol_face_close)
            mBinding.tvResultContent.setTextColor(Color.BLACK)
            mBinding.tvResultSpeechs.setImageResource(R.drawable.mol_face_left_play_normal)
            mBinding.choiceParent.setBackgroundResource(R.drawable.mol_face_enlarge_dialog_bg)
        } else {
            mBinding.faceDialogClose.setImageResource(R.drawable.mol_face_enlarge_dialog_close)
            mBinding.tvResultContent.setTextColor(Color.WHITE)
            mBinding.tvResultSpeechs.setImageResource(R.drawable.mol_face_right_play_normal)
            getAttrColor(com.transsion.widgetslib.R.attr.os_platform_basic_color)?.let {
                mBinding.choiceParent.setBackgroundColor(it)
            }
        }
    }

    private fun initObserver(resultViewModel: ResultViewModel) {
        resultViewModel.voicePlayStatus.observe(this, Observer {
            when (it) {
                ResultViewModel.VoicePlayStatus.START -> {
                    if (isHiOS) {
                        mBinding.tvResultSpeechs.setImageResource(R.drawable.mol_voice_start_anim)
                    } else {
                        mBinding.tvResultSpeechs.setImageResource(R.drawable.mol_voice_start_anim_black)
                    }
                    voiceAnimationDrawable = mBinding.tvResultSpeechs.drawable as AnimationDrawable
                    voiceAnimationDrawable.start()
                }
                else -> {
                    voiceAnimationDrawable.stop()
                    if (isHiOS) {
                        mBinding.tvResultSpeechs.setImageResource(R.drawable.mol_face_left_play_normal)
                    } else {
                        mBinding.tvResultSpeechs.setImageResource(R.drawable.mol_face_right_play_normal)
                    }

                    if (ResultViewModel.VoicePlayStatus.ERROR == it) {
                        Toast.makeText(application, R.string.play_error, Toast.LENGTH_SHORT).show()
                    }
                }
            }
        })
    }

    override fun onNewIntent(intent: Intent?) {
        super.onNewIntent(intent)
        parseIntent(intent)
    }

    private fun parseIntent(intent: Intent?) {
        intent?.let {
            mContent = it.getStringExtra(KEY_INTENT_CONTENT)
            mLanguageCode = it.getStringExtra(KEY_INTENT_LANGUAGE_CODE)
        }
    }
}