package com.cy.sts

import android.app.Application
import android.content.Context
import android.speech.SpeechRecognizer
import com.cy.sts.lang.RegionLang
import com.cy.sts.stt.BaiduRecognitionHelper
import com.cy.sts.stt.GoogleRecognitionHelper
import com.cy.sts.stt.SpeechHelper
import com.talpa.tengine.lang.LANG
import io.reactivex.Flowable
import java.io.File

/**
 * STT,Trans,TTS 引擎
 * @author CY 2019/1/13
 */

class STSEngine private constructor(val builder: Builder) : STSEngineImpl(builder) {

    //private val baiduRecognitionHelper: BaiduRecognitionHelper by lazy { BaiduRecognitionHelper(this) }

    private val googleRecognitionHelper by lazy { GoogleRecognitionHelper(this) }

    private var speechHelper: SpeechHelper? = null


    /**
     * 重新设置语言
     */
    fun setLanguage(
        speechLanguage: RegionLang? = builder.speechLanguage,
        translateSourceLanguage: LANG? = builder.translateSourceLanguage,
        translateTargetLanguage: LANG? = builder.translateTargetLanguage,
        textSpeechSourceLanguage: LANG? = builder.textSpeechSourceLanguage
    ): STSEngine {
        builder.translateSourceLanguage = translateSourceLanguage
        builder.translateTargetLanguage = translateTargetLanguage
        builder.speechLanguage = speechLanguage
        builder.textSpeechSourceLanguage = textSpeechSourceLanguage

        speechHelper = null

        return this
    }

    /**
     * Speech To Text
     */
    fun speechToText(byteArray: ByteArray): Flowable<String> {

        if (builder.speechLanguage == null) {
            throw NullPointerException("请初始化translateSourceLanguage,translateTargetLanguage,speechLanguage")
        }
        return sttImpl(byteArray)
    }


    /**
     * Text to Speech
     */
    fun textToSpeech(text: String): Flowable<File> {
        if (builder.textSpeechSourceLanguage == null) {
            throw NullPointerException("请初始化translateSourceLanguage,translateTargetLanguage,speechLanguage")
        }
        return ttsImpl(text)
    }


    /**
     * 开始录音
     */
    override fun startSpeech() {

        super.startSpeech()

        speechHelper = recognitionHelper()

        speechHelper?.startSpeech()
    }

    /**
     * 停止录音，进行识别，翻译
     */
    override fun stopSpeech() {
        super.stopSpeech()
        speechHelper?.stopSpeech()
    }

    /**
     * 取消识别，翻译等一些列操作
     */
    override fun cancelSpeech() {
        speechHelper?.cancelSpeech()
    }

    /**
     * 释放资源
     */
    override fun release() {
        speechHelper?.release()
        speechHelper = null
        builder.eventListener = null
    }


    /**
     * 获取SpeechHelper
     */
    private fun recognitionHelper(): SpeechHelper {
        if (builder.eventListener == null) {
            throw java.lang.NullPointerException("eventListener is null")
        }
        if (builder.speechLanguage == null) {
            throw java.lang.NullPointerException("speechLanguage is null")
        }


//        return if (baiduRecognitionHelper.isSupportLanguage(builder.speechLanguage!!)) {
//
//            if (speechHelper is GoogleRecognitionHelper) {
//                speechHelper?.release()
//            }
//
//            baiduRecognitionHelper
//        } else

        return if (SpeechRecognizer.isRecognitionAvailable(builder.context)) {

            if (googleRecognitionHelper.isSupportLanguage(builder.speechLanguage!!)) {

                if (speechHelper is BaiduRecognitionHelper) {
                    speechHelper?.release()
                }
                googleRecognitionHelper
            } else {
                throw Exception("不支持识别该语言")
            }
        } else {
            throw Exception("请安装 Google APP")
        }
    }

    data class Builder(
        val context: Application,
        var speechLanguage: RegionLang? = null,//Speech to text 源语言
        var translateSourceLanguage: LANG? = null,//翻译源语言
        var translateTargetLanguage: LANG? = null,//翻译目标语言
        var textSpeechSourceLanguage: LANG? = null,//Text to Speech 源语言
        var eventListener: EventListener? = null//事件回调
    ) {

        fun build() = STSEngine(builder = this)
    }
}

