package com.cy.sts

import android.os.Bundle
import com.cy.sts.asr.STTFactory
import com.cy.sts.asr.impl.BaiduSTT
import com.cy.sts.asr.impl.GoogleSTT
import com.cy.sts.common.Logcat
import com.cy.sts.stt.SpeechHelper
import com.cy.sts.tts.TTSFactory
import com.cy.sts.tts.impl.BaiduTTS
import com.cy.sts.tts.impl.GoogleTTS
import com.talpa.tengine.Result
import com.talpa.tengine.Trans
import com.talpa.tengine.Translator
import com.talpa.tengine.lang.LANG
import io.reactivex.Flowable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import okhttp3.ResponseBody
import java.io.File
import java.util.concurrent.Executors

/**
 *
 * @author CY 2019/1/13
 */
open class STSEngineImpl(private val builder: STSEngine.Builder) {

    /**
     * 线程池
     */
    private val cachedScheduler = Schedulers.from(Executors.newCachedThreadPool())

    /**
     * 返回数据包
     */
    private val bundle = Bundle()


    private var translateDisposable: Disposable? = null


    init {
        /*val config = Translator.DefaultConfig(
            cacheFile = builder.context.externalCacheDir,
            debug = BuildConfig.DEBUG,
            baiduTranslateFreeEnabled = true,
            bingTranslateFreeEnabled = true,
            googleTranslateFreeEnabled = true,
            icibaTranslateFreeEnabled = true,
            omiTranslateFreeEnabled = true,
            tencentTranslateFreeEnabled = true,
            youdaoTranslateFreeEnabled = true,
            googleCloudTranslateChargeEnabled = true,
            microsoftTranslateChargeEnabled = true
        )*/
        Translator.init(builder.context)
    }

    open fun startSpeech() {
        if (translateDisposable?.isDisposed == false) {
            translateDisposable?.dispose()
        }
    }

    open fun stopSpeech() {

    }

    open fun cancelSpeech() {
        if (translateDisposable?.isDisposed == false) {
            translateDisposable?.dispose()
        }
    }

    open fun release() {

    }

    /**
     * Speech to Text Impl
     */
    protected fun sttImpl(bytes: ByteArray): Flowable<String> {
        val googleSTT = getSTTFlowable<GoogleSTT>(bytes)
        val baiduSTT = getSTTFlowable<BaiduSTT>(bytes)
        return baiduSTT
            .onErrorResumeNext(googleSTT)
            .switchIfEmpty(googleSTT)
            .switchIfEmpty(Flowable.just(""))
        //.onErrorResumeNext(googleSTT)

    }


    /*
    /**
     * Translate Impl
     */
    protected fun transImpl(text: String): Flowable<String> {

        val googleFlowable = getTranslateFlowable<GoogleTranslate>(text).publish().autoConnect()
        val baiduFlowable = getTranslateFlowable<BaiduTranslate>(text).publish().autoConnect()

        val icibaFlowable = getTranslateFlowable<IcibaTranslate>(text).publish().autoConnect()
        val omiFlowable = getTranslateFlowable<OmiTranslate>(text).publish().autoConnect()
        val tencentFlowable = getTranslateFlowable<TencentTranslate>(text).publish().autoConnect()
        val trycanFlowable = getTranslateFlowable<TrycanTranslate>(text).publish().autoConnect()
        val youdaoFlowable = getTranslateFlowable<YoudaoTranslate>(text).publish().autoConnect()

        return googleFlowable
            .switchIfEmpty(baiduFlowable)//没有返回值，或者不支持翻译时调用备用翻译
            .doOnError { it.printStackTrace() }
            .onErrorResumeNext(baiduFlowable)//发生错误时，调用备用翻译
            .switchIfEmpty(icibaFlowable)
            .onErrorResumeNext(icibaFlowable)
            .switchIfEmpty(tencentFlowable)
            .onErrorResumeNext(tencentFlowable)
            .switchIfEmpty(omiFlowable)
            .onErrorResumeNext(omiFlowable)
            .switchIfEmpty(trycanFlowable)
            .onErrorResumeNext(trycanFlowable)
            .switchIfEmpty(youdaoFlowable)
            .onErrorResumeNext(youdaoFlowable)
            .switchIfEmpty(Flowable.just(""))

    }

    */

    /**
     * TTS Impl
     */
    protected fun ttsImpl(text: String): Flowable<File> {

        val list = arrayOf(GoogleTTS(builder.context), BaiduTTS(builder.context))

        val lang: LANG =
            builder.textSpeechSourceLanguage
                ?: throw IllegalArgumentException("textSpeechSourceLanguage is null")

        val supportList = list.filter { it.isSupportLanguage(lang) }

        var flowable: Flowable<File>? = null
        for (tts in supportList) {
            val ttsFlow = getTTSFlowable(tts, text)
            flowable = flowable?.onErrorResumeNext(ttsFlow) ?: ttsFlow
        }

        return flowable?.onErrorReturn { File("") } ?: Flowable.fromCallable { File("") }

    }


    /**
     * 翻译与播报
     */
    fun transWithSpeech(text: String) {

        val sourceLanguage: LANG = builder.translateSourceLanguage!!
        val targetLanguage: LANG = builder.translateTargetLanguage!!

//        val p = Pattern.compile("\\s*|\t|\r|\n")
//        val m = p.matcher(text)
//        val newText = m.replaceAll("")
        //val array = arrayOf(text)
        val trans =
            Trans(sourceLanguage, targetLanguage, text)//arrayOf("今天天气如何?", "你", "今天开心吗", "今天天气如何")

        val translatorFlowable = Translator.translate(trans)
        translateDisposable = translatorFlowable
            .takeUntil { tran ->
                val result = tran.result

                val isSuccess = result?.code == Trans.SUCCESS
                if (!isSuccess) {
                    eventListener().onEvent(SpeechHelper.CALLBACK_TRANS_FAIL)
                }
                !isSuccess
            }//
            .subscribeOn(cachedScheduler)
            .observeOn(AndroidSchedulers.mainThread())
            .doOnNext { tran ->
                eventListener().onEvent(SpeechHelper.CALLBACK_TRANSLATE_RESULT, bundle.apply {
                    putSerializable(SpeechHelper.EXTRA_TRANSLATE_RESULT, tran)
                })
            }
            .map { tran ->
                val result = tran.result
                result!!.translation//translationList.joinToString("\n")
            }
            .observeOn(cachedScheduler)
            .flatMap(::ttsImpl)
            .doOnError {
                it.printStackTrace()
                //println("SpeechHelper 播报失败：${it.message}")
            }
            // .onErrorReturnItem(File("###"))
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe({ file ->
                eventListener().onEvent(SpeechHelper.CALLBACK_TTS_RESULT, bundle.apply {
                    putString(SpeechHelper.EXTRA_TTS_RESULT, file.path)
                })

            }, { throwable: Throwable ->
                throwable.printStackTrace()
                eventListener().onEvent(SpeechHelper.CALLBACK_ERROR, bundle.apply {
                    putSerializable(SpeechHelper.EXTRA_THROWABLE, throwable)
                    putInt(SpeechHelper.EXTRA_ERROR_CODE, SpeechHelper.ERROR_OTHER)
                })
            })


    }

    @Synchronized
    fun transWithSpeech(text: String, historyId: String) {

        val sourceLanguage: LANG = builder.translateSourceLanguage!!
        val targetLanguage: LANG = builder.translateTargetLanguage!!

//        val p = Pattern.compile("\\s*|\t|\r|\n")
//        val m = p.matcher(text)
//        val newText = m.replaceAll("")
        //val array = arrayOf(text)
        val trans =
            Trans(sourceLanguage, targetLanguage, text)//arrayOf("今天天气如何?", "你", "今天开心吗", "今天天气如何")
        val rs = Trans.Result()
        rs.historyId = historyId
        trans.result = rs
        val translatorFlowable = Translator.translate(trans)
        translateDisposable = translatorFlowable
            .takeUntil { tran ->
                val it = tran.result
                it?.historyId = historyId
                val isSuccess = it?.code == Trans.SUCCESS
                if (!isSuccess) {
                    eventListener().onEvent(SpeechHelper.CALLBACK_TRANS_FAIL, bundle.apply {
                        putString(SpeechHelper.EXTRA_TTS_HISTORYID, historyId)
                    })
                }
                !isSuccess
            }//
            .subscribeOn(cachedScheduler)
            .observeOn(AndroidSchedulers.mainThread())
            .doOnNext { tran ->
                eventListener().onEvent(SpeechHelper.CALLBACK_TRANSLATE_RESULT, bundle.apply {
                    putSerializable(SpeechHelper.EXTRA_TRANSLATE_RESULT, tran)
                })
            }
            .map { tran ->
                val result2 = tran.result
                result2!!.translation//translationList.joinToString("\n")
            }
            .observeOn(cachedScheduler)
            .flatMap(::ttsImpl)
            .doOnError {
                it.printStackTrace()
                //println("SpeechHelper 播报失败：${it.message}")
            }
            // .onErrorReturnItem(File("###"))
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe({ file ->
                eventListener().onEvent(SpeechHelper.CALLBACK_TTS_RESULT, bundle.apply {
                    putString(SpeechHelper.EXTRA_TTS_RESULT, file.path)
                    putString(SpeechHelper.EXTRA_TTS_HISTORYID, historyId)
                })

            }, { throwable: Throwable ->
                throwable.printStackTrace()
                eventListener().onEvent(SpeechHelper.CALLBACK_ERROR, bundle.apply {
                    putSerializable(SpeechHelper.EXTRA_THROWABLE, throwable)
                    putString(SpeechHelper.EXTRA_TTS_HISTORYID, historyId)
                    putInt(SpeechHelper.EXTRA_ERROR_CODE, SpeechHelper.ERROR_OTHER)
                })
            })


    }

    private fun eventListener(): EventListener = builder.eventListener!!

    /*

    /**
     * Translate Flowable
     */
    private inline fun <reified T : TranslateFactory> getTranslateFlowable(text: String): Flowable<String> {
        val sourceLanguage: LANG = builder.translateSourceLanguage!!
        val targetLanguage: LANG = builder.translateTargetLanguage!!

        return Factory.createInstance<T>(builder.context).flatMap { translator ->
            val log = "text=$text sourceLanguage=${sourceLanguage.name} " +
                    " targetLanguage=${targetLanguage.name}  TranslateBy=${T::class.java.simpleName}"
            Logcat.d(msg = log)
            if (translator.isSupportLanguage(sourceLanguage) && translator.isSupportLanguage(targetLanguage)) {
                when (T::class.java.name) {
                    GoogleTranslate::class.java.name -> {
                        val googleTk = InputStreamReader(builder.context.assets.open("tk/Google.js"))
                        (translator as GoogleTranslate).tkReader = googleTk
                    }
                    BaiduTranslate::class.java.name -> {
                        val baiduTk = InputStreamReader(builder.context.assets.open("tk/Baidu.js"))
                        (translator as BaiduTranslate).tkReader = baiduTk
                    }
                }
                translator.setFormData(sourceLanguage, targetLanguage, text)

                translator.execute().flatMap { responseBody ->
                    Flowable.just(translator.parses(responseBody))
                }
            } else {
                throw Exception("not support language")
            }
        }
    }


   */


    /**
     * STT Flowable
     */
    private inline fun <reified T : STTFactory> getSTTFlowable(bytes: ByteArray): Flowable<String> {
        val speechSourceLanguage = builder.speechLanguage!!
        return Factory.createInstance<T>(builder.context).flatMap { stt ->

            if (stt.isSupportLanguage(speechSourceLanguage)) {
                val log =
                    "bytesLength=${bytes.size} speechSourceLanguage=${speechSourceLanguage.name}  STTBy=${T::class.java.simpleName}"
                Logcat.d(msg = log)

                stt.setRequestContent(bytes, speechSourceLanguage)

                val flowable = stt.execute()

                flowable.flatMap { responseBody ->
                    Flowable.just(stt.parses(responseBody))
                }

            } else {
                throw Exception("not support language")
            }
        }
    }

    /**
     * TTS Flowable
     */
    private fun getTTSFlowable(tts: TTSFactory, text: String): Flowable<File> {

        val textSpeechSourceLanguage: LANG =
            builder.textSpeechSourceLanguage
                ?: throw IllegalArgumentException("textSpeechSourceLanguage is null")


        return when (tts.javaClass.name) {
            GoogleTTS::class.java.name -> {

                //val googleTk = InputStreamReader(builder.context.resources.assets.open("tk/Google.js"))
                val googleTTS = (tts as GoogleTTS).apply {
                    //tkReader = googleTk
                    setFormData(textSpeechSourceLanguage, text)
                }
                googleTTS.execute().flatMap { res: ResponseBody -> googleTTS.parse(res) }
            }
            else -> {
                val baiduTTS = (tts as BaiduTTS)//BaiduTTS(context!!.applicationContext)
                baiduTTS.setFormData(textSpeechSourceLanguage, text)
                baiduTTS.execute().flatMap { res: ResponseBody -> baiduTTS.parse(res) }
            }
        }


//        val flowable = Factory.createInstance<T>(builder.context)
//        return flowable.flatMap { tts ->
//
//            if (tts.isSupportLanguage(textSpeechSourceLanguage!!)) {
//
//            } else {
//                //Flowable.empty()
//                throw Exception("not support language  textSpeechSourceLanguage=$textSpeechSourceLanguage  ${tts.javaClass.name}")
//            }
//
//        }

    }

}