package com.cy.sts.stt

import android.text.TextUtils
import com.baidu.speech.EventManagerFactory
import com.baidu.speech.asr.SpeechConstant
import com.cy.sts.STSEngine
import com.cy.sts.common.Logcat
import com.cy.sts.lang.RegionLang
import org.json.JSONObject
import java.util.*

/**
 * 百度语音识别 http://ai.baidu.com/docs#/ASR-Android-SDK/079ab014
 * @author CY 19-2-13
 */
class BaiduRecognitionHelper(stsEngine: STSEngine) : com.baidu.speech.EventListener, SpeechHelper(stsEngine) {

    override fun setLangSupport(langMap: HashMap<RegionLang, String>) {

        langMap[RegionLang.EN_AU] = "1737"//英语（澳大利亚）   English (Australia)
        langMap[RegionLang.EN_CA] = "1737"//英语（加拿大）   English (Canada)
        langMap[RegionLang.EN_GH] = "1737"//英语（加纳）   English (Ghana)
        langMap[RegionLang.EN_GB] = "1737"//英语（英国）   English (Great Britain)
        langMap[RegionLang.EN_IN] = "1737"//英语（印度）   English (India)
        langMap[RegionLang.EN_IE] = "1737"//英语（爱尔兰）   English (Ireland)
        langMap[RegionLang.EN_KE] = "1737"//英语（肯尼亚）   English (Kenya)
        langMap[RegionLang.EN_NZ] = "1737"//英语（新西兰）   English (New Zealand)
        langMap[RegionLang.EN_NG] = "1737"//英语（尼日利亚）   English (Nigeria)
        langMap[RegionLang.EN_PH] = "1737"//英语（菲律宾）   English (Philippines)
        langMap[RegionLang.EN_ZA] = "1737"//英语（南非）   English (South Africa)
        langMap[RegionLang.EN_TZ] = "1737"//英语（坦桑尼亚）   English (Tanzania)
        langMap[RegionLang.EN_US] = "1737"//英语（美国）   English (United States)

        langMap[RegionLang.CMN_HANT_TW] = "1536"//中文、普通话（台湾繁体）   國語 (台灣)
        langMap[RegionLang.YUE_HANT_HK] = "1637"//中文、粤语（香港繁体）   廣東話 (香港)
        langMap[RegionLang.CMN_HANS_HK] = "1536"//中文、普通话（香港简体）   普通話 (香港)
        langMap[RegionLang.CMN_HANS_CN] = "1537"//中文、普通话（中国简体）   普通话 (中国大陆)
    }

    /**
     * 提供音频录制
     */
    private val baiduASR by lazy { EventManagerFactory.create(context, "asr") }

    /**
     * 百度识别对象
     */
    private val baiduRecognition = BaiduRecognition(context)


    override fun startSpeech() {

        start()
    }

    override fun stopSpeech() {

        stop()
    }

    override fun cancelSpeech() {
        baiduASR.send(SpeechConstant.ASR_CANCEL, null, null, 0, 0)
    }

    override fun release() {
        cancelSpeech()
        baiduASR.unregisterListener(this)
    }

    private fun dispose() {
//        if (translateDisposable?.isDisposed == false) {
//            translateDisposable?.dispose()
//        }
    }

    /**
     * 开始识别，采用百度音频录制
     * 如果百度支持语言，则直接采用识别的结果，如果不支持，则采用Google识别
     */
    private fun start() {
        // dispose()
        val pid = if (baiduRecognition.isSupportLanguage(recognitionLang())) {
            baiduRecognition.langMap[recognitionLang()]
        } else {
            val default = Locale.getDefault()

            if (Locale.ENGLISH.language == default.language) {
                "1737"//识别英文
            } else {
                "1537"//识别大陆中文
            }
        }

        baiduASR.registerListener(this)

        val json = JSONObject()
            .put("pid", pid)
            .put("accept-audio-data", true)//true 返回音频数据
            .put("disable-punctuation", false)
            .put("accept-audio-volume", true)
            .toString()
        //  "{\"accept-audio-data\":false,\"disable-punctuation\":false,\"accept-audio-volume\":true,\"pid\":1736}"
        baiduASR.send(SpeechConstant.ASR_START, json, null, 0, 0)
    }

    private fun stop() {
        baiduASR.send(SpeechConstant.ASR_STOP, null, null, 0, 0)
    }


    override fun onEvent(name: String, params: String?, data: ByteArray?, offset: Int, length: Int) {
        when (name) {
            SpeechConstant.CALLBACK_EVENT_ASR_READY -> {
                // 引擎就绪，可以说话，一般在收到此事件后通过UI通知用户可以说话了

                eventListener().onEvent(CALLBACK_STT_READY)
            }

            SpeechConstant.CALLBACK_EVENT_ASR_BEGIN -> {

                eventListener().onEvent(CALLBACK_STT_BEGIN)
            }
            SpeechConstant.CALLBACK_EVENT_ASR_PARTIAL -> {//识别结果
                //{"results_recognition":["你"],"result_type":"partial_result","best_result":"你","origin_result":{"corpus_no":6652503758586274654,"err_no":0,"result":{"word":["你"]},"sn":"ea492eaf-e4e7-4b80-8193-028c078f5156"},"error":0}
                if (baiduRecognition.isSupportLanguage(recognitionLang()) && params != null) {

                    val jsonObject = JSONObject(params)
                    val text = jsonObject.getJSONArray("results_recognition")[0].toString()
                    partial(text)
                }

            }
            SpeechConstant.CALLBACK_EVENT_ASR_END -> {
                eventListener().onEvent(CALLBACK_STT_END)
            }
            SpeechConstant.CALLBACK_EVENT_ASR_FINISH -> {
                val json = JSONObject(params)
                val error = json.getInt("error")

                if (error == 0) {
                    success()
                } else {
                    error(error)
                }
                //{"origin_result":{"sn":"cuid=4AB5D1D34E9180EA6BB86ABFF5A576C1|0&sn=8afcc2dc-bc2e-4c47-a638-0283936577c9&nettype=3","error":2,"desc":"Download network read failed","sub_error":2005},"error":2,"desc":"Download network read failed","sub_error":2005}  data=null   offset=0  length=0
                // 识别结束

            }
            SpeechConstant.CALLBACK_EVENT_ASR_VOLUME -> {
                //返回音量大小
                //onEvent## name=asr.volume  params={"volume":287,"volume-percent":5}  data=null   offset=0  length=0
                val jsonObject = JSONObject(params)
                val volume = jsonObject.getDouble("volume").toFloat()
                val volumePercent = jsonObject.getInt("volume-percent")
                eventListener().onEvent(CALLBACK_STT_VOLUME, bundle.apply {
                    putFloat(EXTRA_STT_VOLUME, volume)
                    putInt(EXTRA_STT_VOLUME_PERCENT, volumePercent)
                })
//                println("BaiduRecognitionHelper#volume  volume=$volume  volume_percent=$volumePercent")
            }
            SpeechConstant.CALLBACK_EVENT_ASR_AUDIO -> {
                //返回音频数据
                //onEvent## name=asr.audio  params=null  data=640   offset=0  length=640
                eventListener().onEvent(CALLBACK_STT_AUDIO, bundle.apply { putByteArray(EXTRA_STT_AUDIO, data) })
//                if (!baiduRecognition.isSupportLanguage(recognitionLang()) &&
//                    this.javaClass.name != GoogleRecognition::class.java.name
//                ) {//不支持该语言的话
//
//                    //采用Google语音识别
//                }
            }

        }

        // ... 支持的输出事件和事件支持的事件参数见“输入和输出参数”一节
        val message = "onEvent## name=$name  params=$params  data=${data?.size}   offset=$offset  length=$length"
        Logcat.d(msg = message)
    }

    /**
     * 识别到的内容
     */
    private fun partial(text: String) {
        eventListener().onEvent(CALLBACK_STT_PARTIAL, bundle.apply {
            putString(EXTRA_STT_PARTIAL, text)
        })
    }

    /**
     * 识别结束
     */
    private fun success() {
        val text = bundle.getString(EXTRA_STT_PARTIAL)

        eventListener().onEvent(CALLBACK_STT_FINISH, bundle)

//        if (!TextUtils.isEmpty(text)) {
//            stsEngine.transWithSpeech(text!!)
//        }
    }

    /**
     * 错误
     */
    private fun error(error: Int) {

        eventListener().onEvent(CALLBACK_ERROR, bundle.apply { putInt(EXTRA_ERROR_CODE, error) })
    }
}


/*
    错误领域	描述	错误码	错误描述及可能原因
    1	网络超时		出现原因可能为网络已经连接但质量比较差，建议检测网络状态
    1000	DNS连接超时
    1001	网络连接超时
    1002	网络读取超时
    1003	上行网络连接超时
    1004	上行网络读取超时
    1005	下行网络连接超时
    1006	下行网络读取超时
    2	网络连接失败		出现原因可能是网络权限被禁用，或网络确实未连接，需要开启网络或检测无法联网的原因
    2000	网络连接失败
    2001	网络读取失败
    2002	上行网络连接失败
    2003	上行网络读取失败
    2004	下行网络连接失败
    2005	下行网络读取失败
    2006	下行数据异常
    2100	本地网络不可用
    3	音频错误		出现原因可能为：未声明录音权限，或 被安全软件限制，或 录音设备被占用，需要开发者检测权限声明。
    3001	录音机打开失败
    3002	录音机参数错误
    3003	录音机不可用
    3006	录音机读取失败
    3007	录音机关闭失败
    3008	文件打开失败
    3009	文件读取失败
    3010	文件关闭失败
    3100	VAD异常，通常是VAD资源设置不正确
    3101	长时间未检测到人说话，请重新识别
    3102	检测到人说话，但语音过短
    4	协议错误		出现原因可能是appid和appkey的鉴权失败
    4001	协议出错
    4002	协议出错
    4003	识别出错
    4004	鉴权错误 ，一般情况是pid appkey secretkey不正确
    5	客户端调用错误		一般是开发阶段的调用错误，需要开发者检测调用逻辑或对照文档和demo进行修复。
    5001	无法加载so库
    5002	识别参数有误
    5003	获取token失败
    5004	客户端DNS解析失败
    5005
    6	超时		语音过长，请配合语音识别的使用场景，如避开嘈杂的环境等
    6001	未开启长语音时，当输入语音超过60s时，会报此错误
    7	没有识别结果		信噪比差，请配合语音识别的使用场景，如避开嘈杂的环境等
    7001	没有匹配的识别结果。当检测到语音结束，或手动结束时，服务端收到的音频数据质量有问题，导致没有识别结果
    8	引擎忙		一般是开发阶段的调用错误，出现原因是上一个会话尚未结束，就让SDK开始下一次识别。SDK目前只支持单任务运行，即便创建多个实例，也只能有一个实例处于工作状态
    8001	识别引擎繁忙 。当识别正在进行时，再次启动识别，会报busy。
    9	缺少权限		参见demo中的权限设置
    9001	没有录音权限 通常是没有配置录音权限：android.permission.RECORD_AUDIO
    10	其它错误		出现原因如：使用离线识别但未将EASR.so集成到程序中；离线授权的参数填写不正确；参数设置错误等。
    10001	离线引擎异常
    10002	没有授权文件
    10003	授权文件不可用
    10004	离线参数设置错误
    10005	引擎没有被初始化
    10006	模型文件不可用
    10007	语法文件不可用
    10008	引擎重置失败
    10009	引擎初始化失败
    10010	引擎释放失败
    10011	引擎不支持
    10012	离线引擎识别失败 。离线识别引擎只能识别grammar文件中约定好的固定的话术，即使支持的话术，识别率也不如在线。请确保说的话清晰，是grammar中文件定义的，测试成功一次后，可以保存录音，便于测试。
*/