package com.cy.sts.stt

import android.content.Context
import android.os.Bundle
import com.cy.sts.EventListener
import com.cy.sts.STSEngine
import com.cy.sts.lang.RegionLang
import java.util.*

/**
 * @author CY 19-2-18
 */
abstract class SpeechHelper(val stsEngine: STSEngine) {

    val context: Context = stsEngine.builder.context
    fun recognitionLang(): RegionLang = stsEngine.builder.speechLanguage!!
    fun eventListener(): EventListener = stsEngine.builder.eventListener!!

    /**
     * 支持的语言Map
     */
    val langMap: HashMap<RegionLang, String> = HashMap()

    /**
     * 识别的语言
     */
    var recognitionLang: RegionLang? = null


    /**
     * 返回数据包
     */
    val bundle = Bundle()


    // var translateDisposable: Disposable? = null


    init {
        this.setLangSupport(langMap)
    }

    /**
     * 设置支持的语言
     */
    protected abstract fun setLangSupport(langMap: HashMap<RegionLang, String>)

    /**
     * 是否支持该语言
     */
    fun isSupportLanguage(lang: RegionLang): Boolean = langMap.contains(lang)

    abstract fun startSpeech()

    abstract fun stopSpeech()

    abstract fun cancelSpeech()

    abstract fun release()

    companion object {

        //发送事件码
        const val EVENT_STS_START = 1
        const val EVENT_STS_STOP = 2
        const val EVENT_TRANSLATE = 3
        const val EVENT_TTS = 4

        //结果返回码
        const val CALLBACK_STT_READY = 100//准备识别
        const val CALLBACK_STT_BEGIN = 101//开始识别
        const val CALLBACK_STT_PARTIAL = 102//识别部分结果返回，同时根据参数是否进行翻译结果返回
        const val CALLBACK_STT_END = 103//识别结束
        const val CALLBACK_STT_FINISH = 104//识别完成，同时根据参数是否返回播报音频数据
        //const val CALLBACK_STT_EXIT = 105//识别退出
        const val CALLBACK_TRANSLATE_RESULT = 106//单独翻译结果
        const val CALLBACK_TTS_RESULT = 107//播报结果
        const val CALLBACK_ERROR = 108//发生错误
        const val CALLBACK_STT_VOLUME = 109//音量
        const val CALLBACK_STT_AUDIO = 110//音频数据

        const val CALLBACK_TRANS_FAIL = 111//翻译失败

        //发送时附加数据
        const val EXTRA_TRANSLATE_TEXT = "extra_translate_text"//待翻译文本
        const val EXTRA_TTS_TEXT = "extra_text_to_speech_text"//待转换语音文本
        const val EXTRA_TRANSLATE_ENABLE = "extra_translate_enable"//是否开启翻译功能
        const val EXTRA_TTS_ENABLE = "extra_tts_enable"//是否开启text to speech

        //返回结果时附加数据
        const val EXTRA_STT_PARTIAL = "extra_stt_partial"//STT结果
        const val EXTRA_TRANSLATE_RESULT = "extra_translate_result"//翻译结果
        const val EXTRA_TTS_RESULT = "extra_tts_result"//TTS结果
        const val EXTRA_TTS_HISTORYID = "extra_tts_historyid"//HistoryId
        const val EXTRA_THROWABLE = "extra_throwable"// error  Throwable Serializable
        const val EXTRA_ERROR_CODE = "extra_error_code"// error code

        //Volume 数据
        const val EXTRA_STT_VOLUME = "extra_stt_volume"//STT 音量
        const val EXTRA_STT_VOLUME_PERCENT = "extra_stt_volume_percent"//STT 音量百分比
        const val EXTRA_STT_AUDIO = "extra_stt_audio"//STT 音频数据


        /** Network operation timed out. */
        const val ERROR_NETWORK_TIMEOUT = 1

        /** Other network related errors. */
        const val ERROR_NETWORK = 2

        /** Audio recording error. */
        const val ERROR_AUDIO = 3

        /** Server sends error status. */
        const val ERROR_SERVER = 4

        /** Other client side errors. */
        const val ERROR_CLIENT = 5

        /** No speech input */
        const val ERROR_NO_SPEECH_INPUT = 6

        /** No recognition result matched. */
        const val ERROR_NO_MATCH = 7

        /** RecognitionService busy. */
        const val ERROR_RECOGNIZER_BUSY = 8

        /** Insufficient permissions */
        const val ERROR_INSUFFICIENT_PERMISSIONS = 9

        const val ERROR_OTHER = 10//其它错误


/*
    错误领域	描述	错误码	错误描述及可能原因
    1	网络超时		出现原因可能为网络已经连接但质量比较差，建议检测网络状态
    1000	DNS连接超时
    1001	网络连接超时
    1002	网络读取超时
    1003	上行网络连接超时
    1004	上行网络读取超时
    1005	下行网络连接超时
    1006	下行网络读取超时
    2	网络连接失败		出现原因可能是网络权限被禁用，或网络确实未连接，需要开启网络或检测无法联网的原因
    2000	网络连接失败
    2001	网络读取失败
    2002	上行网络连接失败
    2003	上行网络读取失败
    2004	下行网络连接失败
    2005	下行网络读取失败
    2006	下行数据异常
    2100	本地网络不可用
    3	音频错误		出现原因可能为：未声明录音权限，或 被安全软件限制，或 录音设备被占用，需要开发者检测权限声明。
    3001	录音机打开失败
    3002	录音机参数错误
    3003	录音机不可用
    3006	录音机读取失败
    3007	录音机关闭失败
    3008	文件打开失败
    3009	文件读取失败
    3010	文件关闭失败
    3100	VAD异常，通常是VAD资源设置不正确
    3101	长时间未检测到人说话，请重新识别
    3102	检测到人说话，但语音过短
    4	协议错误		出现原因可能是appid和appkey的鉴权失败
    4001	协议出错
    4002	协议出错
    4003	识别出错
    4004	鉴权错误 ，一般情况是pid appkey secretkey不正确
    5	客户端调用错误		一般是开发阶段的调用错误，需要开发者检测调用逻辑或对照文档和demo进行修复。
    5001	无法加载so库
    5002	识别参数有误
    5003	获取token失败
    5004	客户端DNS解析失败
    5005
    6	超时		语音过长，请配合语音识别的使用场景，如避开嘈杂的环境等
    6001	未开启长语音时，当输入语音超过60s时，会报此错误
    7	没有识别结果		信噪比差，请配合语音识别的使用场景，如避开嘈杂的环境等
    7001	没有匹配的识别结果。当检测到语音结束，或手动结束时，服务端收到的音频数据质量有问题，导致没有识别结果
    8	引擎忙		一般是开发阶段的调用错误，出现原因是上一个会话尚未结束，就让SDK开始下一次识别。SDK目前只支持单任务运行，即便创建多个实例，也只能有一个实例处于工作状态
    8001	识别引擎繁忙 。当识别正在进行时，再次启动识别，会报busy。
    9	缺少权限		参见demo中的权限设置
    9001	没有录音权限 通常是没有配置录音权限：android.permission.RECORD_AUDIO
    10	其它错误		出现原因如：使用离线识别但未将EASR.so集成到程序中；离线授权的参数填写不正确；参数设置错误等。
    10001	离线引擎异常
    10002	没有授权文件
    10003	授权文件不可用
    10004	离线参数设置错误
    10005	引擎没有被初始化
    10006	模型文件不可用
    10007	语法文件不可用
    10008	引擎重置失败
    10009	引擎初始化失败
    10010	引擎释放失败
    10011	引擎不支持
    10012	离线引擎识别失败 。离线识别引擎只能识别grammar文件中约定好的固定的话术，即使支持的话术，识别率也不如在线。请确保说的话清晰，是grammar中文件定义的，测试成功一次后，可以保存录音，便于测试。
*/

    }
}