package com.talpa.tengine.free

import com.google.gson.Gson
import com.google.gson.JsonElement
import com.talpa.tengine.lang.LANG
import com.talpa.tengine.request.RequestHelper
import io.reactivex.Flowable
import okhttp3.MediaType
import okhttp3.RequestBody
import okhttp3.ResponseBody

/**
 *  微软翻译
 *  @author CY 2019-01-13
 */
class BingTranslate : TranslateFactory() {

    override val isSupportMultiTextTranslate: Boolean = false

    override fun parses(response: retrofit2.Response<ResponseBody>): List<String> {
        //{"statusCode":200,"translationResponse":"How are you doing"}
        if (response.isSuccessful) {
            val text = response.body()?.string()
            if (text == null || text == "") {
                throw Exception("${this.javaClass.simpleName} text=$text")
            }

            /*[
                {
                    "detectedLanguage": {
                    "language": "zh-Hans",
                    "score": 1
                },
                    "translations": [
                    {
                        "text": "How are you doing",
                        "to": "en"
                    }
                    ]
                }
            ]*/

            //{"statusCode":400}

            val jsonElements = Gson().fromJson<JsonElement>(text, JsonElement::class.java)
            if (jsonElements.isJsonObject) {
                throw Exception("${this.javaClass.simpleName} text=$text")
            }
            val jsonArray = jsonElements.asJsonArray

            if (jsonArray.size() == 0) {
                throw Exception("${this.javaClass.simpleName} text=$text")
            }
            val jo = jsonArray[0].asJsonObject
            val language = jo["detectedLanguage"].asJsonObject["language"].asString
            val translations = jo["translations"].asJsonArray
            val list = ArrayList<String>()
            for (trans in translations) {
                list.add(trans.asJsonObject["text"].asString)
            }
           // detectLang = microsoftCodeToLang(language)
            return list
        }

        throw Exception("Bing Code=${response.code()}")

    }

    override fun setFormData(from: String, to: String, text: String): TranslateFactory {
        //formData["from"] = langMap[from]!!

        val fromLang = if (from == LANG.AUTO) {
            "auto-detect"
        } else {
            langMap[from]!!
        }
        formData["fromLang"] = fromLang
        formData["to"] = langMap[to]!!
        formData["text"] = text



        return this
    }

    override fun execute(): Flowable<retrofit2.Response<ResponseBody>> {

        val headMap = hashMapOf(
            "Content-Type" to "application/x-www-form-urlencoded",
            "user-agent" to "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/80.0.3987.132 Safari/537.36"
        )
        val mediaType =
            MediaType.parse("application/x-www-form-urlencoded")

        val fromLang = formData["fromLang"]
        val to = formData["to"]
        val text = formData["text"]

        val content = "fromLang=$fromLang&text=$text&to=$to"

        val requestBody = RequestBody.create(mediaType, content)
        return RequestHelper.postBody(
            TRANS_BASE_URL_V3,
            TRANS_PATHS_V3,
            requestBody = requestBody,
            headerMap = headMap
        )

    }


    override fun setLangSupport(langMap: HashMap<String, String>) {
        langMap[LANG.AUTO] = "auto-detect"
        langMap[LANG.NL] = "af"//南非荷兰语
        langMap[LANG.AR] = "ar"//阿拉伯语
        langMap[LANG.BG] = "bg"//保加利亚语
        langMap[LANG.BN] = "bn-BD"//孟加拉语
        langMap[LANG.FA] = "bs-Latn"//波斯尼亚语
        langMap[LANG.CA] = "ca"//加泰隆语
        langMap[LANG.CS] = "cs"//捷克语
        langMap[LANG.CY] = "cy"//威尔士语
        langMap[LANG.DA] = "da"//丹麦语
        langMap[LANG.DE] = "de"//德语
        langMap[LANG.EL] = "el"//希腊语
        langMap[LANG.EN] = "en"//英语
        langMap[LANG.ES] = "es"//西班牙语
        langMap[LANG.ET] = "et"//爱沙尼亚语
        langMap[LANG.FA] = "fa"//波斯语
        langMap[LANG.FI] = "fi"//芬兰语
        langMap[LANG.TL] = "fil"//菲律宾语
        //langMap[LANG.FJ] = "fj"//斐济
        langMap[LANG.FR] = "fr"//法语
        langMap[LANG.HE] = "he"//希伯来语
        langMap[LANG.HI] = "hi"//印地语
        langMap[LANG.HR] = "hr"//克罗地亚语
        langMap[LANG.HT] = "ht"//海地克里奥尔语
        langMap[LANG.HU] = "hu"//匈牙利语
        langMap[LANG.ID] = "id"//印度尼西亚语
        langMap[LANG.IS] = "is"//冰岛语
        langMap[LANG.IT] = "it"//意大利语
        langMap[LANG.JA] = "ja"//日语
        langMap[LANG.KO] = "ko"//朝鲜语
        langMap[LANG.LT] = "lt"//立陶宛语
        langMap[LANG.LV] = "lv"//拉脱维亚语
        langMap[LANG.MG] = "mg"//马尔加什语
        langMap[LANG.MS] = "ms"//马来语
        langMap[LANG.MT] = "mt"//马耳他语
        langMap[LANG.HMN] = "mww"//白苗文
        langMap[LANG.NO] = "nb"//挪威语
        langMap[LANG.NL] = "nl"//荷兰语
        //langMap[LANG.OTQ] = "otq"//克雷塔罗奥托米语
        langMap[LANG.PL] = "pl"//波兰语
        langMap[LANG.PT] = "pt"//葡萄牙语
        langMap[LANG.RO] = "ro"//罗马尼亚语
        langMap[LANG.RU] = "ru"//俄语
        langMap[LANG.SK] = "sk"//斯洛伐克语
        langMap[LANG.SL] = "sl"//斯洛文尼亚语
        langMap[LANG.SM] = "sm"//萨摩亚语
        langMap[LANG.SR] = "sr-Cyrl"//塞尔维亚语 (西里尔文)
        langMap[LANG.ST] = "sr-Latn"//塞尔维亚语 (拉丁文)
        langMap[LANG.SV] = "sv"//瑞典语
        langMap[LANG.SW] = "sw"//斯瓦希里语
        langMap[LANG.TA] = "ta"//泰米尔语
        langMap[LANG.TE] = "te"//泰卢固语
        langMap[LANG.TH] = "th"//泰语
        //langMap[LANG.FILH] = "tlh"//克林贡语
        //langMap[LANG.TO] = "to"//汤加语
        langMap[LANG.TR] = "tr"//土耳其语
        //langMap[LANG.TY] = "ty"//塔希提语
        langMap[LANG.UK] = "uk"//乌克兰语
        langMap[LANG.UR] = "ur"//乌尔都语
        langMap[LANG.VI] = "vi"//越南语
        //langMap[LANG.YUA] = "yua"//尤卡坦玛雅语
        //langMap[LANG.YUE] = "yue"//粤语(繁体)
        langMap[LANG.ZH_CN] = "zh-Hans"//简体中文
        langMap[LANG.ZH_TW] = "zh-Hant"//繁体中文
    }


    companion object {
        //https://cn.bing.com/ttranslatev3?isVertical=1&&IG=4F67D73793144875B9BD98B0E6F4C6C3&IID=translator.5028.3

        //https://cn.bing.com/ttranslatev3?fromLang=auto-detect&text=he&to=fr
        private const val TRANS_BASE_URL_V3 = "https://cn.bing.com/"
        private const val TRANS_PATHS_V3 = "ttranslatev3"
    }
}

fun main() {
    val translate = BingTranslate()

    val startTime = System.currentTimeMillis()
    translate.setFormData(LANG.EN, LANG.HI, "hello")
    val d = translate.execute().subscribe({

        val t = translate.parses(it)
        println("response=$t  ${time(startTime)}s")
    }, {
        it.printStackTrace()
        //println("error  ${time(startTime)}s")
    })

}

private fun time(startTime: Long): String {
    val lastTime = System.currentTimeMillis()
    val time = String.format("%.2f", (lastTime - startTime) / 1000.0)
    // println("onFailure  $time s")
    return time
}