package com.talpa.overlay.view

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.graphics.PixelFormat
import android.graphics.Rect
import android.os.Build
import android.os.Message
import android.text.TextUtils
import android.view.Gravity
import android.view.MotionEvent
import android.view.View
import android.view.WindowManager
import android.view.accessibility.AccessibilityNodeInfo
import android.widget.EditText
import android.widget.ProgressBar
import android.widget.TextView
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.overlay.R
import com.talpa.overlay.RxRelay
import com.talpa.overlay.RxRelay.registerByEventBus
import com.talpa.overlay.RxRelay.unregisterByEventBus
import com.talpa.overlay.data.readOverlayEditTextLanguageTag
import com.talpa.overlay.data.readOverlayTextLanguageTag
import com.talpa.overlay.data.writeOverlayTextLanguageTag
import com.talpa.overlay.service.AccessService
import com.talpa.overlay.service.nodeText
import com.talpa.overlay.state.FloatingStateMachine
import com.talpa.overlay.state.FloatingStateMachine.grammarCheckEnable
import com.talpa.overlay.translate.translateForAllNodes
import com.talpa.overlay.view.FloatingManager.FROM
import com.talpa.overlay.view.FloatingManager.WHOLE
import com.talpa.overlay.view.overlay.BaseOverlayView
import com.talpa.overlay.view.overlay.OverlayImpl
import com.talpa.overlay.view.overlay.OverlayImpl.Companion.EXTRA_ERROR
import com.talpa.tengine.Trans
import com.talpa.tengine.UNKNOWN
import com.talpa.tengine.lang.LANG
import com.talpa.translate.repository.box.translate.CombinedTrans
import io.reactivex.functions.Consumer
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import org.greenrobot.eventbus.Subscribe
import org.greenrobot.eventbus.ThreadMode
import org.jetbrains.anko.find
import org.jetbrains.anko.layoutInflater
import org.jetbrains.anko.textColorResource
import java.util.*
import kotlin.collections.HashMap
import kotlin.collections.LinkedHashSet


/**
 * 多个覆盖视图,即全局翻译
 * @author CY 2019-09-29
 */
class MultiOverlayView(val context: Context) {

    companion object {

        /**
         * max sum
         */
        const val MAX_SUM = 30
    }

    private val views by lazy { arrayListOf<View>() }

    private var nodeSet: LinkedHashSet<AccessibilityNodeInfo>? = null

    private val nodeRectMap: HashMap<String, Rect> = hashMapOf()

    /**
     * 最小高度限制
     */
    private val limitHeight by lazy { context.resources.getDimension(R.dimen.dp16) }

    /**
     * 本地广播
     */
    private val localBroadcastManager by lazy { LocalBroadcastManager.getInstance(context) }

    /**
     * 是否需要打点
     */
    private var shouldLogEvent = false

    /**
     * 应用包名
     */
    private var packageName: String? = null

    /**
     * 开始全局翻译
     */
    fun startMultiOverlay() {
        shouldLogEvent = true
        registerByEventBus(this)
        AccessService.startServiceForStartMultiNodes(context)
    }

    /**
     * 停止全局翻译
     */
    fun stopMultiOverlay() {
        shouldLogEvent = false
        //removeOverlayViews()
        //AccessService.startServiceForStopMultiNodes(context)
        unregisterByEventBus(this)
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onMessageEvent(message: Message) {
        when (message.what) {
            RxRelay.EVENT_NODE_INFO_MULTI -> {

                if (FloatingStateMachine.currentState() !is FloatingStateMachine.FloatingState.HighLightState) {
                    return
                }

                val nodeSet = message.obj as? LinkedHashSet<AccessibilityNodeInfo> ?: return

                removeOverlayViews()
                addOverlayViews(nodeSet)
                this.nodeSet = nodeSet

                if (shouldLogEvent) {
                    shouldLogEvent = false
                    /*val targetLanguageTag = readOverlayTextLanguageTag(context)

                    if (targetLanguageTag != null) {
                        logStartTranslate(
                            targetLanguageLocaleTag = targetLanguageTag,
                            packageName = packageName ?: ""
                        )
                    }*/

                }
            }
        }
    }

    /**
     * 添加悬浮视图
     */
    private fun addOverlayViews(set: LinkedHashSet<AccessibilityNodeInfo>) {
        for ((index, nodeInfo) in set.withIndex()) {

            if (index >= MAX_SUM) {
                break
            }

            val rectInScreen = Rect()
            nodeInfo.getBoundsInScreen(rectInScreen)

            if (rectInScreen.height() < limitHeight) {
                continue
            }

            val filter = nodeInfo.className != EditText::class.java.name

            if (filter) {
                views.add(createOverlayView(nodeInfo, rectInScreen))
            }

        }

    }

    /**
     * 移除悬浮视图
     */
    private fun removeOverlayViews() {

        val wm = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
        for (view in views) {
            /*if (view.isAttachedToWindow) {
                wm.removeView(view)
            }*/
            try {
                wm.removeView(view)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
        this.nodeSet?.map { info -> info.recycle() }
        this.nodeSet?.clear()
        views.clear()
        nodeRectMap.clear()
    }

    /**
     * 创建悬浮视图
     */
    @SuppressLint("ClickableViewAccessibility")
    @Synchronized
    private fun createOverlayView(nodeInfo: AccessibilityNodeInfo, rectInScreen: Rect): View {
//        val rectInScreen = Rect()
//        nodeInfo.getBoundsInScreen(rectInScreen)

        val contentView = context.layoutInflater.inflate(R.layout.layout_content_view_simple, null)

        val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)

        val content = nodeInfo.nodeText()

        packageName = nodeInfo.packageName?.toString()

        //tvTranslation.text = content

        val sourceText = content.toString().trim()
        if (!TextUtils.isEmpty(content)) {
            trans(tvTranslation, sourceText, contentView, false)
        }


        nodeRectMap[content.toString()] = rectInScreen

        val params = overlayParams()

        params.x = rectInScreen.left
        params.y = rectInScreen.top
        params.width = rectInScreen.width()
        params.height = rectInScreen.height()

        /*if (BuildConfig.DEBUG) {
            if (params.height > 400) {
                val drawingOrder = nodeInfo.drawingOrder
                println("params.height>>>400  =${params.height}  content=$content \t drawingOrder=$drawingOrder\t   nodeInfo=$nodeInfo")
            }
        }*/


        val wm = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager

        if (contentView.parent == null && !contentView.isAttachedToWindow) {
            try {
                wm.addView(contentView, params)
            } catch (e: Exception) {
                e.printStackTrace()
            }

        }

        (contentView as BaseOverlayView).setOnBackClickListener = object : BaseOverlayView.BackClickListener {
            override fun onBackClickListener(): Boolean {
                var flag = false
                for (view in views) {
                    try {
                        if (view.isAttachedToWindow) {
                            wm.removeView(view)
                            flag = true
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
                return flag
            }

        }

        contentView.setTag(R.id.id_content_view_node_info, nodeInfo)

        contentView.setOnTouchListener { v, event ->

            if (event.action == MotionEvent.ACTION_OUTSIDE) {
                removeOverlayViews()
                return@setOnTouchListener true
            }
            return@setOnTouchListener false
        }
        return contentView
    }


    /**
     * 悬浮视图Params
     */
    private fun overlayParams(layoutParams: WindowManager.LayoutParams? = null): WindowManager.LayoutParams {
        val type = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
            WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY
        else WindowManager.LayoutParams.TYPE_PHONE
        val flags = WindowManager.LayoutParams.FLAG_WATCH_OUTSIDE_TOUCH or
                WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL or
                /*WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE or*/
                WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN or
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED
        val params =
            layoutParams ?: WindowManager.LayoutParams(type, flags, PixelFormat.TRANSLUCENT)
        //
        params.x = 0
        params.y = 0
        params.width = 0
        params.height = 0

        params.gravity = Gravity.START or Gravity.TOP

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            params.layoutInDisplayCutoutMode =
                WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES
        }

        params.format = PixelFormat.RGBA_8888
        return params
    }

    private fun showProgressBar(contentView: View?) {

        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.VISIBLE
    }

    private fun hideProgressBar(contentView: View?) {
        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.GONE
    }

    /**
     * 处理翻译逻辑
     */
    private fun trans(tvTranslation: TextView, sourceText: String, contentView: View, needGrammarCheck: Boolean) {
        val targetLanguageTag = readOverlayEditTextLanguageTag(context) ?: LANG.EN
        val sourceLanguageTag = readOverlayTextLanguageTag(context) ?: Locale.getDefault().language

        //targetLanguageTag
        //    ?: throw Exception("please set text translate targetLanguage,to see writeOverlayTextLanguageTag()")

//        if (!context.isNetworkConnected()) {
//            tvTranslation.setText(R.string.network_no_connect)
//            tvTranslation.textColorResource = R.color.color_floating_failure
//            hideProgressBar(contentView)
//            return
//        }

        //  showProgressBar(contentView)


        if (targetLanguageTag != null) {
            logStartTranslate(
                targetLanguageLocaleTag = targetLanguageTag,
                packageName = packageName ?: "",
                sourceText = sourceText
            )
        }

        val onNext = Consumer<CombinedTrans> { combinedTrans ->
            hideProgressBar(contentView)
            when (combinedTrans.trans?.result?.code) {
                Trans.SUCCESS -> {
                    tvTranslation.text = combinedTrans.trans?.result?.translation
                    logSuccessTranslate(
                        targetLanguageTag = targetLanguageTag ?: "",
                        packageName = packageName ?: "",
                        source = combinedTrans.trans?.result?.source ?: UNKNOWN
                    )
                }
//                Trans.ERROR_NETWORK -> {
//                    val message = context.resources.getString(R.string.network_error)
//                    tvTranslation.text = message
//                    tvTranslation.textColorResource = R.color.color_floating_failure
//                    logFailTranslate(message)
//                }
                /*Trans.ERROR_NETWORK_TIMEOUT -> {
                    tvTranslation.setText(R.string.network_timeout)
                    tvTranslation.textColorResource = R.color.color_floating_failure
                }*/
                else -> {
                    tvTranslation.setText(R.string.text_translating_error)
                    tvTranslation.textColorResource = R.color.color_floating_failure
                    val message = combinedTrans.trans?.result?.errorMessage ?: "failure"
                    logFailTranslate(message)
                }
            }
        }
        val onError = Consumer<Throwable> { throwable ->
            tvTranslation.setText(R.string.text_translating_error)
            tvTranslation.textColorResource = R.color.color_floating_failure
            throwable?.printStackTrace()
            hideProgressBar(contentView)
            val message = throwable?.message
            if (message != null) {
                logFailTranslate(message)
            }

        }


        when {
            targetLanguageTag != null -> {
                val disposable = translateForAllNodes(
                    tvTranslation,
                    sourceText,
                    sourceLanguageTag,
                    targetLanguageTag,
                    onNext,
                    onError,
                    needGrammarCheck
                )
            }
            /*else -> {
                showLanguageListDialog(context) { locale ->
                    val tag = locale.toLanguageTag()
                    writeOverlayTextLanguageTag(context, tag)
                    val disposable =
                        translateForAllNodes(tvTranslation, sourceText, tag, onNext, onError, needGrammarCheck)
                }

            }*/
        }
    }


    /**
     * 打点开始翻译
     */
    private fun logStartTranslate(
        targetLanguageLocaleTag: String,
        packageName: String,
        sourceText: String
    ) {

        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_START).apply {
            putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(FROM, WHOLE)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
        }

        GlobalScope.launch {
            localBroadcastManager.sendBroadcast(intent)
        }
    }

    private fun logSuccessTranslate(
        targetLanguageTag: String,
        packageName: String,
        source: String
    ) {
        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_SUCCESS).apply {
            //  putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageTag)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
            putExtra(OverlayImpl.EXTRA_SOURCE, source)
        }

        localBroadcastManager.sendBroadcast(intent)
    }

    private fun logFailTranslate(message: String) {


        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_ERROR).apply {
            //  putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            //putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
            //putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
            putExtra(EXTRA_ERROR, message)
        }

        localBroadcastManager.sendBroadcast(intent)
    }
}