package com.talpa.overlay.view.overlay

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.graphics.Rect
import android.os.Binder
import android.os.Bundle
import android.os.Handler
import android.text.TextUtils
import android.util.Log
import android.view.MotionEvent
import android.view.View
import android.view.WindowManager
import android.view.accessibility.AccessibilityNodeInfo
import android.widget.ProgressBar
import android.widget.TextView
import android.widget.Toast
import com.talpa.overlay.R
import com.talpa.overlay.data.readEditNodeInfo
import com.talpa.overlay.data.readOverlayEditTextLanguageTag
import com.talpa.overlay.data.readOverlayTextLanguageTag
import com.talpa.overlay.data.writeEditNodeInfo
import com.talpa.overlay.service.nodeText
import com.talpa.overlay.state.FloatingStateMachine
import com.talpa.overlay.state.FloatingStateMachine.grammarCheckEnable
import com.talpa.overlay.translate.translateForEditText
import com.talpa.overlay.view.FloatingContainer
import com.talpa.tengine.STORE
import com.talpa.tengine.Trans
import com.talpa.tengine.UNKNOWN
import com.talpa.tengine.lang.LANG
import com.talpa.translate.repository.box.ObjectBox
import com.talpa.translate.repository.box.translate.CombinedTrans
import com.trello.rxlifecycle3.kotlin.bindToLifecycle
import io.reactivex.functions.Consumer
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import org.jetbrains.anko.find
import org.jetbrains.anko.layoutInflater
import org.jetbrains.anko.textColorResource
import java.util.*

/**
 * EDIT Overlay
 *
 * @author CY 2019-11-13
 *
 */
class EditTextOverlayView(context: Context, private val handler: Handler? = null) :
    OverlayImpl(context) {

    companion object {
        const val TAG = "EditTextOverlayView"
    }

    private val rectInScreen = Rect()

    @SuppressLint("ClickableViewAccessibility")
    override fun createContentView(): View {
        val view = context.layoutInflater.inflate(R.layout.layout_content_edit, null)
        view.setOnTouchListener { v, event ->

            when (event.action) {
                MotionEvent.ACTION_OUTSIDE -> {
                    removeContentView()
                }
            }

            return@setOnTouchListener false
        }

        (view as BaseOverlayView).setOnBackClickListener = object : BaseOverlayView.BackClickListener {
            override fun onBackClickListener(): Boolean {
                try {
                    if (view.isAttachedToWindow) {
                        windowManager.removeView(view)
                        if (grammarDetailView.isAttachedToWindow) {
                            windowManager.removeView(grammarDetailView)
                        }
                        return true
                    } else {
                        return false
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
                return false
            }
        }

        return view
    }

    override fun addContentView(x: Int, y: Int) {
        super.addContentView(x, y)
        val params = overlayParams(contentView.layoutParams as? WindowManager.LayoutParams)
        params.x = x
        params.y = y

        contentView.visibility = View.GONE
        try {
            if (!contentView.isAttachedToWindow && contentView.parent == null) {
                windowManager.addView(contentView, params)
            } else {
                windowManager.updateViewLayout(contentView, params)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun addContentView(params: WindowManager.LayoutParams) {
        if (!contentView.isAttachedToWindow && contentView.parent == null) {
            try {
                windowManager.addView(contentView, params)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    override fun updateContentView(nodeInfo: AccessibilityNodeInfo) {

        editTextNodeInfo(nodeInfo)
    }


    private fun showContentView(nodeInfo: AccessibilityNodeInfo) {
        val params = overlayParams(contentView.layoutParams as? WindowManager.LayoutParams)
        params.flags = params.flags or WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE //这里不加上 not focusable的话会抢焦点导致输入法收起来

        addContentView(params)

        nodeInfo.getBoundsInScreen(rectInScreen)
        // val params = contentView.layoutParams as WindowManager.LayoutParams
        params.x = rectInScreen.left
        params.y = rectInScreen.top
        params.width = rectInScreen.width()
        params.height = rectInScreen.height()

        //contentView.setBackgroundResource(R.drawable.shape_edit_bg)
        if (contentView.visibility != View.VISIBLE) {
            contentView.visibility = View.VISIBLE
        }

        if (FloatingStateMachine.grammarCheckEnable) {
            contentView.setBackgroundResource(R.drawable.shape_overlay_grammar_bg)
        } else {
            contentView.setBackgroundResource(R.drawable.shape_overlay_bg)
        }

        try {
            windowManager.updateViewLayout(contentView, params)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    override fun removeContentView() {
        if (contentView.isAttachedToWindow) {
            try {
                val tvStatus = contentView.findViewById<TextView>(R.id.tv_status)
                tvStatus?.visibility = View.GONE
                tvStatus?.text = ""
                windowManager.removeViewImmediate(contentView)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
        if (grammarDetailView.isAttachedToWindow) {
            try {
                windowManager.removeViewImmediate(grammarDetailView)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
        super.removeContentView()
    }

    /**
     * 处理文本编辑框节点
     */
    private fun editTextNodeInfo(nodeInfo: AccessibilityNodeInfo) {

        val text = nodeInfo.nodeText()

        val sourceText = text.toString().trim()

        val packageName = nodeInfo.packageName?.toString() ?: ""

        if (!TextUtils.isEmpty(sourceText)) {

            val targetText = restoreEditInfo(nodeInfo)
            val targetLanguageTag = readOverlayEditTextLanguageTag(context) ?: LANG.EN
            val sourceLanguageTag = readOverlayTextLanguageTag(context) ?: Locale.getDefault().language
            //targetLanguageTag
            //?: throw Exception("please set text translate targetLanguage,to see writeOverlayEditTextLanguageTag()")

            if (targetText == null) {
                dealNodeInfo(sourceText, sourceLanguageTag, targetLanguageTag, packageName, nodeInfo)
            } else {
                executorAction(nodeInfo, targetText)
            }
        }
    }

    /**
     * Restore Edit Text Node Info
     */
    private fun restoreEditInfo(
        nodeInfo: AccessibilityNodeInfo
    ): String? {
        val languageTag = readOverlayEditTextLanguageTag(context) ?: return null
        return readEditNodeInfo(context, nodeInfo, languageTag)
    }

    /**
     * Store Edit Text Node Info
     */
    private fun storeEditInfo(
        text: String,
        translation: String,
        toLanguageTag: String,
        nodeInfo: AccessibilityNodeInfo
    ) {
        writeEditNodeInfo(context, text, translation, toLanguageTag, nodeInfo)
    }

    /**
     * 执行无障碍服务动作
     */
    private fun executorAction(nodeInfo: AccessibilityNodeInfo, text: String) {
        try {
            val arguments = Bundle()
            arguments.putCharSequence(AccessibilityNodeInfo.ACTION_ARGUMENT_SET_TEXT_CHARSEQUENCE, text)
            //可能被回收
            nodeInfo.performAction(AccessibilityNodeInfo.ACTION_SET_TEXT, arguments)
        } catch (e: Exception) {
            //e.printStackTrace()
        }

    }

    private fun showProgressBar(contentView: View?) {
        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.VISIBLE
    }

    private fun hideProgressBar(contentView: View?) {
        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.GONE
    }

    /**
     * 翻译前
     */
    private fun preTranslate(nodeInfo: AccessibilityNodeInfo) {
        showContentView(nodeInfo)
        showProgressBar(contentView)
    }


    /**
     * Success Trans
     */
    private fun successTranslate(
        sourceText: String,
        translation: String,
        targetLanguageTag: String,
        nodeInfo: AccessibilityNodeInfo
    ) {

        executorAction(nodeInfo, translation)

        storeEditInfo(sourceText, translation, targetLanguageTag, nodeInfo)

        hideProgressBar(contentView)

        handler?.sendEmptyMessage(FloatingContainer.WHAT_TRANSLATE_SUCCESS)
    }

    /**
     * Failure Trans
     */
    private fun failureTranslate(
        sourceText: String,
        targetLanguageTag: String,
        packageName: String,
        errorMessage: String
    ) {
        val tvStatus = contentView.findViewById<TextView>(R.id.tv_status)
        tvStatus?.setText(R.string.text_translating_error)
        tvStatus?.textColorResource = R.color.color_floating_failure
        tvStatus?.visibility = View.VISIBLE
        logFailTranslate(
            ACTION_EDIT_TEXT_TRANSLATE_ERROR,
            sourceText,
            targetLanguageTag,
            packageName,
            errorMessage = errorMessage
        )
        hideStatus(tvStatus)
        hideProgressBar(contentView)

        handler?.sendEmptyMessage(FloatingContainer.WHAT_TRANSLATE_FAILURE)

    }

    private fun dealNodeInfo(
        sourceText: String,
        sourceLanguageTag: String,
        targetLanguageTag: String,
        packageName: String,
        nodeInfo: AccessibilityNodeInfo
    ) {
        showProgressBar(contentView)
        logStartTranslate(
            ACTION_EDIT_TEXT_TRANSLATE_START,
            sourceText,
            targetLanguageTag,
            packageName
        )

        /**
         *  it.text,
         *  it.translation,
         *  it.sourceLanguageLocaleTag,
         *  it.targetLanguageLocaleTag,
         *  it.star
         */
        val d = readTranslateHistory(sourceText, sourceLanguageTag, targetLanguageTag)
            .bindToLifecycle(contentView)
            .subscribe({

                if (!TextUtils.isEmpty(it.text) && !TextUtils.isEmpty(it.translation)) {

                    logSuccessTranslate(
                        ACTION_EDIT_TEXT_TRANSLATE_SUCCESS,
                        sourceText,
                        it.sourceLanguageTag,
                        it.targetLanguageTag,
                        packageName,
                        isCached = true,
                        source = STORE
                    )
                    if (sourceText == it.translation) {
                        //撤回
                        executorAction(nodeInfo, it.text)
                        hideProgressBar(contentView)
                    } else {
                        successTranslate(
                            sourceText,
                            it.translation,
                            targetLanguageTag,
                            nodeInfo
                        )
                    }
                } else {
                    translate(sourceText, sourceLanguageTag, targetLanguageTag, packageName, nodeInfo)
                }
            }, {
                it.printStackTrace()
                translate(sourceText, sourceLanguageTag, targetLanguageTag, packageName, nodeInfo)
            })
    }

    private fun translate(
        sourceText: String,
        sourceLanguageTag: String,
        targetLanguageTag: String,
        packageName: String,
        nodeInfo: AccessibilityNodeInfo
    ) {


        preTranslate(nodeInfo)

        val onSuccess = Consumer { combinedTrans: CombinedTrans ->
            when (combinedTrans.trans?.result?.code) {
                Trans.SUCCESS -> {
                    val translation = combinedTrans?.trans?.result?.translation

                    if (translation != null) {

                        val sourceLanguage = combinedTrans?.trans?.result!!.detectLang ?: "AUTO"
                        logSuccessTranslate(
                            ACTION_EDIT_TEXT_TRANSLATE_SUCCESS,
                            sourceText,
                            sourceLanguage,
                            targetLanguageTag,
                            packageName,
                            source = combinedTrans?.trans?.result?.source ?: UNKNOWN
                        )
                        insertTranslateHistory(sourceText, translation, targetLanguageTag)

                        successTranslate(sourceText, translation, targetLanguageTag, nodeInfo)

                        if (grammarCheckEnable) {

                            combinedTrans?.grammarNew?.let { grammarNew ->
                                if (grammarNew.results?.alerts.isNullOrEmpty()) {
                                    contentView.post {
                                        Toast.makeText(context,
                                            context.getString(R.string.grammar_no_error_tips1), Toast.LENGTH_SHORT)
                                            .show()
                                        removeContentView()
                                    }
                                    return@Consumer
                                }
                                grammarDetailView.visibility = View.VISIBLE
                                grammarDetailView.setText(context.getString(R.string.grammar_detail, grammarNew.results?.alerts?.size ?: 1))
                                grammarDetailView.setOnClickListener {
                                    it.context.startActivity(Intent(GRAMMAR_CHECK_RESULT).apply {

                                        val bundle = Bundle()
                                        /*bundle.putParcelable(GRAMMAR_RESULT, grammarNew)
                                        bundle.putString(GRAMMAR_SOURCE, combinedTrans.grammarSourceText)*/
                                        bundle.putBinder(
                                            GRAMMAR_BINDER, GrammarBinder(
                                                grammarNew, combinedTrans.grammarSourceText
                                            )
                                        )

                                        putExtras(bundle)
                                        addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                                    })
                                    removeContentView()
                                }

                                grammarDetailView.measure(
                                    View.MeasureSpec.makeMeasureSpec(
                                        0,
                                        View.MeasureSpec.UNSPECIFIED
                                    ), View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED)
                                )

                                //val bounds = Rect()
                                //nodeInfo.getBoundsInScreen(bounds)

                                val params = overlayParams(grammarDetailView.layoutParams as? WindowManager.LayoutParams)
                                params.flags = params.flags or WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE
                                params.width = grammarDetailView.measuredWidth
                                params.height = grammarDetailView.measuredHeight
                                params.x = rectInScreen.left + rectInScreen.width() - grammarDetailView.measuredWidth
                                params.y = rectInScreen.top - GRAMMAR_PADDING_TOP - grammarDetailView.measuredHeight
                                if (!grammarDetailView.isAttachedToWindow) {
                                    //grammarDetailView.visibility = View.INVISIBLE
                                    windowManager.addView(grammarDetailView, params)
                                }

                                val text = contentView.findViewById<TextView>(R.id.tv_status)
                                text.visibility = View.VISIBLE
                                text.setText(translation)

                            } ?: let {
                                grammarDetailView.visibility = View.INVISIBLE
                                combinedTrans?.sourceLocale?.let {
                                    if (!"en".equals(it)) {
                                        grammarDetailView.setText(R.string.eng_support)
                                    }
                                } ?: let {
                                    grammarDetailView.setText(R.string.grammar_check_error)
                                }
                                contentView.post {
                                    /*Toast.makeText(context,
                                        context.getString(R.string.grammar_check_error), Toast.LENGTH_SHORT)
                                        .show()*/
                                    removeContentView()
                                }
                            }

                        } else {
                            contentView.post {
                                removeContentView()
                            }
                        }

                    } else {
                        val errorMessage = "SUCCESS but translation==null"
                        failureTranslate(
                            sourceText,
                            targetLanguageTag,
                            packageName,
                            errorMessage
                        )
                    }

                }
                else -> {

                    val errorMessage = combinedTrans?.trans?.result?.errorMessage ?: "unknown"
                    failureTranslate(
                        sourceText,
                        targetLanguageTag,
                        packageName,
                        errorMessage
                    )

                }
            }
            // hideProgressBar(contentView)

        }

        val onError = Consumer<Throwable> {
            /*val tvStatus = contentView.findViewById<TextView>(R.id.tv_status)
            tvStatus?.setText(R.string.text_translating_error)
            tvStatus?.textColorResource = R.color.color_floating_failure
            tvStatus?.visibility = View.VISIBLE
            logFailTranslate(
                ACTION_EDIT_TEXT_TRANSLATE_ERROR,
                sourceText,
                targetLanguageTag,
                packageName,
                errorMessage = it.message ?: "unknown"
            )*/

            val errorMessage = it.message ?: "unknown"
            failureTranslate(
                sourceText,
                targetLanguageTag,
                packageName,
                errorMessage
            )

            //hideProgressBar(contentView)

        }

        translateForEditText(sourceText, sourceLanguageTag, targetLanguageTag, onSuccess, onError, context = context, needGrammarCheck = FloatingStateMachine.grammarCheckEnable)

    }

    private fun hideStatus(tvStatus: TextView?) {
        tvStatus?.postDelayed({
            removeContentView()
        }, 1200)
    }
}