package com.talpa.overlay.view

import android.content.Context
import android.content.Intent
import android.graphics.Rect
import android.os.Binder
import android.os.Bundle
import android.text.TextUtils
import android.util.Log
import android.view.accessibility.AccessibilityNodeInfo
import android.widget.EditText
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.lifecycleScope
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.overlay.R
import com.talpa.overlay.data.readOverlayEditTextLanguageTag
import com.talpa.overlay.data.readOverlayTextLanguageTag
import com.talpa.overlay.databinding.MultiOverlayLayoutBinding
import com.talpa.overlay.service.nodeText
import com.talpa.overlay.view.overlay.OverlayImpl
import com.talpa.tengine.SERVER
import com.talpa.tengine.Trans
import com.talpa.tengine.UNKNOWN
import com.talpa.tengine.lang.LANG
import com.talpa.translator.link.translate
import io.reactivex.Observable
import kotlinx.coroutines.*
import java.util.*
import java.util.concurrent.TimeUnit

/**
 * Create by chenjunsheng on 2021/7/22
 */
class MultiOverlayActivity: AppCompatActivity() {

    data class MultiData(
        var nodeInfo: LinkedHashSet<AccessibilityNodeInfo>
    ): Binder()

    companion object {
        const val NODE_INFOS = "node_infos"
        fun start(context: Context, data: MultiData) {
            val intent = Intent(context, MultiOverlayActivity::class.java)
            val bundle = Bundle()
            bundle.putBinder(NODE_INFOS, data)
            intent.putExtra(NODE_INFOS, bundle)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
        }
    }

    private var nodeSet: LinkedHashSet<AccessibilityNodeInfo>? = null

    /**
     * 本地广播
     */
    private val localBroadcastManager by lazy { LocalBroadcastManager.getInstance(this) }

    /**
     * 最小高度限制
     */
    private val limitHeight by lazy { resources.getDimension(R.dimen.dp16) }

    private lateinit var binding: MultiOverlayLayoutBinding

    override fun finish() {
        super.finish()
        overridePendingTransition(0, 0)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        overridePendingTransition(0, 0)
        super.onCreate(savedInstanceState)
        nodeSet = (intent.getBundleExtra(NODE_INFOS)?.getBinder(NODE_INFOS) as? MultiData)?.nodeInfo
        if (nodeSet.isNullOrEmpty()) {
            finish()
            return
        }
        //需要设置这个 flag 才能调用 setStatusBarColor 来设置状态栏颜色
        //window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
        //设置状态栏颜色
        window.setStatusBarColor(
            ContextCompat.getColor(
                this,
                android.R.color.transparent
            )
        )
        binding = MultiOverlayLayoutBinding.inflate(layoutInflater)
        binding.root.setTouchOutSideListener(object : MultiLocateView.OnTouchOutsideListener {
            override fun touchOutside() {
                finish()
            }

        })
        setContentView(binding.root)
        //RxRelay.registerByEventBus(this)

        //removeOverlayViews()
        addOverlayViews(nodeSet!!)

    }

    private val multiOverlayViewModel by viewModels<MultiOverlayViewModel> {
        ViewModelProvider.AndroidViewModelFactory.getInstance(application)
    }

    data class NodeInfo(
        var nodeInfo: AccessibilityNodeInfo,
        var rectInScreen: Rect,
        var transResponse: String? = null
    ) {
        override fun equals(other: Any?): Boolean {
            return if (other !is NodeInfo) {
                false
            } else other.nodeInfo == nodeInfo
        }
    }
    /**
     * 添加悬浮视图
     */
    private fun addOverlayViews(set: LinkedHashSet<AccessibilityNodeInfo>) {
        val nodeInfoList = arrayListOf<NodeInfo>()
        for ((index, nodeInfo) in set.withIndex()) {

            if (index >= MultiOverlayView.MAX_SUM) {
                break
            }

            val rectInScreen = Rect()
            nodeInfo.getBoundsInScreen(rectInScreen)

            if (rectInScreen.height() < limitHeight) {
                continue
            }

            val filter = nodeInfo.className != EditText::class.java.name

            if (filter && !TextUtils.isEmpty(nodeInfo.nodeText())) {
                nodeInfoList.add(NodeInfo(nodeInfo, rectInScreen))
                //views.add(createOverlayView(nodeInfo, rectInScreen))
            }

        }

        val targetLanguageTag = readOverlayTextLanguageTag(this) ?: LANG.EN
        val sourceLanguageTag = LANG.AUTO


        logStartTranslate(
            targetLanguageLocaleTag = targetLanguageTag,
            packageName = packageName ?: "",
            sourceText = nodeInfoList
                .map {
                    val content = it.nodeInfo.nodeText()
                    content.toString().trim()
                }.joinToString(",")
        )
        //binding.root.addOverlay(nodeInfoList)

        lifecycleScope.launch(Dispatchers.Main) {
            var index = 0
            var flag = false
            nodeInfoList.onEach { nodeInfo ->

                binding.root.addOverlay(nodeInfo)
                async(Dispatchers.IO) {
                    val request = Trans(
                        sourceLanguageTag,
                        targetLanguageTag,
                        nodeInfo.nodeInfo.nodeText().toString().trim()
                    )
                    val trans = try {
                        this@MultiOverlayActivity.translate(
                            request
                        ).toFuture().get(8, TimeUnit.SECONDS)
                    } catch (e: Exception) {
                        e.printStackTrace()
                        Log.d("cjslog", "translate error", e)
                        request
                    }
                    nodeInfo.transResponse =
                        trans.result?.translation ?: getString(R.string.translate_fail)

                    withContext(Dispatchers.Main) {
                        index++
                        if (trans.result != null && !trans.result?.translation.isNullOrEmpty()) {
                            flag = true
                        }

                        binding.root.updateOverlay(nodeInfo)

                        if (index == nodeInfoList.size - 1) {
                            if (!flag) {
                                logFailTranslate("failure")
                            } else {
                                logSuccessTranslate(
                                    targetLanguageTag = targetLanguageTag ?: "",
                                    packageName = packageName ?: "",
                                    source = trans.result?.source ?: UNKNOWN
                                )
                            }
                        }
                    }
                }
            }
        }

        //Log.d("cjslog", "source tag:${sourceLanguageTag} ${targetLanguageTag}")
        /*multiOverlayViewModel.multiTrans(sourceLanguageTag, targetLanguageTag, nodeInfoList
            .map {
                val content = it.nodeInfo.nodeText()
                content.toString().trim()
            }).observe(this, androidx.lifecycle.Observer {
                if (it != null) {
                    it.result?.texts?.forEachIndexed { index, s ->
                        nodeInfoList[index].transResponse = s
                    }
                    binding.root.updateTransResult()
                    logSuccessTranslate(
                        targetLanguageTag = targetLanguageTag ?: "",
                        packageName = packageName ?: "",
                        source = SERVER
                    )
                } else {
                    logFailTranslate("failure")
                }
        })*/

    }

    /**
     * 打点开始翻译
     */
    private fun logStartTranslate(
        targetLanguageLocaleTag: String,
        packageName: String,
        sourceText: String
    ) {

        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_START).apply {
            putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(FloatingManager.FROM, FloatingManager.WHOLE)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
        }

        GlobalScope.launch {
            localBroadcastManager.sendBroadcast(intent)
        }
    }

    private fun logSuccessTranslate(
        targetLanguageTag: String,
        packageName: String,
        source: String
    ) {
        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_SUCCESS).apply {
            //  putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageTag)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
            putExtra(OverlayImpl.EXTRA_SOURCE, source)
        }

        localBroadcastManager.sendBroadcast(intent)
    }

    private fun logFailTranslate(message: String) {


        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_ERROR).apply {
            //  putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            //putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
            //putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
            putExtra(OverlayImpl.EXTRA_ERROR, message)
        }

        localBroadcastManager.sendBroadcast(intent)
    }
}