package com.talpa.overlay.view

import android.animation.Animator
import android.animation.ObjectAnimator
import android.app.ActivityManager
import android.app.Application
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.res.Configuration
import android.graphics.PixelFormat
import android.graphics.Point
import android.os.Build
import android.os.Message
import android.util.Log
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.WindowManager
import android.widget.ImageView
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.overlay.R
import com.talpa.overlay.RxRelay.EVENT_COPY_ENTER
import com.talpa.overlay.RxRelay.EVENT_DARK_ENTER
import com.talpa.overlay.RxRelay.EVENT_DARK_EXIT
import com.talpa.overlay.RxRelay.EVENT_DETECT_ENTER
import com.talpa.overlay.RxRelay.EVENT_DETECT_EXIT
import com.talpa.overlay.RxRelay.EVENT_FIND_LOCATION_ENTER
import com.talpa.overlay.RxRelay.EVENT_HIGHLIGHT_ENTER
import com.talpa.overlay.RxRelay.EVENT_HIGHLIGHT_EXIT
import com.talpa.overlay.RxRelay.EVENT_IDLE_ENTER
import com.talpa.overlay.RxRelay.EVENT_IDLE_EXIT
import com.talpa.overlay.RxRelay.EVENT_INVISIBLE_ENTER
import com.talpa.overlay.RxRelay.EVENT_LIGHT_ENTER
import com.talpa.overlay.RxRelay.EVENT_LIGHT_EXIT
import com.talpa.overlay.RxRelay.EVENT_MENU_ENTER
import com.talpa.overlay.RxRelay.EVENT_MENU_EXIT
import com.talpa.overlay.RxRelay.registerByEventBus
import com.talpa.overlay.Transfer
import com.talpa.overlay.databinding.LayoutFloatingBinding
import com.talpa.overlay.state.FloatingStateMachine
import com.talpa.overlay.state.FloatingStateMachine.WHAT_COPY_ENTER
import com.talpa.overlay.state.FloatingStateMachine.WHAT_INVISIBLE_ENTER
import com.talpa.overlay.state.FloatingStateMachine.WHAT_LIGHT_ENTER
import com.talpa.overlay.state.FloatingStateMachine.currentState
import com.talpa.overlay.state.FloatingStateMachine.grammarCheckEnable
import com.talpa.overlay.tools.canDrawOverlays
import com.talpa.overlay.view.menu.FloatingMenu
import com.talpa.translate.kv.KeyValue
import com.tapla.textspeech.TextSpeech
import io.reactivex.Flowable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.plugins.RxJavaPlugins
import io.reactivex.schedulers.Schedulers
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import org.greenrobot.eventbus.Subscribe
import org.greenrobot.eventbus.ThreadMode
import org.jetbrains.anko.windowManager
import java.util.*

/**
 * 管理悬浮球
 *
 * @author cy 19-7-29
 */
object FloatingManager {

    private lateinit var mApplication: Application

    /**
     * 悬浮视图
     */
    private val floatingView: FloatingContainer by lazy { createFloatingView(mApplication) }

    /**
     * 手柄
     */
    private val floatingHandleView by lazy { floatingView.findViewById<ImageView>(R.id.floating_handle) }

    /**
     * 识别 View
     */
    private val floatingDetectorView by lazy { floatingView.findViewById<ImageView>(R.id.floating) }


    /**
     * WindowManager
     */
    private val windowManager by lazy { getContext().getSystemService(Context.WINDOW_SERVICE) as WindowManager }


    /**
     * Params
     */
    private val floatingParams by lazy {
        floatingParams()
    }


    /**
     * 悬浮菜单
     */
    private val floatingMenu: FloatingMenu by lazy { FloatingMenu(mApplication) }

    /**
     * 多个覆盖视图
     */
    private val multiOverlayView: MultiOverlayView by lazy { MultiOverlayView(getContext()) }

    /**
     * 打开悬浮球广播
     */
    const val BROADCAST_ACTION_FLOATING_OPEN = "com.talpa.overlay.BROADCAST_ACTION_FLOATING_OPEN"

    /**
     * 关闭悬浮球广播
     */
    const val BROADCAST_ACTION_FLOATING_CLOSE = "com.talpa.overlay.BROADCAST_ACTION_FLOATING_CLOSE"

    const val BROADCAST_ACTION_FLOATING_CLIPBOARD =
        "com.talpa.overlay.BROADCAST_ACTION_FLOATING_CLIPBOARD"

    const val BROADCAST_ACTION_ACCESS_SHOW = "com.talpa.overlay.BROADCAST_ACTION_ACCESS_SHOW"

    const val BROADCAST_ACTION_ACCESS_CLICK = "com.talpa.overlay.BROADCAST_ACTION_ACCESS_CLICK"

    const val PART = "part"
    const val WHOLE = "whole"
    const val FROM = "from"


    /**
     * Overlay Init
     */
    fun init(application: Application) {
        mApplication = application
        registerByEventBus(this)
        FloatingStateMachine.setupStateMachine()
        Transfer.init(application)
        TextSpeech.init(application = getApplication())

        registerLocalReceiver(context = application)
        RxJavaPlugins.setErrorHandler {
            Log.d("cjslog", "rx java plugins error")
        }
    }

    private fun getContext() = mApplication

    private fun getApplication() = mApplication

    internal fun getLocationOnScreen(intArray: IntArray) {
        floatingView.getLocationOnScreen(intArray)
    }

    fun changeLanguage(textLanguage: String?, editLanguage: String?) {
        if (textLanguage != Locale.ENGLISH.language && editLanguage != Locale.ENGLISH.language) {
            if (FloatingStateMachine.currentState() is FloatingStateMachine.FloatingState.StaticState
            ) {
                MainScope().launch {
                    grammarCheckEnable = false
                    floatingHandleView.setImageResource(R.drawable.float_state_static_light)
                }
            }
        }
    }

    private fun registerLocalReceiver(context: Context) {
        val filter = IntentFilter()
        filter.addAction(BROADCAST_ACTION_FLOATING_OPEN)
        filter.addAction(BROADCAST_ACTION_FLOATING_CLOSE)
        filter.addAction(BROADCAST_ACTION_FLOATING_CLIPBOARD)
        val localReceiver = LocalReceiver()
        LocalBroadcastManager.getInstance(context).registerReceiver(localReceiver, filter)
    }

    private class LocalReceiver : BroadcastReceiver() {
        override fun onReceive(context: Context?, intent: Intent?) {

            when (intent?.action) {
                BROADCAST_ACTION_FLOATING_OPEN -> {
                    FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
                }
                BROADCAST_ACTION_FLOATING_CLOSE -> {
                    FloatingStateMachine.sendMessage(WHAT_INVISIBLE_ENTER)
                }
                BROADCAST_ACTION_FLOATING_CLIPBOARD -> {
                    FloatingStateMachine.sendMessage(WHAT_COPY_ENTER)
                }
            }
        }
    }

    /**
     * 显示悬浮球
     */
    @Synchronized
    private fun showFloatingView() {

        val getParams = {
            val floatingDefaultHeight = getContext().floatingDefaultHeight()
            val floatingDefaultWidth = getContext().floatingDefaultWidth()

            val params = floatingParams
            params.width = floatingDefaultWidth//WindowManager.LayoutParams.WRAP_CONTENT
            params.height = floatingDefaultHeight//WindowManager.LayoutParams.WRAP_CONTENT

            val point = readFloatPosition()
            params.x = point.x
            params.y = point.y
            saveFloatPosition(point.x, point.y)
            params
        }

        val subscribe = { params: WindowManager.LayoutParams ->
            if (!isAttachedToWindow() && getContext().canDrawOverlays()) {
                try {
                    if (floatingView.getTag(R.id.id_floating) != null) {
                        windowManager.removeViewImmediate(floatingView)
                        windowManager.addView(floatingView, params)
                        animShowFloatingView()
                    } else {
                        windowManager.addView(floatingView, params)
                        floatingView.setTag(R.id.id_floating, Any())
                        animShowFloatingView()
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }

            }

        }

        val dis = Flowable.fromCallable(getParams)
            .subscribeOn(Schedulers.single())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(subscribe, Throwable::printStackTrace)


    }

    private fun animShowFloatingView() {
        val delay = 200L
        //val scaleX = ObjectAnimator.ofFloat(floatingView, "scaleX", floatingView.alpha, 0F)
        //val scaleY = ObjectAnimator.ofFloat(floatingView, "scaleY", floatingView.alpha, 0F)
        val alphaAnim = ObjectAnimator.ofFloat(floatingView, "alpha", floatingView.alpha, 1f)

        alphaAnim.setDuration(delay)
        alphaAnim.start()
    }

    /**
     * 隐藏悬浮球
     */
    @Synchronized
    private fun hideFloatingView() {

        when (FloatingStateMachine.lastState()) {
            is FloatingStateMachine.FloatingState.StaticState.IdleState -> {
                if (floatingView.isAttachedToWindow) {
                    windowManager.removeView(floatingView)
                    floatingView.setTag(R.id.id_floating, null)
                }
            }
            else -> {
                animHideFloatingView()
            }
        }

    }

    /**
     * 动画显示
     */
    private fun animHideFloatingView() {
        if (floatingView.isAttachedToWindow) {
            val delay = 200L
            //val scaleX = ObjectAnimator.ofFloat(floatingView, "scaleX", floatingView.alpha, 0F)
            //val scaleY = ObjectAnimator.ofFloat(floatingView, "scaleY", floatingView.alpha, 0F)
            val alphaAnim = ObjectAnimator.ofFloat(floatingView, "alpha", floatingView.alpha, 0f)

            val listener = object : Animator.AnimatorListener {
                override fun onAnimationRepeat(animation: Animator?) {
                }

                override fun onAnimationEnd(animation: Animator?) {
                    if (floatingView.isAttachedToWindow) {
                        windowManager.removeViewImmediate(floatingView)
                        floatingView.setTag(R.id.id_floating, null)
                    }
                }

                override fun onAnimationCancel(animation: Animator?) {
                }

                override fun onAnimationStart(animation: Animator?) {
                }
            }

            alphaAnim.addListener(listener)
            alphaAnim.setDuration(delay)
            alphaAnim.start()
            /*AnimatorSet().apply {
                duration = delay
                playTogether(scaleX, scaleY)
                addListener(listener)
                start()
            }*/

        }
    }

    /**
     * 悬浮视图是否显示
     */
    fun isFloatingVisible(context: Context): Boolean {
        return context.canDrawOverlays() && floatingView.isAttachedToWindow
    }

    /**
     * 更新悬浮视图的位置信息
     */
    private fun updateFloatingView(
        x: Int = floatingParams.x,
        y: Int = floatingParams.y,
        width: Int = floatingParams.width,
        height: Int = floatingParams.height
    ) {
        if (floatingView.isAttachedToWindow) {
            val params = floatingParams
            params.width = width//dp120//WindowManager.LayoutParams.WRAP_CONTENT
            params.height = height// dp120//WindowManager.LayoutParams.WRAP_CONTENT
            params.x = x//point.x
            params.y = y//point.y
            try {
                windowManager.updateViewLayout(floatingView, params)
            } catch (e: java.lang.Exception) {
                e.printStackTrace()
            }

        }

    }

    /**
     * 悬浮球是否显示
     */
    internal fun isAttachedToWindow(): Boolean {

        //val isShown = floatingView.isShown
        //val isDirty = floatingView.isDirty

        return floatingView.isAttachedToWindow //floatingView.isAttachedToWindow && floatingView.parent != null
    }

    private fun createFloatingView(context: Context): FloatingContainer {

        val floatingBinding = LayoutFloatingBinding.inflate(LayoutInflater.from(context))

        return floatingBinding.root as FloatingContainer
    }


    fun onConfigurationChanged(newConfig: Configuration?) {


        if (currentState() is FloatingStateMachine.FloatingState.StaticState.IdleState) {
            idleFloatingImage()
        }

    }

    /*　×××××××××××××××××××××STATE START×××××××××××××××××××××××××××××××××××××××××　*/


    /**
     * ENTER IDLE STATE
     */
    private fun enterIdle() {

        idleFloatingImage()
        if (!isAttachedToWindow()) {

            showFloatingView()
        }


    }

    private fun exitIdle() {
        // updateFloatingView(width = floatingDefaultWidth)
    }

    private fun idleFloatingImage() {

        val outSize = Point()
        windowManager.defaultDisplay.getSize(outSize)

        val screenWidth = outSize.x

        val x = floatingParams.x

        val resId = if (x > screenWidth / 2) {
            R.drawable.float_state_idle_right
        } else {
            R.drawable.float_state_idle_left
        }

        floatingHandleView.setImageResource(resId)
        floatingHandleView.setBackgroundResource(0)
    }

    /**
     * 进入静止Light 状态
     */
    private fun enterLight() {
        //floatingHandleView.setImageResource(R.drawable.layer_state_static_light)
        if (grammarCheckEnable) {
            floatingHandleView.setImageResource(R.drawable.float_state_static_grammar_light)
        } else {
            floatingHandleView.setImageResource(R.drawable.float_state_static_light)
        }
        //floatingHandleView.setBackgroundResource(R.drawable.float_state_white_bg)
        floatingHandleView.scaleType = ImageView.ScaleType.CENTER

        val floatingDefaultHeight = getContext().floatingDefaultHeight()
        val floatingDefaultWidth = getContext().floatingDefaultWidth()

        val point = readFloatPosition()

        updateFloatingView(point.x, point.y, floatingDefaultWidth, floatingDefaultHeight)

        if (!isAttachedToWindow()) {

            showFloatingView()
        }

        floatingHandleView.postDelayed({
            FloatingStateMachine.sendMessage(EVENT_DARK_ENTER)
        }, MILLIS_DIM)

        getContext().floatingSaveVisibleState(visible = true)
    }

    private fun exitLight() {
    }

    /**
     * 进入静止 Dark 状态
     */
    private fun enterDark() {
        floatingHandleView.alpha = 0.5f

        if (!isAttachedToWindow()) {

            showFloatingView()
        }
        floatingHandleView.postDelayed({
            FloatingStateMachine.sendMessage(EVENT_IDLE_ENTER)
        }, MILLIS_DIM)
    }

    private fun exitDark() {
        floatingHandleView.alpha = 1f
    }

    /**
     * 进入探测状态
     */
    private fun enterDetect() {
        floatingDetectorView.visibility = View.VISIBLE
        if (grammarCheckEnable) {
            floatingDetectorView.setImageResource(R.drawable.float_state_working_grammar_detecting)
        } else {
            floatingDetectorView.setImageResource(R.drawable.float_state_working_detecting)
        }
        floatingHandleView.visibility = View.INVISIBLE

        //val dp24 = getContext().resources.getDimension(R.dimen.dp30).toInt()

        //floatingView.detectViewOffset(0 - dp24, 0 - dp24)
        if (!isAttachedToWindow()) {

            showFloatingView()
        }
    }

    private fun exitDetect() {
        floatingDetectorView.visibility = View.INVISIBLE
        floatingHandleView.visibility = View.VISIBLE

        /*val params = floatingHandleView.layoutParams as ConstraintLayout.LayoutParams
        params.circleConstraint = 0
        params.circleAngle = 0f
        params.circleRadius = 0
        floatingHandleView.layoutParams = params*/
    }

    private fun enterFindLocation() {
        floatingHandleView.setImageResource(R.drawable.layer_state_working_find_location)
    }

    /**
     * 进入高亮模式，即全局翻译
     */
    private fun enterHighLight() {
        //floatingHandleView.setImageResource(R.drawable.layer_state_highlight)
        floatingHandleView.setImageResource(R.drawable.float_state_highlight)
        //floatingHandleView.setBackgroundResource(R.drawable.float_state_white_bg)
        //multiOverlayView.startMultiOverlay()
        MultiOverlayActivity.start(getContext())
    }

    private fun exitHighLight() {
        multiOverlayView.stopMultiOverlay()
    }

    /**
     * 进入不可见状态
     */
    private fun enterInvisible() {

        hideFloatingView()

        getContext().floatingSaveVisibleState(visible = false)
    }

    /**
     * 进入复制翻译状态
     */
    private fun enterCopy() {
        floatingHandleView.setImageResource(R.drawable.layer_state_copy)

        floatingHandleView.postDelayed({
            if (FloatingStateMachine.currentState() is FloatingStateMachine.FloatingState.CopyState) {
                FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
            }
        }, 5000)

    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    internal fun onNext(message: Message) {

        when (message.what) {

            EVENT_IDLE_ENTER -> {

                enterIdle()
            }
            EVENT_IDLE_EXIT -> {
                exitIdle()
            }
            EVENT_LIGHT_ENTER -> {

                enterLight()
            }
            EVENT_LIGHT_EXIT -> {
                exitLight()
            }
            EVENT_DARK_ENTER -> {

                enterDark()
            }
            EVENT_DARK_EXIT -> {
                exitDark()
            }
            EVENT_DETECT_ENTER -> {

                enterDetect()
            }
            EVENT_DETECT_EXIT -> {

                exitDetect()
            }
            EVENT_FIND_LOCATION_ENTER -> {
                enterFindLocation()
            }
            EVENT_MENU_ENTER -> {
                hideFloatingView()
                val point = readFloatPosition()
                floatingMenu.enterMenu(getContext(), point)
            }
            EVENT_MENU_EXIT -> {
                //showFloatingView()
                floatingMenu.exitMenu()
            }
            EVENT_HIGHLIGHT_ENTER -> {
                enterHighLight()
            }
            EVENT_HIGHLIGHT_EXIT -> {
                exitHighLight()
            }
            EVENT_INVISIBLE_ENTER -> {
                enterInvisible()
            }
            EVENT_COPY_ENTER -> {
                enterCopy()
            }
        }
    }


    /*　×××××××××××××××××××××STATE END×××××××××××××××××××××××××××××××××××××××××　*/


    private fun saveFloatPosition(x: Int, y: Int) {
        if (ActivityManager.isUserAMonkey()) {
            return
        }

        KeyValue.put(PREFER_KEY_FLOATING_INIT_X, x)
        KeyValue.put(PREFER_KEY_FLOATING_INIT_Y, y)
        /*val mmkv = MMKV.defaultMMKV()
        mmkv.encode(FloatingManager.PREFER_KEY_FLOATING_INIT_X, x)
        mmkv.encode(FloatingManager.PREFER_KEY_FLOATING_INIT_Y, y)*/
    }

    private fun readFloatPosition(): Point {

        /*val mmkv = MMKV.defaultMMKV()
        var x = mmkv.decodeInt(PREFER_KEY_FLOATING_INIT_X, -1)
        var y = mmkv.decodeInt(PREFER_KEY_FLOATING_INIT_Y, -1)*/
        var x = KeyValue.get(PREFER_KEY_FLOATING_INIT_X, -1) ?: -1
        var y = KeyValue.get(PREFER_KEY_FLOATING_INIT_Y, -1) ?: -1

        if (x == -1 || y == -1) {
            val size = getContext().floatingDefaultPoint()
            x = size.x
            y = size.y
        }

        return Point(x, y)
    }


    /**
     * 初始化坐标
     */
    const val PREFER_KEY_FLOATING_INIT_X = "floating_init_x"
    const val PREFER_KEY_FLOATING_INIT_Y = "floating_init_y"
    const val PREFER_KEY_FLOATING_VISIBLE = "floating_visible" //true visible,false invisible
    const val PREFER_NAME = "Floating"

    private const val MILLIS_DIM = 3000L//进入Dark 模式时间间隔
}

/**
 * 悬浮视图默认位置
 */
internal fun Context.floatingDefaultPoint(): Point {
    val outSize = Point()

    val display = this.windowManager.defaultDisplay

    display.getSize(outSize)

    val x = 0
    val y = (outSize.y) / 2

    outSize.set(x, y)
    return outSize
}

/**
 * 悬浮视图默认宽
 */
internal fun Context.floatingDefaultWidth() = resources.getDimension(R.dimen.dp68).toInt()

/**
 * 悬浮视图默认高
 */
internal fun Context.floatingDefaultHeight() = resources.getDimension(R.dimen.dp68).toInt()

/**
 * 保存悬浮球坐标
 */
internal fun Context.floatingSavePosition(x: Int, y: Int) {
    if (ActivityManager.isUserAMonkey()) {
        return
    }
    /*val mmkv = MMKV.defaultMMKV()
    mmkv.encode(FloatingManager.PREFER_KEY_FLOATING_INIT_X, x)
    mmkv.encode(FloatingManager.PREFER_KEY_FLOATING_INIT_Y, y)*/

    KeyValue.put(FloatingManager.PREFER_KEY_FLOATING_INIT_X, x)
    KeyValue.put(FloatingManager.PREFER_KEY_FLOATING_INIT_Y, y)
}

/**
 * 重置为默认位置
 */
internal fun Context.floatingResetDefaultPosition(): Point {
    val point = floatingDefaultPoint()
    floatingSavePosition(point.x, point.y)
    return point
}

///**
// * 悬浮球是否为可见状态，从 Shared Preference 读取
// */
//fun Context.floatingVisibleState(visible: Boolean = false): Boolean {
//    return MMKV.defaultMMKV().decodeBool(FloatingManager.PREFER_KEY_FLOATING_VISIBLE, visible)
//    /* val prefer = getSharedPreferences(FloatingManager.PREFER_NAME, Context.MODE_PRIVATE)
//     return prefer.getBoolean(FloatingManager.PREFER_KEY_FLOATING_VISIBLE, visible)*/
//}

/**
 * 悬浮球是否为可见状态，true visible,false invisible
 */
internal fun Context.floatingSaveVisibleState(visible: Boolean) {
    if (ActivityManager.isUserAMonkey()) {
        return
    }
    //MMKV.defaultMMKV().encode(FloatingManager.PREFER_KEY_FLOATING_VISIBLE, visible)
    KeyValue.put(FloatingManager.PREFER_KEY_FLOATING_VISIBLE, visible)
}


/**
 * 悬浮球 Params
 */
internal fun floatingParams(): WindowManager.LayoutParams {
    val type = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
        WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY
    else WindowManager.LayoutParams.TYPE_PHONE
    val flags = WindowManager.LayoutParams.FLAG_WATCH_OUTSIDE_TOUCH or
            WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL or
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE or
            WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN or
            WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED

    val params = WindowManager.LayoutParams(type, flags, PixelFormat.TRANSLUCENT)

    params.gravity = Gravity.START or Gravity.TOP

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
        params.layoutInDisplayCutoutMode =
            WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES
    }

    params.format = PixelFormat.RGBA_8888
    return params
}

/**
 * 悬浮视图 Params
 */
internal fun overlayViewParams(): WindowManager.LayoutParams {
    val type = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
        WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY
    else WindowManager.LayoutParams.TYPE_PHONE
    val flags = WindowManager.LayoutParams.FLAG_WATCH_OUTSIDE_TOUCH or
            /*WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL or
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE or*/ //若是添加了，则不能监听按钮，若是不添加，则会遮挡无障碍服务识别
            WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN or
            WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED

    val params = WindowManager.LayoutParams(type, flags, PixelFormat.TRANSLUCENT)

    //params.gravity = Gravity.START or Gravity.TOP

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
        params.layoutInDisplayCutoutMode =
            WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES
    }

    params.format = PixelFormat.RGBA_8888

    //params.windowAnimations =0// R.style.AnimationOverlay

    return params
}

/**
 * Open
 */
fun Context.open(): Boolean {
    LocalBroadcastManager.getInstance(applicationContext)
        .sendBroadcast(Intent(FloatingManager.BROADCAST_ACTION_FLOATING_OPEN))
    return false
}

/**
 * Close
 */
fun Context.close() {
    LocalBroadcastManager.getInstance(applicationContext)
        .sendBroadcast(Intent(FloatingManager.BROADCAST_ACTION_FLOATING_CLOSE))
}

/**
 * Close
 */
fun Context.showClipboardFloating() {
    LocalBroadcastManager.getInstance(applicationContext)
        .sendBroadcast(Intent(FloatingManager.BROADCAST_ACTION_FLOATING_CLIPBOARD))
}