package com.talpa.overlay.view

import android.content.Context
import android.content.Intent
import android.graphics.Rect
import android.os.Bundle
import android.os.Message
import android.text.TextUtils
import android.util.Log
import android.view.WindowManager
import android.view.accessibility.AccessibilityNodeInfo
import android.widget.EditText
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.overlay.R
import com.talpa.overlay.RxRelay
import com.talpa.overlay.data.readOverlayEditTextLanguageTag
import com.talpa.overlay.data.readOverlayTextLanguageTag
import com.talpa.overlay.databinding.MultiOverlayLayoutBinding
import com.talpa.overlay.service.AccessService
import com.talpa.overlay.service.nodeText
import com.talpa.overlay.state.FloatingStateMachine
import com.talpa.overlay.view.overlay.OverlayImpl
import com.talpa.tengine.SERVER
import com.talpa.tengine.UNKNOWN
import com.talpa.tengine.lang.LANG
import com.talpa.translate.network.TransResponse
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import org.greenrobot.eventbus.Subscribe
import org.greenrobot.eventbus.ThreadMode
import java.util.*
import kotlin.collections.LinkedHashSet

/**
 * Create by chenjunsheng on 2021/7/22
 */
class MultiOverlayActivity: AppCompatActivity() {

    companion object {
        fun start(context: Context) {
            val intent = Intent(context, MultiOverlayActivity::class.java)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
        }
    }

    private var nodeSet: LinkedHashSet<AccessibilityNodeInfo>? = null

    /**
     * 本地广播
     */
    private val localBroadcastManager by lazy { LocalBroadcastManager.getInstance(this) }

    /**
     * 最小高度限制
     */
    private val limitHeight by lazy { resources.getDimension(R.dimen.dp16) }

    private lateinit var binding: MultiOverlayLayoutBinding

    override fun finish() {
        super.finish()
        overridePendingTransition(0, 0)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        overridePendingTransition(0, 0)
        super.onCreate(savedInstanceState)
        //需要设置这个 flag 才能调用 setStatusBarColor 来设置状态栏颜色
        window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
        //设置状态栏颜色
        window.setStatusBarColor(
            ContextCompat.getColor(
                this,
                android.R.color.transparent
            )
        )
        binding = MultiOverlayLayoutBinding.inflate(layoutInflater)
        binding.root.setTouchOutSideListener(object : MultiLocateView.OnTouchOutsideListener {
            override fun touchOutside() {
                finish()
            }

        })
        setContentView(binding.root)
        RxRelay.registerByEventBus(this)
        AccessService.startServiceForStartMultiNodes(this)
    }

    private val multiOverlayViewModel by viewModels<MultiOverlayViewModel> {
        ViewModelProvider.AndroidViewModelFactory.getInstance(application)
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onMessageEvent(message: Message) {
        when (message.what) {
            RxRelay.EVENT_NODE_INFO_MULTI -> {

                if (FloatingStateMachine.currentState() !is FloatingStateMachine.FloatingState.HighLightState) {
                    return
                }

                val nodeSet = message.obj as? LinkedHashSet<AccessibilityNodeInfo> ?: return

                //removeOverlayViews()
                addOverlayViews(nodeSet)
                this.nodeSet = nodeSet

            }
        }
    }

    data class NodeInfo(
        var nodeInfo: AccessibilityNodeInfo,
        var rectInScreen: Rect,
        var transResponse: String? = null
    )
    /**
     * 添加悬浮视图
     */
    private fun addOverlayViews(set: LinkedHashSet<AccessibilityNodeInfo>) {
        val nodeInfoList = arrayListOf<NodeInfo>()
        for ((index, nodeInfo) in set.withIndex()) {

            if (index >= MultiOverlayView.MAX_SUM) {
                break
            }

            val rectInScreen = Rect()
            nodeInfo.getBoundsInScreen(rectInScreen)

            if (rectInScreen.height() < limitHeight) {
                continue
            }

            val filter = nodeInfo.className != EditText::class.java.name

            if (filter && !TextUtils.isEmpty(nodeInfo.nodeText())) {
                nodeInfoList.add(NodeInfo(nodeInfo, rectInScreen))
                //views.add(createOverlayView(nodeInfo, rectInScreen))
            }

        }

        val targetLanguageTag = readOverlayEditTextLanguageTag(this) ?: LANG.EN
        val sourceLanguageTag = readOverlayTextLanguageTag(this) ?: Locale.getDefault().language


        logStartTranslate(
            targetLanguageLocaleTag = targetLanguageTag,
            packageName = packageName ?: "",
            sourceText = nodeInfoList
                .map {
                    val content = it.nodeInfo.nodeText()
                    content.toString().trim()
                }.joinToString(",")
        )
        binding.root.addOverlay(nodeInfoList)
        multiOverlayViewModel.multiTrans(sourceLanguageTag, targetLanguageTag, nodeInfoList
            .map {
                val content = it.nodeInfo.nodeText()
                content.toString().trim()
            }).observe(this, androidx.lifecycle.Observer {
                if (it != null) {
                    Log.d("cjslog", "fffffffff")
                    it.result?.texts?.forEachIndexed { index, s ->
                        nodeInfoList[index].transResponse = s
                    }
                    binding.root.updateTransResult()
                    logSuccessTranslate(
                        targetLanguageTag = targetLanguageTag ?: "",
                        packageName = packageName ?: "",
                        source = SERVER
                    )
                } else {
                    logFailTranslate("failure")
                }
        })

    }

    /**
     * 打点开始翻译
     */
    private fun logStartTranslate(
        targetLanguageLocaleTag: String,
        packageName: String,
        sourceText: String
    ) {

        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_START).apply {
            putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(FloatingManager.FROM, FloatingManager.WHOLE)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
        }

        GlobalScope.launch {
            localBroadcastManager.sendBroadcast(intent)
        }
    }

    private fun logSuccessTranslate(
        targetLanguageTag: String,
        packageName: String,
        source: String
    ) {
        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_SUCCESS).apply {
            //  putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageTag)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
            putExtra(OverlayImpl.EXTRA_SOURCE, source)
        }

        localBroadcastManager.sendBroadcast(intent)
    }

    private fun logFailTranslate(message: String) {


        val intent = Intent(OverlayImpl.ACTION_MULTI_TRANSLATE_ERROR).apply {
            //  putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            //putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
            //putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
            putExtra(OverlayImpl.EXTRA_ERROR, message)
        }

        localBroadcastManager.sendBroadcast(intent)
    }
}